"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InterfaceVpcEndpoint = exports.InterfaceVpcEndpointAwsService = exports.InterfaceVpcEndpointService = exports.GatewayVpcEndpoint = exports.GatewayVpcEndpointAwsService = exports.VpcEndpointType = exports.VpcEndpoint = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const cxschema = require("../../cloud-assembly-schema");
const core_1 = require("../../core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const peer_1 = require("./peer");
const port_1 = require("./port");
const security_group_1 = require("./security-group");
const util_1 = require("./util");
class VpcEndpoint extends core_1.Resource {
    /**
     * Adds a statement to the policy document of the VPC endpoint. The statement
     * must have a Principal.
     *
     * Not all interface VPC endpoints support policy. For more information
     * see https://docs.aws.amazon.com/vpc/latest/userguide/vpce-interface.html
     *
     * @param statement the IAM statement to add
     */
    addToPolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        if (!statement.hasPrincipal) {
            throw new Error('Statement must have a `Principal`.');
        }
        if (!this.policyDocument) {
            this.policyDocument = new iam.PolicyDocument();
        }
        this.policyDocument.addStatements(statement);
    }
}
exports.VpcEndpoint = VpcEndpoint;
_a = JSII_RTTI_SYMBOL_1;
VpcEndpoint[_a] = { fqn: "monocdk.aws_ec2.VpcEndpoint", version: "1.149.0" };
/**
 * The type of VPC endpoint.
 */
var VpcEndpointType;
(function (VpcEndpointType) {
    /**
     * Interface
     *
     * An interface endpoint is an elastic network interface with a private IP
     * address that serves as an entry point for traffic destined to a supported
     * service.
     */
    VpcEndpointType["INTERFACE"] = "Interface";
    /**
     * Gateway
     *
     * A gateway endpoint is a gateway that is a target for a specified route in
     * your route table, used for traffic destined to a supported AWS service.
     */
    VpcEndpointType["GATEWAY"] = "Gateway";
})(VpcEndpointType = exports.VpcEndpointType || (exports.VpcEndpointType = {}));
/**
 * An AWS service for a gateway VPC endpoint.
 */
class GatewayVpcEndpointAwsService {
    constructor(name, prefix) {
        this.name = `${prefix || 'com.amazonaws'}.${core_1.Aws.REGION}.${name}`;
    }
}
exports.GatewayVpcEndpointAwsService = GatewayVpcEndpointAwsService;
_b = JSII_RTTI_SYMBOL_1;
GatewayVpcEndpointAwsService[_b] = { fqn: "monocdk.aws_ec2.GatewayVpcEndpointAwsService", version: "1.149.0" };
GatewayVpcEndpointAwsService.DYNAMODB = new GatewayVpcEndpointAwsService('dynamodb');
GatewayVpcEndpointAwsService.S3 = new GatewayVpcEndpointAwsService('s3');
/**
 * A gateway VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class GatewayVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_ec2_GatewayVpcEndpointProps(props);
        const subnets = props.subnets
            ? util_1.flatten(props.subnets.map(s => props.vpc.selectSubnets(s).subnets))
            : [...props.vpc.privateSubnets, ...props.vpc.publicSubnets, ...props.vpc.isolatedSubnets];
        const routeTableIds = util_1.allRouteTableIds(subnets);
        if (routeTableIds.length === 0) {
            throw new Error('Can\'t add a gateway endpoint to VPC; route table IDs are not available');
        }
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            policyDocument: core_1.Lazy.any({ produce: () => this.policyDocument }),
            routeTableIds,
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.GATEWAY,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    static fromGatewayVpcEndpointId(scope, id, gatewayVpcEndpointId) {
        class Import extends VpcEndpoint {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = gatewayVpcEndpointId;
            }
        }
        return new Import(scope, id);
    }
}
exports.GatewayVpcEndpoint = GatewayVpcEndpoint;
_c = JSII_RTTI_SYMBOL_1;
GatewayVpcEndpoint[_c] = { fqn: "monocdk.aws_ec2.GatewayVpcEndpoint", version: "1.149.0" };
/**
 * A custom-hosted service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointService {
    constructor(name, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = false;
        this.name = name;
        this.port = port || 443;
    }
}
exports.InterfaceVpcEndpointService = InterfaceVpcEndpointService;
_d = JSII_RTTI_SYMBOL_1;
InterfaceVpcEndpointService[_d] = { fqn: "monocdk.aws_ec2.InterfaceVpcEndpointService", version: "1.149.0" };
/**
 * An AWS service for an interface VPC endpoint.
 */
class InterfaceVpcEndpointAwsService {
    constructor(name, prefix, port) {
        /**
         * Whether Private DNS is supported by default.
         */
        this.privateDnsDefault = true;
        const region = core_1.Lazy.uncachedString({
            produce: (context) => core_1.Stack.of(context.scope).region,
        });
        const defaultEndpointPrefix = core_1.Lazy.uncachedString({
            produce: (context) => {
                const regionName = core_1.Stack.of(context.scope).region;
                return this.getDefaultEndpointPrefix(name, regionName);
            },
        });
        const defaultEndpointSuffix = core_1.Lazy.uncachedString({
            produce: (context) => {
                const regionName = core_1.Stack.of(context.scope).region;
                return this.getDefaultEndpointSuffix(name, regionName);
            },
        });
        this.name = `${prefix || defaultEndpointPrefix}.${region}.${name}${defaultEndpointSuffix}`;
        this.port = port || 443;
    }
    /**
     * Get the endpoint prefix for the service in the specified region
     * because the prefix for some of the services in cn-north-1 and cn-northwest-1 are different
     *
     * For future maintenance， the vpc endpoint services could be fetched using AWS CLI Commmand:
     * aws ec2 describe-vpc-endpoint-services
     */
    getDefaultEndpointPrefix(name, region) {
        var _g;
        const VPC_ENDPOINT_SERVICE_EXCEPTIONS = {
            'cn-north-1': ['application-autoscaling', 'athena', 'autoscaling', 'awsconnector', 'cassandra',
                'cloudformation', 'codedeploy-commands-secure', 'databrew', 'dms', 'ebs', 'ec2', 'ecr.api', 'ecr.dkr',
                'elasticbeanstalk', 'elasticfilesystem', 'elasticfilesystem-fips', 'execute-api', 'imagebuilder',
                'iotsitewise.api', 'iotsitewise.data', 'kinesis-streams', 'lambda', 'license-manager', 'monitoring',
                'rds', 'redshift', 'redshift-data', 's3', 'sagemaker.api', 'sagemaker.featurestore-runtime',
                'sagemaker.runtime', 'servicecatalog', 'sms', 'sqs', 'states', 'sts', 'synthetics', 'transcribe',
                'transcribestreaming', 'transfer', 'xray'],
            'cn-northwest-1': ['application-autoscaling', 'athena', 'autoscaling', 'awsconnector', 'cassandra',
                'cloudformation', 'codedeploy-commands-secure', 'databrew', 'dms', 'ebs', 'ec2', 'ecr.api', 'ecr.dkr',
                'elasticbeanstalk', 'elasticfilesystem', 'elasticfilesystem-fips', 'execute-api', 'imagebuilder',
                'kinesis-streams', 'lambda', 'license-manager', 'monitoring', 'rds', 'redshift', 'redshift-data', 's3',
                'sagemaker.api', 'sagemaker.featurestore-runtime', 'sagemaker.runtime', 'servicecatalog', 'sms', 'sqs',
                'states', 'sts', 'synthetics', 'transcribe', 'transcribestreaming', 'transfer', 'workspaces', 'xray'],
        };
        if ((_g = VPC_ENDPOINT_SERVICE_EXCEPTIONS[region]) === null || _g === void 0 ? void 0 : _g.includes(name)) {
            return 'cn.com.amazonaws';
        }
        else {
            return 'com.amazonaws';
        }
    }
    /**
     * Get the endpoint suffix for the service in the specified region.
     * In cn-north-1 and cn-northwest-1, the vpc endpoint of transcribe is:
     *   cn.com.amazonaws.cn-north-1.transcribe.cn
     *   cn.com.amazonaws.cn-northwest-1.transcribe.cn
     * so suffix '.cn' should be return in these scenarios.
     *
     * For future maintenance， the vpc endpoint services could be fetched using AWS CLI Commmand:
     * aws ec2 describe-vpc-endpoint-services
     */
    getDefaultEndpointSuffix(name, region) {
        var _g;
        const VPC_ENDPOINT_SERVICE_EXCEPTIONS = {
            'cn-north-1': ['transcribe'],
            'cn-northwest-1': ['transcribe'],
        };
        return ((_g = VPC_ENDPOINT_SERVICE_EXCEPTIONS[region]) === null || _g === void 0 ? void 0 : _g.includes(name)) ? '.cn' : '';
    }
}
exports.InterfaceVpcEndpointAwsService = InterfaceVpcEndpointAwsService;
_e = JSII_RTTI_SYMBOL_1;
InterfaceVpcEndpointAwsService[_e] = { fqn: "monocdk.aws_ec2.InterfaceVpcEndpointAwsService", version: "1.149.0" };
InterfaceVpcEndpointAwsService.SAGEMAKER_NOTEBOOK = new InterfaceVpcEndpointAwsService('notebook', 'aws.sagemaker');
InterfaceVpcEndpointAwsService.ATHENA = new InterfaceVpcEndpointAwsService('athena');
InterfaceVpcEndpointAwsService.CLOUDFORMATION = new InterfaceVpcEndpointAwsService('cloudformation');
InterfaceVpcEndpointAwsService.CLOUDTRAIL = new InterfaceVpcEndpointAwsService('cloudtrail');
InterfaceVpcEndpointAwsService.CODEBUILD = new InterfaceVpcEndpointAwsService('codebuild');
InterfaceVpcEndpointAwsService.CODEBUILD_FIPS = new InterfaceVpcEndpointAwsService('codebuild-fips');
InterfaceVpcEndpointAwsService.CODECOMMIT = new InterfaceVpcEndpointAwsService('codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_FIPS = new InterfaceVpcEndpointAwsService('codecommit-fips');
InterfaceVpcEndpointAwsService.CODEGURU_PROFILER = new InterfaceVpcEndpointAwsService('codeguru-profiler');
InterfaceVpcEndpointAwsService.CODEGURU_REVIEWER = new InterfaceVpcEndpointAwsService('codeguru-reviewer');
InterfaceVpcEndpointAwsService.CODEPIPELINE = new InterfaceVpcEndpointAwsService('codepipeline');
InterfaceVpcEndpointAwsService.CONFIG = new InterfaceVpcEndpointAwsService('config');
InterfaceVpcEndpointAwsService.EC2 = new InterfaceVpcEndpointAwsService('ec2');
InterfaceVpcEndpointAwsService.EC2_MESSAGES = new InterfaceVpcEndpointAwsService('ec2messages');
InterfaceVpcEndpointAwsService.ECR = new InterfaceVpcEndpointAwsService('ecr.api');
InterfaceVpcEndpointAwsService.ECR_DOCKER = new InterfaceVpcEndpointAwsService('ecr.dkr');
InterfaceVpcEndpointAwsService.ECS = new InterfaceVpcEndpointAwsService('ecs');
InterfaceVpcEndpointAwsService.ECS_AGENT = new InterfaceVpcEndpointAwsService('ecs-agent');
InterfaceVpcEndpointAwsService.ECS_TELEMETRY = new InterfaceVpcEndpointAwsService('ecs-telemetry');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM = new InterfaceVpcEndpointAwsService('elasticfilesystem');
InterfaceVpcEndpointAwsService.ELASTIC_FILESYSTEM_FIPS = new InterfaceVpcEndpointAwsService('elasticfilesystem-fips');
InterfaceVpcEndpointAwsService.ELASTIC_INFERENCE_RUNTIME = new InterfaceVpcEndpointAwsService('elastic-inference.runtime');
InterfaceVpcEndpointAwsService.ELASTIC_LOAD_BALANCING = new InterfaceVpcEndpointAwsService('elasticloadbalancing');
InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS = new InterfaceVpcEndpointAwsService('events');
InterfaceVpcEndpointAwsService.APIGATEWAY = new InterfaceVpcEndpointAwsService('execute-api');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT = new InterfaceVpcEndpointAwsService('git-codecommit');
InterfaceVpcEndpointAwsService.CODECOMMIT_GIT_FIPS = new InterfaceVpcEndpointAwsService('git-codecommit-fips');
InterfaceVpcEndpointAwsService.GLUE = new InterfaceVpcEndpointAwsService('glue');
InterfaceVpcEndpointAwsService.KEYSPACES = new InterfaceVpcEndpointAwsService('cassandra', '', 9142);
InterfaceVpcEndpointAwsService.KINESIS_STREAMS = new InterfaceVpcEndpointAwsService('kinesis-streams');
InterfaceVpcEndpointAwsService.KINESIS_FIREHOSE = new InterfaceVpcEndpointAwsService('kinesis-firehose');
InterfaceVpcEndpointAwsService.KMS = new InterfaceVpcEndpointAwsService('kms');
InterfaceVpcEndpointAwsService.CLOUDWATCH_LOGS = new InterfaceVpcEndpointAwsService('logs');
InterfaceVpcEndpointAwsService.CLOUDWATCH = new InterfaceVpcEndpointAwsService('monitoring');
InterfaceVpcEndpointAwsService.RDS = new InterfaceVpcEndpointAwsService('rds');
InterfaceVpcEndpointAwsService.RDS_DATA = new InterfaceVpcEndpointAwsService('rds-data');
InterfaceVpcEndpointAwsService.SAGEMAKER_API = new InterfaceVpcEndpointAwsService('sagemaker.api');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME = new InterfaceVpcEndpointAwsService('sagemaker.runtime');
InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME_FIPS = new InterfaceVpcEndpointAwsService('sagemaker.runtime-fips');
InterfaceVpcEndpointAwsService.SECRETS_MANAGER = new InterfaceVpcEndpointAwsService('secretsmanager');
InterfaceVpcEndpointAwsService.SERVICE_CATALOG = new InterfaceVpcEndpointAwsService('servicecatalog');
InterfaceVpcEndpointAwsService.SNS = new InterfaceVpcEndpointAwsService('sns');
InterfaceVpcEndpointAwsService.SQS = new InterfaceVpcEndpointAwsService('sqs');
InterfaceVpcEndpointAwsService.SSM = new InterfaceVpcEndpointAwsService('ssm');
InterfaceVpcEndpointAwsService.SSM_MESSAGES = new InterfaceVpcEndpointAwsService('ssmmessages');
InterfaceVpcEndpointAwsService.STS = new InterfaceVpcEndpointAwsService('sts');
InterfaceVpcEndpointAwsService.TRANSFER = new InterfaceVpcEndpointAwsService('transfer.server');
InterfaceVpcEndpointAwsService.STORAGE_GATEWAY = new InterfaceVpcEndpointAwsService('storagegateway');
InterfaceVpcEndpointAwsService.REKOGNITION = new InterfaceVpcEndpointAwsService('rekognition');
InterfaceVpcEndpointAwsService.REKOGNITION_FIPS = new InterfaceVpcEndpointAwsService('rekognition-fips');
InterfaceVpcEndpointAwsService.STEP_FUNCTIONS = new InterfaceVpcEndpointAwsService('states');
InterfaceVpcEndpointAwsService.LAMBDA = new InterfaceVpcEndpointAwsService('lambda');
InterfaceVpcEndpointAwsService.TRANSCRIBE = new InterfaceVpcEndpointAwsService('transcribe');
InterfaceVpcEndpointAwsService.XRAY = new InterfaceVpcEndpointAwsService('xray');
/**
 * A interface VPC endpoint.
 * @resource AWS::EC2::VPCEndpoint
 */
class InterfaceVpcEndpoint extends VpcEndpoint {
    constructor(scope, id, props) {
        var _g, _h;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_ec2_InterfaceVpcEndpointProps(props);
        const securityGroups = props.securityGroups || [new security_group_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
            })];
        this.securityGroupId = securityGroups[0].securityGroupId;
        this.connections = new connections_1.Connections({
            defaultPort: port_1.Port.tcp(props.service.port),
            securityGroups,
        });
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(peer_1.Peer.ipv4(props.vpc.vpcCidrBlock));
        }
        // Determine which subnets to place the endpoint in
        const subnetIds = this.endpointSubnets(props);
        const endpoint = new ec2_generated_1.CfnVPCEndpoint(this, 'Resource', {
            privateDnsEnabled: (_h = (_g = props.privateDnsEnabled) !== null && _g !== void 0 ? _g : props.service.privateDnsDefault) !== null && _h !== void 0 ? _h : true,
            policyDocument: core_1.Lazy.any({ produce: () => this.policyDocument }),
            securityGroupIds: securityGroups.map(s => s.securityGroupId),
            serviceName: props.service.name,
            vpcEndpointType: VpcEndpointType.INTERFACE,
            subnetIds,
            vpcId: props.vpc.vpcId,
        });
        this.vpcEndpointId = endpoint.ref;
        this.vpcEndpointCreationTimestamp = endpoint.attrCreationTimestamp;
        this.vpcEndpointDnsEntries = endpoint.attrDnsEntries;
        this.vpcEndpointNetworkInterfaceIds = endpoint.attrNetworkInterfaceIds;
    }
    /**
     * Imports an existing interface VPC endpoint.
     */
    static fromInterfaceVpcEndpointAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_InterfaceVpcEndpointAttributes(attrs);
        const securityGroups = attrs.securityGroupId
            ? [security_group_1.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)]
            : attrs.securityGroups;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.vpcEndpointId = attrs.vpcEndpointId;
                this.connections = new connections_1.Connections({
                    defaultPort: port_1.Port.tcp(attrs.port),
                    securityGroups,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Determine which subnets to place the endpoint in. This is in its own function
     * because there's a lot of code.
     */
    endpointSubnets(props) {
        var _g;
        const lookupSupportedAzs = (_g = props.lookupSupportedAzs) !== null && _g !== void 0 ? _g : false;
        const subnetSelection = props.vpc.selectSubnets({ ...props.subnets, onePerAz: true });
        const subnets = subnetSelection.subnets;
        // Sanity check the subnet count
        if (!subnetSelection.isPendingLookup && subnetSelection.subnets.length == 0) {
            throw new Error('Cannot create a VPC Endpoint with no subnets');
        }
        // If we aren't going to lookup supported AZs we'll exit early, returning the subnetIds from the provided subnet selection
        if (!lookupSupportedAzs) {
            return subnetSelection.subnetIds;
        }
        // Some service names, such as AWS service name references, use Tokens to automatically fill in the region
        // If it is an InterfaceVpcEndpointAwsService, then the reference will be resolvable since it only references the region
        const isAwsService = core_1.Token.isUnresolved(props.service.name) && props.service instanceof InterfaceVpcEndpointAwsService;
        // Determine what service name gets pass to the context provider
        // If it is an AWS service it will have a REGION token
        const lookupServiceName = isAwsService ? core_1.Stack.of(this).resolve(props.service.name) : props.service.name;
        // Check that the lookup will work
        this.validateCanLookupSupportedAzs(subnets, lookupServiceName);
        // Do the actual lookup for AZs
        const availableAZs = this.availableAvailabilityZones(lookupServiceName);
        const filteredSubnets = subnets.filter(s => availableAZs.includes(s.availabilityZone));
        // Throw an error if the lookup filtered out all subnets
        // VpcEndpoints must be created with at least one AZ
        if (filteredSubnets.length == 0) {
            throw new Error(`lookupSupportedAzs returned ${availableAZs} but subnets have AZs ${subnets.map(s => s.availabilityZone)}`);
        }
        return filteredSubnets.map(s => s.subnetId);
    }
    /**
     * Sanity checking when looking up AZs for an endpoint service, to make sure it won't fail
     */
    validateCanLookupSupportedAzs(subnets, serviceName) {
        // Having any of these be true will cause the AZ lookup to fail at synthesis time
        const agnosticAcct = core_1.Token.isUnresolved(this.stack.account);
        const agnosticRegion = core_1.Token.isUnresolved(this.stack.region);
        const agnosticService = core_1.Token.isUnresolved(serviceName);
        // Having subnets with Token AZs can cause the endpoint to be created with no subnets, failing at deployment time
        const agnosticSubnets = subnets.some(s => core_1.Token.isUnresolved(s.availabilityZone));
        const agnosticSubnetList = core_1.Token.isUnresolved(subnets.map(s => s.availabilityZone));
        // Context provider cannot make an AWS call without an account/region
        if (agnosticAcct || agnosticRegion) {
            throw new Error('Cannot look up VPC endpoint availability zones if account/region are not specified');
        }
        // The AWS call will fail if there is a Token in the service name
        if (agnosticService) {
            throw new Error(`Cannot lookup AZs for a service name with a Token: ${serviceName}`);
        }
        // The AWS call return strings for AZs, like us-east-1a, us-east-1b, etc
        // If the subnet AZs are Tokens, a string comparison between the subnet AZs and the AZs from the AWS call
        // will not match
        if (agnosticSubnets || agnosticSubnetList) {
            const agnostic = subnets.filter(s => core_1.Token.isUnresolved(s.availabilityZone));
            throw new Error(`lookupSupportedAzs cannot filter on subnets with Token AZs: ${agnostic}`);
        }
    }
    availableAvailabilityZones(serviceName) {
        // Here we check what AZs the endpoint service is available in
        // If for whatever reason we can't retrieve the AZs, and no context is set,
        // we will fall back to all AZs
        const availableAZs = core_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.ENDPOINT_SERVICE_AVAILABILITY_ZONE_PROVIDER,
            dummyValue: this.stack.availabilityZones,
            props: { serviceName },
        }).value;
        if (!Array.isArray(availableAZs)) {
            throw new Error(`Discovered AZs for endpoint service ${serviceName} must be an array`);
        }
        return availableAZs;
    }
}
exports.InterfaceVpcEndpoint = InterfaceVpcEndpoint;
_f = JSII_RTTI_SYMBOL_1;
InterfaceVpcEndpoint[_f] = { fqn: "monocdk.aws_ec2.InterfaceVpcEndpoint", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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