"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.flatten = exports.allRouteTableIds = exports.range = exports.ImportSubnetGroup = exports.subnetId = exports.subnetGroupNameFromConstructId = exports.defaultSubnetName = exports.slugify = void 0;
const vpc_1 = require("./vpc");
/**
 * Turn an arbitrary string into one that can be used as a CloudFormation identifier by stripping special characters
 *
 * (At the moment, no efforts are taken to prevent collisions, but we can add that later when it becomes necessary).
 */
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
exports.slugify = slugify;
/**
 * The default names for every subnet type
 */
function defaultSubnetName(type) {
    switch (type) {
        case vpc_1.SubnetType.PUBLIC: return 'Public';
        case vpc_1.SubnetType.PRIVATE_WITH_NAT: return 'Private';
        case vpc_1.SubnetType.PRIVATE_ISOLATED: return 'Isolated';
    }
}
exports.defaultSubnetName = defaultSubnetName;
/**
 * Return a subnet name from its construct ID
 *
 * All subnet names look like NAME <> "Subnet" <> INDEX
 */
function subnetGroupNameFromConstructId(subnet) {
    return subnet.node.id.replace(/Subnet\d+$/, '');
}
exports.subnetGroupNameFromConstructId = subnetGroupNameFromConstructId;
/**
 * Make the subnet construct ID from a name and number
 */
function subnetId(name, i) {
    return `${name}Subnet${i + 1}`;
}
exports.subnetId = subnetId;
class ImportSubnetGroup {
    constructor(subnetIds, names, routeTableIds, type, availabilityZones, idField, nameField, routeTableIdField) {
        this.availabilityZones = availabilityZones;
        this.subnetIds = subnetIds || [];
        this.routeTableIds = routeTableIds || [];
        this.groups = this.subnetIds.length / this.availabilityZones.length;
        if (Math.floor(this.groups) !== this.groups) {
            // eslint-disable-next-line max-len
            throw new Error(`Number of ${idField} (${this.subnetIds.length}) must be a multiple of availability zones (${this.availabilityZones.length}).`);
        }
        if (this.routeTableIds.length !== this.subnetIds.length && routeTableIds != null) {
            // We don't err if no routeTableIds were provided to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            /* eslint-disable max-len */
            throw new Error(`Number of ${routeTableIdField} (${this.routeTableIds.length}) must be equal to the amount of ${idField} (${this.subnetIds.length}).`);
        }
        this.names = this.normalizeNames(names, defaultSubnetName(type), nameField);
    }
    import(scope) {
        return range(this.subnetIds.length).map(i => {
            const k = Math.floor(i / this.availabilityZones.length);
            return vpc_1.Subnet.fromSubnetAttributes(scope, subnetId(this.names[k], i), {
                availabilityZone: this.pickAZ(i),
                subnetId: this.subnetIds[i],
                routeTableId: this.routeTableIds[i],
            });
        });
    }
    /**
     * Return a list with a name for every subnet
     */
    normalizeNames(names, defaultName, fieldName) {
        // If not given, return default
        if (names === undefined || names.length === 0) {
            return [defaultName];
        }
        // If given, must match given subnets
        if (names.length !== this.groups) {
            throw new Error(`${fieldName} must have an entry for every corresponding subnet group, got: ${JSON.stringify(names)}`);
        }
        return names;
    }
    /**
     * Return the i'th AZ
     */
    pickAZ(i) {
        return this.availabilityZones[i % this.availabilityZones.length];
    }
}
exports.ImportSubnetGroup = ImportSubnetGroup;
/**
 * Generate the list of numbers of [0..n)
 */
function range(n) {
    const ret = [];
    for (let i = 0; i < n; i++) {
        ret.push(i);
    }
    return ret;
}
exports.range = range;
/**
 * Return the union of table IDs from all selected subnets
 */
function allRouteTableIds(subnets) {
    const ret = new Set();
    for (const subnet of subnets) {
        if (subnet.routeTable && subnet.routeTable.routeTableId) {
            ret.add(subnet.routeTable.routeTableId);
        }
    }
    return Array.from(ret);
}
exports.allRouteTableIds = allRouteTableIds;
function flatten(xs) {
    return Array.prototype.concat.apply([], xs);
}
exports.flatten = flatten;
//# sourceMappingURL=data:application/json;base64,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