"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MultipartUserData = exports.MultipartBody = exports.UserData = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts
     */
    static forLinux(options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_LinuxUserDataOptions(options);
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    static forOperatingSystem(os) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_OperatingSystemType(os);
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
_a = JSII_RTTI_SYMBOL_1;
UserData[_a] = { fqn: "monocdk.aws_ec2.UserData", version: "1.149.0" };
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        var _d;
        const shebang = (_d = this.props.shebang) !== null && _d !== void 0 ? _d : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'` + (params.region !== undefined ? ` --region ${params.region}` : ''));
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim());
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = resource.node.defaultChild.logicalId;
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop` + (params.region !== undefined ? ` -Region ${params.region}` : ''));
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _d;
        this.addCommands(`&'${params.filePath}' ${(_d = params.arguments) !== null && _d !== void 0 ? _d : ''}`.trim(), `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = resource.node.defaultChild.logicalId;
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
/**
 * The base class for all classes which can be used as {@link MultipartUserData}.
 */
class MultipartBody {
    constructor() {
    }
    /**
     * Constructs the new `MultipartBody` wrapping existing `UserData`. Modification to `UserData` are reflected
     * in subsequent renders of the part.
     *
     * For more information about content types see {@link MultipartBodyOptions.contentType}.
     *
     * @param userData user data to wrap into body part
     * @param contentType optional content type, if default one should not be used
     */
    static fromUserData(userData, contentType) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_UserData(userData);
        return new MultipartBodyUserDataWrapper(userData, contentType);
    }
    /**
     * Constructs the raw `MultipartBody` using specified body, content type and transfer encoding.
     *
     * When transfer encoding is specified (typically as Base64), it's caller responsibility to convert body to
     * Base64 either by wrapping with `Fn.base64` or by converting it by other converters.
     */
    static fromRawBody(opts) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_MultipartBodyOptions(opts);
        return new MultipartBodyRaw(opts);
    }
}
exports.MultipartBody = MultipartBody;
_b = JSII_RTTI_SYMBOL_1;
MultipartBody[_b] = { fqn: "monocdk.aws_ec2.MultipartBody", version: "1.149.0" };
/**
 * Content type for shell scripts
 */
MultipartBody.SHELL_SCRIPT = 'text/x-shellscript; charset="utf-8"';
/**
 * Content type for boot hooks
 */
MultipartBody.CLOUD_BOOTHOOK = 'text/cloud-boothook; charset="utf-8"';
/**
 * The raw part of multi-part user data, which can be added to {@link MultipartUserData}.
 */
class MultipartBodyRaw extends MultipartBody {
    constructor(props) {
        super();
        this.props = props;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.props.contentType}`);
        if (this.props.transferEncoding != null) {
            result.push(`Content-Transfer-Encoding: ${this.props.transferEncoding}`);
        }
        // One line free after separator
        result.push('');
        if (this.props.body != null) {
            result.push(this.props.body);
            // The new line added after join will be consumed by encapsulating or closing boundary
        }
        return result;
    }
}
/**
 * Wrapper for `UserData`.
 */
class MultipartBodyUserDataWrapper extends MultipartBody {
    constructor(userData, contentType) {
        super();
        this.userData = userData;
        this.contentType = contentType || MultipartBody.SHELL_SCRIPT;
    }
    /**
     * Render body part as the string.
     */
    renderBodyPart() {
        const result = [];
        result.push(`Content-Type: ${this.contentType}`);
        result.push('Content-Transfer-Encoding: base64');
        result.push('');
        result.push(core_1.Fn.base64(this.userData.render()));
        return result;
    }
}
/**
 * Mime multipart user data.
 *
 * This class represents MIME multipart user data, as described in.
 * [Specifying Multiple User Data Blocks Using a MIME Multi Part Archive](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/bootstrap_container_instance.html#multi-part_user_data)
 *
 */
class MultipartUserData extends UserData {
    constructor(opts) {
        super();
        this.parts = [];
        jsiiDeprecationWarnings.monocdk_aws_ec2_MultipartUserDataOptions(opts);
        let partsSeparator;
        // Validate separator
        if ((opts === null || opts === void 0 ? void 0 : opts.partsSeparator) != null) {
            if (new RegExp(MultipartUserData.BOUNDRY_PATTERN).test(opts.partsSeparator)) {
                throw new Error(`Invalid characters in separator. Separator has to match pattern ${MultipartUserData.BOUNDRY_PATTERN}`);
            }
            else {
                partsSeparator = opts.partsSeparator;
            }
        }
        else {
            partsSeparator = '+AWS+CDK+User+Data+Separator==';
        }
        this.opts = {
            partsSeparator: partsSeparator,
        };
    }
    /**
     * Adds a part to the list of parts.
     */
    addPart(part) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_MultipartBody(part);
        this.parts.push(part);
    }
    /**
     * Adds a multipart part based on a UserData object.
     *
     * If `makeDefault` is true, then the UserData added by this method
     * will also be the target of calls to the `add*Command` methods on
     * this MultipartUserData object.
     *
     * If `makeDefault` is false, then this is the same as calling:
     *
     * ```ts
     * declare const multiPart: ec2.MultipartUserData;
     * declare const userData: ec2.UserData;
     * declare const contentType: string;
     *
     * multiPart.addPart(ec2.MultipartBody.fromUserData(userData, contentType));
     * ```
     *
     * An undefined `makeDefault` defaults to either:
     * - `true` if no default UserData has been set yet; or
     * - `false` if there is no default UserData set.
     */
    addUserDataPart(userData, contentType, makeDefault) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_UserData(userData);
        this.addPart(MultipartBody.fromUserData(userData, contentType));
        makeDefault = makeDefault !== null && makeDefault !== void 0 ? makeDefault : (this.defaultUserData === undefined ? true : false);
        if (makeDefault) {
            this.defaultUserData = userData;
        }
    }
    render() {
        const boundary = this.opts.partsSeparator;
        // Now build final MIME archive - there are few changes from MIME message which are accepted by cloud-init:
        // - MIME RFC uses CRLF to separate lines - cloud-init is fine with LF \n only
        // Note: new lines matters, matters a lot.
        var resultArchive = new Array();
        resultArchive.push(`Content-Type: multipart/mixed; boundary="${boundary}"`);
        resultArchive.push('MIME-Version: 1.0');
        // Add new line, the next one will be boundary (encapsulating or closing)
        // so this line will count into it.
        resultArchive.push('');
        // Add parts - each part starts with boundary
        this.parts.forEach(part => {
            resultArchive.push(`--${boundary}`);
            resultArchive.push(...part.renderBodyPart());
        });
        // Add closing boundary
        resultArchive.push(`--${boundary}--`);
        resultArchive.push(''); // Force new line at the end
        return resultArchive.join('\n');
    }
    addS3DownloadCommand(params) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_S3DownloadOptions(params);
        if (this.defaultUserData) {
            return this.defaultUserData.addS3DownloadCommand(params);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addExecuteFileCommand(params) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_ExecuteFileOptions(params);
        if (this.defaultUserData) {
            this.defaultUserData.addExecuteFileCommand(params);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addSignalOnExitCommand(resource) {
        jsiiDeprecationWarnings.monocdk_Resource(resource);
        if (this.defaultUserData) {
            this.defaultUserData.addSignalOnExitCommand(resource);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addCommands(...commands) {
        if (this.defaultUserData) {
            this.defaultUserData.addCommands(...commands);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
    addOnExitCommands(...commands) {
        if (this.defaultUserData) {
            this.defaultUserData.addOnExitCommands(...commands);
        }
        else {
            throw new Error(MultipartUserData.USE_PART_ERROR);
        }
    }
}
exports.MultipartUserData = MultipartUserData;
_c = JSII_RTTI_SYMBOL_1;
MultipartUserData[_c] = { fqn: "monocdk.aws_ec2.MultipartUserData", version: "1.149.0" };
MultipartUserData.USE_PART_ERROR = 'MultipartUserData only supports this operation if it has a default UserData. Call addUserDataPart with makeDefault=true.';
MultipartUserData.BOUNDRY_PATTERN = '[^a-zA-Z0-9()+,-./:=?]';
//# sourceMappingURL=data:application/json;base64,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