"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const aspects_1 = require("./aspects");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const ebs_util_1 = require("./private/ebs-util");
const security_group_1 = require("./security-group");
const vpc_1 = require("./vpc");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance
 */
class Instance extends core_1.Resource {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        this.securityGroups = [];
        jsiiDeprecationWarnings.monocdk_aws_ec2_InstanceProps(props);
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_b = props.userData) !== null && _b !== void 0 ? _b : imageConfig.userData;
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.list({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                core_1.Annotations.of(this).addError(`Need exactly 1 subnet to match AZ '${props.availabilityZone}', found ${selected.length}. Use a different availabilityZone.`);
            }
        }
        else {
            if (subnets.length > 0) {
                subnet = subnets[0];
            }
            else {
                core_1.Annotations.of(this).addError(`Did not find any subnets matching '${JSON.stringify(props.vpcSubnets)}', please use a different selection.`);
            }
        }
        if (!subnet) {
            // We got here and we don't have a subnet because of validation errors.
            // Invent one on the spot so the code below doesn't fail.
            subnet = vpc_1.Subnet.fromSubnetAttributes(this, 'DummySubnet', {
                subnetId: 's-notfound',
                availabilityZone: 'az-notfound',
            });
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? ebs_util_1.instanceBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress,
            propagateTagsToVolumeOnCreation: props.propagateTagsToVolumeOnCreation,
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.applyUpdatePolicies(props);
        // Trigger replacement (via new logical ID) on user data change, if specified or cfn-init is being used.
        //
        // This is slightly tricky -- we need to resolve the UserData string (in order to get at actual Asset hashes,
        // instead of the Token stringifications of them ('${Token[1234]}'). However, in the case of CFN Init usage,
        // a UserData is going to contain the logicalID of the resource itself, which means infinite recursion if we
        // try to naively resolve. We need a recursion breaker in this.
        const originalLogicalId = core_1.Stack.of(this).getLogicalId(this.instance);
        let recursing = false;
        this.instance.overrideLogicalId(core_1.Lazy.uncachedString({
            produce: (context) => {
                var _b;
                if (recursing) {
                    return originalLogicalId;
                }
                if (!((_b = props.userDataCausesReplacement) !== null && _b !== void 0 ? _b : props.initOptions)) {
                    return originalLogicalId;
                }
                const md5 = crypto.createHash('md5');
                recursing = true;
                try {
                    md5.update(JSON.stringify(context.resolve(this.userData.render())));
                }
                finally {
                    recursing = false;
                }
                const digest = md5.digest('hex').substr(0, 16);
                return `${originalLogicalId}${digest}`;
            },
        }));
        if (props.requireImdsv2) {
            core_1.Aspects.of(this).add(new aspects_1.InstanceRequireImdsv2Aspect());
        }
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_ISecurityGroup(securityGroup);
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the Instance resource.
     * - Add commands to the instance UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for the `cfn-signal` commands.
     */
    applyCloudFormationInit(init, options = {}) {
        var _b;
        init.attach(this.instance, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
            includeRole: options.includeRole,
            includeUrl: options.includeUrl,
        });
        this.waitForResourceSignal((_b = options.timeout) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(5));
    }
    /**
     * Wait for a single additional resource signal
     *
     * Add 1 to the current ResourceSignal Count and add the given timeout to the current timeout.
     *
     * Use this to pause the CloudFormation deployment to wait for the instances
     * in the AutoScalingGroup to report successful startup during
     * creation and updates. The UserData script needs to invoke `cfn-signal`
     * with a success or failure code after it is done setting up the instance.
     */
    waitForResourceSignal(timeout) {
        var _b, _c;
        const oldResourceSignal = (_b = this.instance.cfnOptions.creationPolicy) === null || _b === void 0 ? void 0 : _b.resourceSignal;
        this.instance.cfnOptions.creationPolicy = {
            ...this.instance.cfnOptions.creationPolicy,
            resourceSignal: {
                count: ((_c = oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.count) !== null && _c !== void 0 ? _c : 0) + 1,
                timeout: ((oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout) ? core_1.Duration.parse(oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout).plus(timeout) : timeout).toIsoString(),
            },
        };
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toIsoString(),
                },
            };
        }
    }
}
exports.Instance = Instance;
_a = JSII_RTTI_SYMBOL_1;
Instance[_a] = { fqn: "monocdk.aws_ec2.Instance", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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