"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.InitConfig = exports.CloudFormationInit = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const machine_image_1 = require("./machine-image");
const cfn_init_internal_1 = require("./private/cfn-init-internal");
/**
 * A CloudFormation-init configuration
 */
class CloudFormationInit {
    constructor(configSets, configs) {
        this._configSets = {};
        this._configs = {};
        Object.assign(this._configSets, configSets);
        Object.assign(this._configs, configs);
    }
    /**
     * Build a new config from a set of Init Elements
     */
    static fromElements(...elements) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_InitElement(elements);
        return CloudFormationInit.fromConfig(new InitConfig(elements));
    }
    /**
     * Use an existing InitConfig object as the default and only config
     */
    static fromConfig(config) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_InitConfig(config);
        return CloudFormationInit.fromConfigSets({
            configSets: {
                default: ['config'],
            },
            configs: { config },
        });
    }
    /**
     * Build a CloudFormationInit from config sets
     */
    static fromConfigSets(props) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_ConfigSetProps(props);
        return new CloudFormationInit(props.configSets, props.configs);
    }
    /**
     * Add a config with the given name to this CloudFormationInit object
     */
    addConfig(configName, config) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_InitConfig(config);
        if (this._configs[configName]) {
            throw new Error(`CloudFormationInit already contains a config named '${configName}'`);
        }
        this._configs[configName] = config;
    }
    /**
     * Add a config set with the given name to this CloudFormationInit object
     *
     * The new configset will reference the given configs in the given order.
     */
    addConfigSet(configSetName, configNames = []) {
        if (this._configSets[configSetName]) {
            throw new Error(`CloudFormationInit already contains a configSet named '${configSetName}'`);
        }
        const unk = configNames.filter(c => !this._configs[c]);
        if (unk.length > 0) {
            throw new Error(`Unknown configs referenced in definition of '${configSetName}': ${unk}`);
        }
        this._configSets[configSetName] = [...configNames];
    }
    /**
     * Attach the CloudFormation Init config to the given resource
     *
     * As an app builder, use `instance.applyCloudFormationInit()` or
     * `autoScalingGroup.applyCloudFormationInit()` to trigger this method.
     *
     * This method does the following:
     *
     * - Renders the `AWS::CloudFormation::Init` object to the given resource's
     *   metadata, potentially adding a `AWS::CloudFormation::Authentication` object
     *   next to it if required.
     * - Updates the instance role policy to be able to call the APIs required for
     *   `cfn-init` and `cfn-signal` to work, and potentially add permissions to download
     *   referenced asset and bucket resources.
     * - Updates the given UserData with commands to execute the `cfn-init` script.
     */
    attach(attachedResource, attachOptions) {
        var _c, _d, _e, _f, _g;
        jsiiDeprecationWarnings.monocdk_CfnResource(attachedResource);
        jsiiDeprecationWarnings.monocdk_aws_ec2_AttachInitOptions(attachOptions);
        if (attachOptions.platform === machine_image_1.OperatingSystemType.UNKNOWN) {
            throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
        }
        const CFN_INIT_METADATA_KEY = 'AWS::CloudFormation::Init';
        if (attachedResource.getMetadata(CFN_INIT_METADATA_KEY) !== undefined) {
            throw new Error(`Cannot bind CfnInit: resource '${attachedResource.node.path}' already has '${CFN_INIT_METADATA_KEY}' attached`);
        }
        // Note: This will not reflect mutations made after attaching.
        const bindResult = this.bind(attachedResource.stack, attachOptions);
        attachedResource.addMetadata(CFN_INIT_METADATA_KEY, bindResult.configData);
        // Need to resolve the various tokens from assets in the config,
        // as well as include any asset hashes provided so the fingerprint is accurate.
        const resolvedConfig = attachedResource.stack.resolve(bindResult.configData);
        const fingerprintInput = { config: resolvedConfig, assetHash: bindResult.assetHash };
        const fingerprint = contentHash(JSON.stringify(fingerprintInput)).substr(0, 16);
        attachOptions.instanceRole.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudformation:DescribeStackResource', 'cloudformation:SignalResource'],
            resources: [core_1.Aws.STACK_ID],
        }));
        if (bindResult.authData) {
            attachedResource.addMetadata('AWS::CloudFormation::Authentication', bindResult.authData);
        }
        // To identify the resources that have the metadata and where the signal
        // needs to be sent, we need { region, stackName, logicalId }
        let resourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${attachedResource.logicalId}`;
        const signalResource = (_d = (_c = attachOptions.signalResource) === null || _c === void 0 ? void 0 : _c.logicalId) !== null && _d !== void 0 ? _d : attachedResource.logicalId;
        let notifyResourceLocator = `--region ${core_1.Aws.REGION} --stack ${core_1.Aws.STACK_NAME} --resource ${signalResource}`;
        // If specified in attachOptions, include arguments in cfn-init/cfn-signal commands
        if (attachOptions.includeUrl) {
            resourceLocator = `${resourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
            notifyResourceLocator = `${notifyResourceLocator} --url https://cloudformation.${core_1.Aws.REGION}.${core_1.Aws.URL_SUFFIX}`;
        }
        if (attachOptions.includeRole) {
            resourceLocator = `${resourceLocator} --role ${attachOptions.instanceRole.roleName}`;
            notifyResourceLocator = `${notifyResourceLocator} --role ${attachOptions.instanceRole.roleName}`;
        }
        const configSets = ((_e = attachOptions.configSets) !== null && _e !== void 0 ? _e : ['default']).join(',');
        const printLog = (_f = attachOptions.printLog) !== null && _f !== void 0 ? _f : true;
        if ((_g = attachOptions.embedFingerprint) !== null && _g !== void 0 ? _g : true) {
            // It just so happens that the comment char is '#' for both bash and PowerShell
            attachOptions.userData.addCommands(`# fingerprint: ${fingerprint}`);
        }
        if (attachOptions.platform === machine_image_1.OperatingSystemType.WINDOWS) {
            const errCode = attachOptions.ignoreFailures ? '0' : '$LASTEXITCODE';
            attachOptions.userData.addCommands(...[
                `cfn-init.exe -v ${resourceLocator} -c ${configSets}`,
                `cfn-signal.exe -e ${errCode} ${notifyResourceLocator}`,
                ...(printLog ? ['type C:\\cfn\\log\\cfn-init.log'] : []),
            ]);
        }
        else {
            const errCode = attachOptions.ignoreFailures ? '0' : '$?';
            attachOptions.userData.addCommands(...[
                // Run a subshell without 'errexit', so we can signal using the exit code of cfn-init
                '(',
                '  set +e',
                `  /opt/aws/bin/cfn-init -v ${resourceLocator} -c ${configSets}`,
                `  /opt/aws/bin/cfn-signal -e ${errCode} ${notifyResourceLocator}`,
                ...(printLog ? ['  cat /var/log/cfn-init.log >&2'] : []),
                ')',
            ]);
        }
    }
    bind(scope, options) {
        const nonEmptyConfigs = mapValues(this._configs, c => c.isEmpty() ? undefined : c);
        const configNameToBindResult = mapValues(nonEmptyConfigs, c => c._bind(scope, options));
        return {
            configData: {
                configSets: mapValues(this._configSets, configNames => configNames.filter(name => nonEmptyConfigs[name] !== undefined)),
                ...mapValues(configNameToBindResult, c => c.config),
            },
            authData: Object.values(configNameToBindResult).map(c => c.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(Object.values(configNameToBindResult).map(c => c.assetHash)),
        };
    }
}
exports.CloudFormationInit = CloudFormationInit;
_a = JSII_RTTI_SYMBOL_1;
CloudFormationInit[_a] = { fqn: "monocdk.aws_ec2.CloudFormationInit", version: "1.149.0" };
/**
 * A collection of configuration elements
 */
class InitConfig {
    constructor(elements) {
        this.elements = new Array();
        this.add(...elements);
    }
    /**
     * Whether this configset has elements or not
     */
    isEmpty() {
        return this.elements.length === 0;
    }
    /**
     * Add one or more elements to the config
     */
    add(...elements) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_InitElement(elements);
        this.elements.push(...elements);
    }
    /**
     * Called when the config is applied to an instance.
     * Creates the CloudFormation representation of the Init config and handles any permissions and assets.
     * @internal
     */
    _bind(scope, options) {
        const bindOptions = {
            instanceRole: options.instanceRole,
            platform: this.initPlatformFromOSType(options.platform),
            scope,
        };
        const packageConfig = this.bindForType(cfn_init_internal_1.InitElementType.PACKAGE, bindOptions);
        const groupsConfig = this.bindForType(cfn_init_internal_1.InitElementType.GROUP, bindOptions);
        const usersConfig = this.bindForType(cfn_init_internal_1.InitElementType.USER, bindOptions);
        const sourcesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SOURCE, bindOptions);
        const filesConfig = this.bindForType(cfn_init_internal_1.InitElementType.FILE, bindOptions);
        const commandsConfig = this.bindForType(cfn_init_internal_1.InitElementType.COMMAND, bindOptions);
        // Must be last!
        const servicesConfig = this.bindForType(cfn_init_internal_1.InitElementType.SERVICE, bindOptions);
        const allConfig = [packageConfig, groupsConfig, usersConfig, sourcesConfig, filesConfig, commandsConfig, servicesConfig];
        const authentication = allConfig.map(c => c === null || c === void 0 ? void 0 : c.authentication).reduce(deepMerge, undefined);
        const assetHash = combineAssetHashesOrUndefined(allConfig.map(c => c === null || c === void 0 ? void 0 : c.assetHash));
        return {
            config: {
                packages: packageConfig === null || packageConfig === void 0 ? void 0 : packageConfig.config,
                groups: groupsConfig === null || groupsConfig === void 0 ? void 0 : groupsConfig.config,
                users: usersConfig === null || usersConfig === void 0 ? void 0 : usersConfig.config,
                sources: sourcesConfig === null || sourcesConfig === void 0 ? void 0 : sourcesConfig.config,
                files: filesConfig === null || filesConfig === void 0 ? void 0 : filesConfig.config,
                commands: commandsConfig === null || commandsConfig === void 0 ? void 0 : commandsConfig.config,
                services: servicesConfig === null || servicesConfig === void 0 ? void 0 : servicesConfig.config,
            },
            authentication,
            assetHash,
        };
    }
    bindForType(elementType, renderOptions) {
        var _c;
        const elements = this.elements.filter(elem => elem.elementType === elementType);
        if (elements.length === 0) {
            return undefined;
        }
        const bindResults = elements.map((e, index) => e._bind({ index, ...renderOptions }));
        return {
            config: (_c = bindResults.map(r => r.config).reduce(deepMerge, undefined)) !== null && _c !== void 0 ? _c : {},
            authentication: bindResults.map(r => r.authentication).reduce(deepMerge, undefined),
            assetHash: combineAssetHashesOrUndefined(bindResults.map(r => r.assetHash)),
        };
    }
    initPlatformFromOSType(osType) {
        switch (osType) {
            case machine_image_1.OperatingSystemType.LINUX: {
                return cfn_init_internal_1.InitPlatform.LINUX;
            }
            case machine_image_1.OperatingSystemType.WINDOWS: {
                return cfn_init_internal_1.InitPlatform.WINDOWS;
            }
            default: {
                throw new Error('Cannot attach CloudFormationInit to an unknown OS type');
            }
        }
    }
}
exports.InitConfig = InitConfig;
_b = JSII_RTTI_SYMBOL_1;
InitConfig[_b] = { fqn: "monocdk.aws_ec2.InitConfig", version: "1.149.0" };
/**
 * Deep-merge objects and arrays
 *
 * Treat arrays as sets, removing duplicates. This is acceptable for rendering
 * cfn-inits, not applicable elsewhere.
 */
function deepMerge(target, src) {
    var _c, _d;
    if (target == null) {
        return src;
    }
    if (src == null) {
        return target;
    }
    for (const [key, value] of Object.entries(src)) {
        if (Array.isArray(value)) {
            if (target[key] && !Array.isArray(target[key])) {
                throw new Error(`Trying to merge array [${value}] into a non-array '${target[key]}'`);
            }
            target[key] = Array.from(new Set([
                ...(_c = target[key]) !== null && _c !== void 0 ? _c : [],
                ...value,
            ]));
            continue;
        }
        if (typeof value === 'object' && value) {
            target[key] = deepMerge((_d = target[key]) !== null && _d !== void 0 ? _d : {}, value);
            continue;
        }
        if (value !== undefined) {
            target[key] = value;
        }
    }
    return target;
}
/**
 * Map a function over values of an object
 *
 * If the mapping function returns undefined, remove the key
 */
function mapValues(xs, fn) {
    const ret = {};
    for (const [k, v] of Object.entries(xs)) {
        const mapped = fn(v);
        if (mapped !== undefined) {
            ret[k] = mapped;
        }
    }
    return ret;
}
// Combines all input asset hashes into one, or if no hashes are present, returns undefined.
function combineAssetHashesOrUndefined(hashes) {
    const hashArray = hashes.filter((x) => x !== undefined);
    return hashArray.length > 0 ? hashArray.join('') : undefined;
}
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
//# sourceMappingURL=data:application/json;base64,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