"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableClass = exports.StreamViewType = exports.ProjectionType = exports.BillingMode = exports.AttributeType = exports.Table = exports.TableEncryption = exports.Operation = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("../../aws-applicationautoscaling");
const cloudwatch = require("../../aws-cloudwatch");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const dynamodb_canned_metrics_generated_1 = require("./dynamodb-canned-metrics.generated");
const dynamodb_generated_1 = require("./dynamodb.generated");
const perms = require("./perms");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("../../core");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
/**
 * Supported DynamoDB table operations.
 */
var Operation;
(function (Operation) {
    /** GetItem */
    Operation["GET_ITEM"] = "GetItem";
    /** BatchGetItem */
    Operation["BATCH_GET_ITEM"] = "BatchGetItem";
    /** Scan */
    Operation["SCAN"] = "Scan";
    /** Query */
    Operation["QUERY"] = "Query";
    /** GetRecords */
    Operation["GET_RECORDS"] = "GetRecords";
    /** PutItem */
    Operation["PUT_ITEM"] = "PutItem";
    /** DeleteItem */
    Operation["DELETE_ITEM"] = "DeleteItem";
    /** UpdateItem */
    Operation["UPDATE_ITEM"] = "UpdateItem";
    /** BatchWriteItem */
    Operation["BATCH_WRITE_ITEM"] = "BatchWriteItem";
    /** TransactWriteItems */
    Operation["TRANSACT_WRITE_ITEMS"] = "TransactWriteItems";
    /** TransactGetItems */
    Operation["TRANSACT_GET_ITEMS"] = "TransactGetItems";
    /** ExecuteTransaction */
    Operation["EXECUTE_TRANSACTION"] = "ExecuteTransaction";
    /** BatchExecuteStatement */
    Operation["BATCH_EXECUTE_STATEMENT"] = "BatchExecuteStatement";
    /** ExecuteStatement */
    Operation["EXECUTE_STATEMENT"] = "ExecuteStatement";
})(Operation = exports.Operation || (exports.Operation = {}));
/**
 * What kind of server-side encryption to apply to this table.
 */
var TableEncryption;
(function (TableEncryption) {
    /**
     * Server-side KMS encryption with a master key owned by AWS.
     */
    TableEncryption["DEFAULT"] = "AWS_OWNED";
    /**
     * Server-side KMS encryption with a customer master key managed by customer.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     *
     * > **NOTE**: if `encryptionKey` is not specified and the `Table` construct creates
     * > a KMS key for you, the key will be created with default permissions. If you are using
     * > CDKv2, these permissions will be sufficient to enable the key for use with DynamoDB tables.
     * > If you are using CDKv1, make sure the feature flag `@aws-cdk/aws-kms:defaultKeyPolicies`
     * > is set to `true` in your `cdk.json`.
     */
    TableEncryption["CUSTOMER_MANAGED"] = "CUSTOMER_MANAGED";
    /**
     * Server-side KMS encryption with a master key managed by AWS.
     */
    TableEncryption["AWS_MANAGED"] = "AWS_MANAGED";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
class TableBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.regionalArns = new Array();
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan, DescribeTable.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.DESCRIBE_TABLE);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, tableActions });
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, streamActions: perms.READ_STREAM_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem, DescribeTable.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee) {
        const tableActions = perms.WRITE_DATA_ACTIONS.concat(perms.DESCRIBE_TABLE);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem, DescribeTable
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.WRITE_DATA_ACTIONS).concat(perms.DESCRIBE_TABLE);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee) {
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions: ['dynamodb:*'] });
    }
    /**
     * Return the given named metric for this Table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensionsMap: {
                TableName: this.tableName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the consumed read capacity units this table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricConsumedReadCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedReadCapacityUnitsSum, props);
    }
    /**
     * Metric for the consumed write capacity units this table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedWriteCapacityUnitsSum, props);
    }
    /**
     * Metric for the system errors this table
     *
     * @deprecated use `metricSystemErrorsForOperations`.
     */
    metricSystemErrors(props) {
        var _b, _c, _d, _e;
        if (!((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) && !((_c = props === null || props === void 0 ? void 0 : props.dimensionsMap) === null || _c === void 0 ? void 0 : _c.Operation)) {
            // 'Operation' must be passed because its an operational metric.
            throw new Error("'Operation' dimension must be passed for the 'SystemErrors' metric.");
        }
        const dimensionsMap = {
            TableName: this.tableName,
            ...(_d = props === null || props === void 0 ? void 0 : props.dimensions) !== null && _d !== void 0 ? _d : {},
            ...(_e = props === null || props === void 0 ? void 0 : props.dimensionsMap) !== null && _e !== void 0 ? _e : {},
        };
        return this.metric('SystemErrors', { statistic: 'sum', ...props, dimensionsMap });
    }
    /**
     * Metric for the user errors. Note that this metric reports user errors across all
     * the tables in the account and region the table resides in.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricUserErrors(props) {
        if (props === null || props === void 0 ? void 0 : props.dimensions) {
            throw new Error("'dimensions' is not supported for the 'UserErrors' metric");
        }
        // overriding 'dimensions' here because this metric is an account metric.
        // see 'UserErrors' in https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/metrics-dimensions.html
        return this.metric('UserErrors', { statistic: 'sum', ...props, dimensionsMap: {} });
    }
    /**
     * Metric for the conditional check failed requests this table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * How many requests are throttled on this table
     *
     * Default: sum over 5 minutes
     *
     * @deprecated Do not use this function. It returns an invalid metric. Use `metricThrottledRequestsForOperation` instead.
     */
    metricThrottledRequests(props) {
        return this.metric('ThrottledRequests', { statistic: 'sum', ...props });
    }
    /**
     * How many requests are throttled on this table, for the given operation
     *
     * Default: sum over 5 minutes
     */
    metricThrottledRequestsForOperation(operation, props) {
        return new cloudwatch.Metric({
            ...dynamodb_canned_metrics_generated_1.DynamoDBMetrics.throttledRequestsSum({ Operation: operation, TableName: this.tableName }),
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the successful request latency this table.
     *
     * By default, the metric will be calculated as an average over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricSuccessfulRequestLatency(props) {
        var _b, _c, _d, _e, _f;
        if (!((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) && !((_c = props === null || props === void 0 ? void 0 : props.dimensionsMap) === null || _c === void 0 ? void 0 : _c.Operation)) {
            throw new Error("'Operation' dimension must be passed for the 'SuccessfulRequestLatency' metric.");
        }
        const dimensionsMap = {
            TableName: this.tableName,
            Operation: (_e = (_d = props.dimensionsMap) === null || _d === void 0 ? void 0 : _d.Operation) !== null && _e !== void 0 ? _e : (_f = props.dimensions) === null || _f === void 0 ? void 0 : _f.Operation,
        };
        return new cloudwatch.Metric({
            ...dynamodb_canned_metrics_generated_1.DynamoDBMetrics.successfulRequestLatencyAverage(dimensionsMap),
            ...props,
            dimensionsMap,
        }).attachTo(this);
    }
    /**
     * Metric for the system errors this table.
     *
     * This will sum errors across all possible operations.
     * Note that by default, each individual metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricSystemErrorsForOperations(props) {
        var _b, _c;
        if ((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) {
            throw new Error("The Operation dimension is not supported. Use the 'operations' property.");
        }
        const operations = (_c = props === null || props === void 0 ? void 0 : props.operations) !== null && _c !== void 0 ? _c : Object.values(Operation);
        const values = this.createMetricsForOperations('SystemErrors', operations, { statistic: 'sum', ...props });
        const sum = new cloudwatch.MathExpression({
            expression: `${Object.keys(values).join(' + ')}`,
            usingMetrics: { ...values },
            color: props === null || props === void 0 ? void 0 : props.color,
            label: 'Sum of errors across all operations',
            period: props === null || props === void 0 ? void 0 : props.period,
        });
        return sum;
    }
    /**
     * Create a map of metrics that can be used in a math expression.
     *
     * Using the return value of this function as the `usingMetrics` property in `cloudwatch.MathExpression` allows you to
     * use the keys of this map as metric names inside you expression.
     *
     * @param metricName The metric name.
     * @param operations The list of operations to create metrics for.
     * @param props Properties for the individual metrics.
     * @param metricNameMapper Mapper function to allow controlling the individual metric name per operation.
     */
    createMetricsForOperations(metricName, operations, props, metricNameMapper) {
        var _b;
        const metrics = {};
        const mapper = metricNameMapper !== null && metricNameMapper !== void 0 ? metricNameMapper : (op => op.toLowerCase());
        if ((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) {
            throw new Error('Invalid properties. Operation dimension is not supported when calculating operational metrics');
        }
        for (const operation of operations) {
            const metric = this.metric(metricName, {
                ...props,
                dimensionsMap: {
                    TableName: this.tableName,
                    Operation: operation,
                    ...props === null || props === void 0 ? void 0 : props.dimensions,
                },
            });
            const operationMetricName = mapper(operation);
            const firstChar = operationMetricName.charAt(0);
            if (firstChar === firstChar.toUpperCase()) {
                // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax
                throw new Error(`Mapper generated an illegal operation metric name: ${operationMetricName}. Must start with a lowercase letter`);
            }
            metrics[operationMetricName] = metric;
        }
        return metrics;
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param opts Options for keyActions, tableActions and streamActions
     */
    combinedGrant(grantee, opts) {
        if (opts.tableActions) {
            const resources = [this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                }))];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.tableActions,
                resourceArns: resources,
                scope: this,
            });
            if (this.encryptionKey && opts.keyActions) {
                this.encryptionKey.grant(grantee, ...opts.keyActions);
            }
            return ret;
        }
        if (opts.streamActions) {
            if (!this.tableStreamArn) {
                throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
            }
            const resources = [this.tableStreamArn];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.streamActions,
                resourceArns: resources,
                scope: this,
            });
            return ret;
        }
        throw new Error(`Unexpected 'action', ${opts.tableActions || opts.streamActions}`);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ TableName: this.tableName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Provides a DynamoDB table.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexSchemas = new Map();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        this.globalReplicaCustomResources = new Array();
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_TableProps(props);
        const { sseSpecification, encryptionKey } = this.parseEncryption(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            this.billingMode = (_b = props.billingMode) !== null && _b !== void 0 ? _b : BillingMode.PAY_PER_REQUEST;
        }
        else {
            this.billingMode = (_c = props.billingMode) !== null && _c !== void 0 ? _c : BillingMode.PROVISIONED;
            if (props.stream) {
                streamSpecification = { streamViewType: props.stream };
            }
        }
        this.validateProvisioning(props);
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.any({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.any({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
            sseSpecification,
            streamSpecification,
            tableClass: props.tableClass,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined,
            contributorInsightsSpecification: props.contributorInsightsEnabled !== undefined ? { enabled: props.contributorInsightsEnabled } : undefined,
            kinesisStreamSpecification: props.kinesisStream ? { streamArn: props.kinesisStream.streamArn } : undefined,
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        this.encryptionKey = encryptionKey;
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', this.tableName);
        }
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions && props.replicationRegions.length > 0) {
            this.createReplicaTables(props.replicationRegions, props.replicationTimeout, props.waitForReplicationToFinish);
        }
    }
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee) {
        jsiiDeprecationWarnings.print("monocdk.aws_dynamodb.Table#grantListStreams", "Use {@link #grantTableListStreams} for more granular permission");
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_TableAttributes(attrs);
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                var _b, _c;
                super(scope, id);
                this.hasIndex = ((_b = attrs.globalIndexes) !== null && _b !== void 0 ? _b : []).length > 0 ||
                    ((_c = attrs.localIndexes) !== null && _c !== void 0 ? _c : []).length > 0;
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.splitArn(attrs.tableArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_GlobalSecondaryIndexProps(props);
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
        });
        this.secondaryIndexSchemas.set(props.indexName, {
            partitionKey: props.partitionKey,
            sortKey: props.sortKey,
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_LocalSecondaryIndexProps(props);
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection,
        });
        this.secondaryIndexSchemas.set(props.indexName, {
            partitionKey: this.tablePartitionKey,
            sortKey: props.sortKey,
        });
    }
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_EnableScalingProps(props);
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_EnableScalingProps(props);
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
        for (const globalReplicaCustomResource of this.globalReplicaCustomResources) {
            globalReplicaCustomResource.node.addDependency(this.tableScaling.scalableWriteAttribute);
        }
        return this.tableScaling.scalableWriteAttribute;
    }
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_EnableScalingProps(props);
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_EnableScalingProps(props);
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Get schema attributes of table or index.
     *
     * @returns Schema of table or index.
     */
    schema(indexName) {
        if (!indexName) {
            return {
                partitionKey: this.tablePartitionKey,
                sortKey: this.tableSortKey,
            };
        }
        let schema = this.secondaryIndexSchemas.get(indexName);
        if (!schema) {
            throw new Error(`Cannot find schema for index: ${indexName}. Use 'addGlobalSecondaryIndex' or 'addLocalSecondaryIndex' to add index`);
        }
        return schema;
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        if (this.globalReplicaCustomResources.length > 0 && this.billingMode === BillingMode.PROVISIONED) {
            const writeAutoScaleAttribute = this.tableScaling.scalableWriteAttribute;
            if (!writeAutoScaleAttribute) {
                errors.push('A global Table that uses PROVISIONED as the billing mode needs auto-scaled write capacity. ' +
                    'Use the autoScaleWriteCapacity() method to enable it.');
            }
            else if (!writeAutoScaleAttribute._scalingPolicyCreated) {
                errors.push('A global Table that uses PROVISIONED as the billing mode needs auto-scaled write capacity with a policy. ' +
                    'Call one of the scaleOn*() methods of the object returned from autoScaleWriteCapacity()');
            }
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexSchemas.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 100) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 100');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE },
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        var _b, _c;
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: (_b = props.projectionType) !== null && _b !== void 0 ? _b : ProjectionType.ALL,
            nonKeyAttributes: (_c = props.nonKeyAttributes) !== null && _c !== void 0 ? _c : undefined,
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType,
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type,
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable',
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions, timeout, waitForReplicationToFinish) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this, { timeout });
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        const onEventHandlerPolicy = new SourceTableAttachedPolicy(this, provider.onEventHandler.role);
        const isCompleteHandlerPolicy = new SourceTableAttachedPolicy(this, provider.isCompleteHandler.role);
        // Permissions in the source region
        this.grant(onEventHandlerPolicy, 'dynamodb:*');
        this.grant(isCompleteHandlerPolicy, 'dynamodb:DescribeTable');
        let previousRegion;
        let previousRegionCondition;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new core_1.CustomResource(this, `Replica${region}`, {
                serviceToken: provider.provider.serviceToken,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                    SkipReplicationCompletedWait: waitForReplicationToFinish == null
                        ? undefined
                        // CFN changes Custom Resource properties to strings anyways,
                        // so let's do that ourselves to make it clear in the handler this is a string, not a boolean
                        : (!waitForReplicationToFinish).toString(),
                },
            });
            currentRegion.node.addDependency(onEventHandlerPolicy.policy, isCompleteHandlerPolicy.policy);
            this.globalReplicaCustomResources.push(currentRegion);
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            let createReplica;
            if (core_1.Token.isUnresolved(stack.region)) {
                createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION)),
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // Save regional arns for grantXxx() methods
            this.regionalArns.push(stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName,
            }));
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is in an ACTIVE state.
            if (previousRegion) {
                if (previousRegionCondition) {
                    // we can't simply use a Dependency,
                    // because the previousRegion is protected by the "different region" Condition,
                    // and you can't have Fn::If in DependsOn.
                    // Instead, rely on Ref adding a dependency implicitly!
                    const previousRegionCfnResource = previousRegion.node.defaultChild;
                    const currentRegionCfnResource = currentRegion.node.defaultChild;
                    currentRegionCfnResource.addMetadata('DynamoDbReplicationDependency', core_1.Fn.conditionIf(previousRegionCondition.logicalId, previousRegionCfnResource.ref, core_1.Aws.NO_VALUE));
                }
                else {
                    currentRegion.node.addDependency(previousRegion);
                }
            }
            previousRegion = currentRegion;
            previousRegionCondition = createReplica;
        }
        // Permissions in the destination regions (outside of the loop to
        // minimize statements in the policy)
        onEventHandlerPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: this.regionalArns,
        }));
    }
    /**
     * Whether this table has indexes
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
    /**
     * Set up key properties and return the Table encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _b;
        let encryptionType = props.encryption;
        if (encryptionType != null && props.serverSideEncryption != null) {
            throw new Error('Only one of encryption and serverSideEncryption can be specified, but both were provided');
        }
        if (props.serverSideEncryption && props.encryptionKey) {
            throw new Error('encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead');
        }
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey != null
                // If there is a configured encryptionKey, the encryption is implicitly CUSTOMER_MANAGED
                ? TableEncryption.CUSTOMER_MANAGED
                // Otherwise, if severSideEncryption is enabled, it's AWS_MANAGED; else undefined (do not set anything)
                : props.serverSideEncryption ? TableEncryption.AWS_MANAGED : undefined;
        }
        if (encryptionType !== TableEncryption.CUSTOMER_MANAGED && props.encryptionKey) {
            throw new Error('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
        }
        if (encryptionType === TableEncryption.CUSTOMER_MANAGED && props.replicationRegions) {
            throw new Error('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
        }
        switch (encryptionType) {
            case TableEncryption.CUSTOMER_MANAGED:
                const encryptionKey = (_b = props.encryptionKey) !== null && _b !== void 0 ? _b : new kms.Key(this, 'Key', {
                    description: `Customer-managed key auto-created for encrypting DynamoDB table at ${this.node.path}`,
                    enableKeyRotation: true,
                });
                return {
                    sseSpecification: { sseEnabled: true, kmsMasterKeyId: encryptionKey.keyArn, sseType: 'KMS' },
                    encryptionKey,
                };
            case TableEncryption.AWS_MANAGED:
                // Not specifying "sseType: 'KMS'" here because it would cause phony changes to existing stacks.
                return { sseSpecification: { sseEnabled: true } };
            case TableEncryption.DEFAULT:
                return { sseSpecification: { sseEnabled: false } };
            case undefined:
                // Not specifying "sseEnabled: false" here because it would cause phony changes to existing stacks.
                return { sseSpecification: undefined };
            default:
                throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
        }
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "monocdk.aws_dynamodb.Table", version: "1.149.0" };
/**
 * Data types for attributes within a table
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 */
var AttributeType;
(function (AttributeType) {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    AttributeType["BINARY"] = "B";
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    AttributeType["NUMBER"] = "N";
    /** Up to 400KiB of UTF-8 encoded text */
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DynamoDB's Read/Write capacity modes.
 */
var BillingMode;
(function (BillingMode) {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    /**
     * Explicitly specified Read/Write capacity units.
     */
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
/**
 * The set of attributes that are projected into the index
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 */
var ProjectionType;
(function (ProjectionType) {
    /** Only the index and primary keys are projected into the index. */
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    ProjectionType["INCLUDE"] = "INCLUDE";
    /** All of the table attributes are projected into the index. */
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
var StreamViewType;
(function (StreamViewType) {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    /** Both the new and the old item images of the item are written to the stream. */
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    /** Only the key attributes of the modified item are written to the stream. */
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
/**
 * DynamoDB's table class.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.TableClasses.html
 */
var TableClass;
(function (TableClass) {
    /** Default table class for DynamoDB. */
    TableClass["STANDARD"] = "STANDARD";
    /** Table class for DynamoDB that reduces storage costs compared to existing DynamoDB Standard tables. */
    TableClass["STANDARD_INFREQUENT_ACCESS"] = "STANDARD_INFREQUENT_ACCESS";
})(TableClass = exports.TableClass || (exports.TableClass = {}));
/**
 * An inline policy that is logically bound to the source table of a DynamoDB Global Tables
 * "cluster". This is here to ensure permissions are removed as part of (and not before) the
 * CleanUp phase of a stack update, when a replica is removed (or the entire "cluster" gets
 * replaced).
 *
 * If statements are added directly to the handler roles (as opposed to in a separate inline
 * policy resource), new permissions are in effect before clean up happens, and so replicas that
 * need to be dropped can no longer be due to lack of permissions.
 */
class SourceTableAttachedPolicy extends core_2.Construct {
    constructor(sourceTable, role) {
        super(sourceTable, `SourceTableAttachedManagedPolicy-${core_1.Names.nodeUniqueId(role.node)}`);
        const policy = new iam.ManagedPolicy(this, 'Resource', {
            // A CF update of the description property of a managed policy requires
            // a replacement. Use the table name in the description to force a managed
            // policy replacement when the table name changes. This way we preserve permissions
            // to delete old replicas in case of a table replacement.
            description: `DynamoDB replication managed policy for table ${sourceTable.tableName}`,
            roles: [role],
        });
        this.policy = policy;
        this.grantPrincipal = new SourceTableAttachedPrincipal(role, policy);
    }
}
/**
 * An `IPrincipal` entity that can be used as the target of `grant` calls, used by the
 * `SourceTableAttachedPolicy` class so it can act as an `IGrantable`.
 */
class SourceTableAttachedPrincipal extends iam.PrincipalBase {
    constructor(role, policy) {
        super();
        this.role = role;
        this.policy = policy;
    }
    get policyFragment() {
        return this.role.policyFragment;
    }
    addToPrincipalPolicy(statement) {
        this.policy.addStatements(statement);
        return {
            policyDependable: this.policy,
            statementAdded: true,
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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