"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseCluster = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const logs = require("../../aws-logs");
const secretsmanager = require("../../aws-secretsmanager");
const core_1 = require("../../core");
const database_secret_1 = require("./database-secret");
const docdb_generated_1 = require("./docdb.generated");
const endpoint_1 = require("./endpoint");
/**
 * A new or imported clustered database.
 */
class DatabaseClusterBase extends core_1.Resource {
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.clusterIdentifier,
            targetType: secretsmanager.AttachmentTargetType.DOCDB_DB_CLUSTER,
        };
    }
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::DocDB::DBCluster
 */
class DatabaseCluster extends DatabaseClusterBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h;
        super(scope, id);
        /**
         * Identifiers of the replicas
         */
        this.instanceIdentifiers = [];
        /**
         * Endpoints which address each individual replica.
         */
        this.instanceEndpoints = [];
        jsiiDeprecationWarnings.monocdk_aws_docdb_DatabaseClusterProps(props);
        this.vpc = props.vpc;
        this.vpcSubnets = props.vpcSubnets;
        // Determine the subnet(s) to deploy the DocDB cluster to
        const { subnetIds, internetConnectivityEstablished } = this.vpc.selectSubnets(this.vpcSubnets);
        // DocDB clusters require a subnet group with subnets from at least two AZs.
        // We cannot test whether the subnets are in different AZs, but at least we can test the amount.
        // See https://docs.aws.amazon.com/documentdb/latest/developerguide/replication.html#replication.high-availability
        if (subnetIds.length < 2) {
            throw new Error(`Cluster requires at least 2 subnets, got ${subnetIds.length}`);
        }
        const subnetGroup = new docdb_generated_1.CfnDBSubnetGroup(this, 'Subnets', {
            dbSubnetGroupDescription: `Subnets for ${id} database`,
            subnetIds,
        });
        // Create the security group for the DB cluster
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: 'DocumentDB security group',
                vpc: this.vpc,
            });
            // HACK: Use an escape-hatch to apply a consistent removal policy to the
            // security group so we don't get errors when trying to delete the stack
            securityGroup.node.defaultChild.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
        }
        this.securityGroupId = securityGroup.securityGroupId;
        // Create the CloudwatchLogsConfiguratoin
        const enableCloudwatchLogsExports = [];
        if (props.exportAuditLogsToCloudWatch) {
            enableCloudwatchLogsExports.push('audit');
        }
        if (props.exportProfilerLogsToCloudWatch) {
            enableCloudwatchLogsExports.push('profiler');
        }
        // Create the secret manager secret if no password is specified
        let secret;
        if (!props.masterUser.password) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUser.username,
                encryptionKey: props.masterUser.kmsKey,
                excludeCharacters: props.masterUser.excludeCharacters,
                secretName: props.masterUser.secretName,
            });
        }
        // Default to encrypted storage
        const storageEncrypted = (_b = props.storageEncrypted) !== null && _b !== void 0 ? _b : true;
        if (props.kmsKey && !storageEncrypted) {
            throw new Error('KMS key supplied but storageEncrypted is false');
        }
        // Create the DocDB cluster
        this.cluster = new docdb_generated_1.CfnDBCluster(this, 'Resource', {
            // Basic
            engineVersion: props.engineVersion,
            dbClusterIdentifier: props.dbClusterName,
            dbSubnetGroupName: subnetGroup.ref,
            port: props.port,
            vpcSecurityGroupIds: [this.securityGroupId],
            dbClusterParameterGroupName: (_c = props.parameterGroup) === null || _c === void 0 ? void 0 : _c.parameterGroupName,
            deletionProtection: props.deletionProtection,
            // Admin
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUser.username,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUser.password.toString(),
            // Backup
            backupRetentionPeriod: (_e = (_d = props.backup) === null || _d === void 0 ? void 0 : _d.retention) === null || _e === void 0 ? void 0 : _e.toDays(),
            preferredBackupWindow: (_f = props.backup) === null || _f === void 0 ? void 0 : _f.preferredWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            // EnableCloudwatchLogsExports
            enableCloudwatchLogsExports: enableCloudwatchLogsExports.length > 0 ? enableCloudwatchLogsExports : undefined,
            // Encryption
            kmsKeyId: (_g = props.kmsKey) === null || _g === void 0 ? void 0 : _g.keyArn,
            storageEncrypted,
        });
        this.cluster.applyRemovalPolicy(props.removalPolicy, {
            applyToUpdateReplacePolicy: true,
        });
        this.clusterIdentifier = this.cluster.ref;
        this.clusterResourceIdentifier = this.cluster.attrClusterResourceId;
        const port = core_1.Token.asNumber(this.cluster.attrPort);
        this.clusterEndpoint = new endpoint_1.Endpoint(this.cluster.attrEndpoint, port);
        this.clusterReadEndpoint = new endpoint_1.Endpoint(this.cluster.attrReadEndpoint, port);
        this.setLogRetention(this, props, enableCloudwatchLogsExports);
        if (secret) {
            this.secret = secret.attach(this);
        }
        // Create the instances
        const instanceCount = (_h = props.instances) !== null && _h !== void 0 ? _h : DatabaseCluster.DEFAULT_NUM_INSTANCES;
        if (instanceCount < 1) {
            throw new Error('At least one instance is required');
        }
        for (let i = 0; i < instanceCount; i++) {
            const instanceIndex = i + 1;
            const instanceIdentifier = props.instanceIdentifierBase != null ? `${props.instanceIdentifierBase}${instanceIndex}`
                : props.dbClusterName != null ? `${props.dbClusterName}instance${instanceIndex}` : undefined;
            const instance = new docdb_generated_1.CfnDBInstance(this, `Instance${instanceIndex}`, {
                // Link to cluster
                dbClusterIdentifier: this.cluster.ref,
                dbInstanceIdentifier: instanceIdentifier,
                // Instance properties
                dbInstanceClass: databaseInstanceType(props.instanceType),
            });
            instance.applyRemovalPolicy(props.removalPolicy, {
                applyToUpdateReplacePolicy: true,
            });
            // We must have a dependency on the NAT gateway provider here to create
            // things in the right order.
            instance.node.addDependency(internetConnectivityEstablished);
            this.instanceIdentifiers.push(instance.ref);
            this.instanceEndpoints.push(new endpoint_1.Endpoint(instance.attrEndpoint, port));
        }
        this.connections = new ec2.Connections({
            defaultPort: ec2.Port.tcp(port),
            securityGroups: [securityGroup],
        });
    }
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_docdb_DatabaseClusterAttributes(attrs);
        class Import extends DatabaseClusterBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: [attrs.securityGroup],
                    defaultPort: this.defaultPort,
                });
                this.clusterIdentifier = attrs.clusterIdentifier;
                this.instanceIdentifiers = attrs.instanceIdentifiers;
                this.clusterEndpoint = new endpoint_1.Endpoint(attrs.clusterEndpointAddress, attrs.port);
                this.clusterReadEndpoint = new endpoint_1.Endpoint(attrs.readerEndpointAddress, attrs.port);
                this.instanceEndpoints = attrs.instanceEndpointAddresses.map(a => new endpoint_1.Endpoint(a, attrs.port));
                this.securityGroupId = attrs.securityGroup.securityGroupId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Sets up CloudWatch log retention if configured.
     */
    setLogRetention(cluster, props, cloudwatchLogsExports) {
        if (props.cloudWatchLogsRetention) {
            for (const log of cloudwatchLogsExports) {
                new logs.LogRetention(cluster, `LogRetention${log}`, {
                    logGroupName: `/aws/docdb/${cluster.clusterIdentifier}/${log}`,
                    retention: props.cloudWatchLogsRetention,
                    role: props.cloudWatchLogsRetentionRole,
                });
            }
        }
    }
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        jsiiDeprecationWarnings.monocdk_Duration(automaticallyAfter);
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for a cluster without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this cluster.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION,
            excludeCharacters: this.node.tryFindChild('Secret')._excludedCharacters,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id, options) {
        jsiiDeprecationWarnings.monocdk_aws_docdb_RotationMultiUserOptions(options);
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for a cluster without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            excludeCharacters: this.node.tryFindChild('Secret')._excludedCharacters,
            application: DatabaseCluster.MULTI_USER_ROTATION_APPLICATION,
            vpc: this.vpc,
            vpcSubnets: this.vpcSubnets,
            target: this,
        });
    }
    /**
     * Adds security groups to this cluster.
     * @param securityGroups The security groups to add.
     */
    addSecurityGroups(...securityGroups) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_ISecurityGroup(securityGroups);
        if (this.cluster.vpcSecurityGroupIds === undefined) {
            this.cluster.vpcSecurityGroupIds = [];
        }
        this.cluster.vpcSecurityGroupIds.push(...securityGroups.map(sg => sg.securityGroupId));
    }
}
exports.DatabaseCluster = DatabaseCluster;
_a = JSII_RTTI_SYMBOL_1;
DatabaseCluster[_a] = { fqn: "monocdk.aws_docdb.DatabaseCluster", version: "1.149.0" };
/**
 * The default number of instances in the DocDB cluster if none are
 * specified
 */
DatabaseCluster.DEFAULT_NUM_INSTANCES = 1;
/**
 * The default port Document DB listens on
 */
DatabaseCluster.DEFAULT_PORT = 27017;
/**
 * The single user secret rotation application.
 */
DatabaseCluster.SINGLE_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER;
/**
 * The multi user secret rotation application.
 */
DatabaseCluster.MULTI_USER_ROTATION_APPLICATION = secretsmanager.SecretRotationApplication.MONGODB_ROTATION_MULTI_USER;
/**
 * Turn a regular instance type into a database instance type
 */
function databaseInstanceType(instanceType) {
    return 'db.' + instanceType.toString();
}
//# sourceMappingURL=data:application/json;base64,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