"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentityPool = exports.IdentityPoolProviderUrl = exports.IdentityPoolProviderType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cognito_1 = require("../../aws-cognito");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const identitypool_role_attachment_1 = require("./identitypool-role-attachment");
/**
 * Types of Identity Pool Login Providers
 */
var IdentityPoolProviderType;
(function (IdentityPoolProviderType) {
    /** Facebook Provider type */
    IdentityPoolProviderType["FACEBOOK"] = "Facebook";
    /** Google Provider Type */
    IdentityPoolProviderType["GOOGLE"] = "Google";
    /** Amazon Provider Type */
    IdentityPoolProviderType["AMAZON"] = "Amazon";
    /** Apple Provider Type */
    IdentityPoolProviderType["APPLE"] = "Apple";
    /** Twitter Provider Type */
    IdentityPoolProviderType["TWITTER"] = "Twitter";
    /** Digits Provider Type */
    IdentityPoolProviderType["DIGITS"] = "Digits";
    /** Open Id Provider Type */
    IdentityPoolProviderType["OPEN_ID"] = "OpenId";
    /** Saml Provider Type */
    IdentityPoolProviderType["SAML"] = "Saml";
    /** User Pool Provider Type */
    IdentityPoolProviderType["USER_POOL"] = "UserPool";
    /** Custom Provider Type */
    IdentityPoolProviderType["CUSTOM"] = "Custom";
})(IdentityPoolProviderType = exports.IdentityPoolProviderType || (exports.IdentityPoolProviderType = {}));
/**
 * Keys for Login Providers - correspond to client id's of respective federation identity providers
 */
class IdentityPoolProviderUrl {
    constructor(
    /** type of Provider Url */
    type, 
    /** value of Provider Url */
    value) {
        this.type = type;
        this.value = value;
        jsiiDeprecationWarnings.monocdk_aws_cognito_identitypool_IdentityPoolProviderType(type);
    }
    /** OpenId Provider Url */
    static openId(url) {
        return new IdentityPoolProviderUrl(IdentityPoolProviderType.OPEN_ID, url);
    }
    /** Saml Provider Url */
    static saml(url) {
        return new IdentityPoolProviderUrl(IdentityPoolProviderType.SAML, url);
    }
    /** User Pool Provider Url */
    static userPool(url) {
        return new IdentityPoolProviderUrl(IdentityPoolProviderType.USER_POOL, url);
    }
    /** Custom Provider Url */
    static custom(url) {
        return new IdentityPoolProviderUrl(IdentityPoolProviderType.CUSTOM, url);
    }
}
exports.IdentityPoolProviderUrl = IdentityPoolProviderUrl;
_a = JSII_RTTI_SYMBOL_1;
IdentityPoolProviderUrl[_a] = { fqn: "monocdk.aws_cognito_identitypool.IdentityPoolProviderUrl", version: "1.149.0" };
/** Facebook Provider Url */
IdentityPoolProviderUrl.FACEBOOK = new IdentityPoolProviderUrl(IdentityPoolProviderType.FACEBOOK, 'graph.facebook.com');
/** Google Provider Url */
IdentityPoolProviderUrl.GOOGLE = new IdentityPoolProviderUrl(IdentityPoolProviderType.GOOGLE, 'accounts.google.com');
/** Amazon Provider Url */
IdentityPoolProviderUrl.AMAZON = new IdentityPoolProviderUrl(IdentityPoolProviderType.AMAZON, 'www.amazon.com');
/** Apple Provider Url */
IdentityPoolProviderUrl.APPLE = new IdentityPoolProviderUrl(IdentityPoolProviderType.APPLE, 'appleid.apple.com');
/** Twitter Provider Url */
IdentityPoolProviderUrl.TWITTER = new IdentityPoolProviderUrl(IdentityPoolProviderType.TWITTER, 'api.twitter.com');
/** Digits Provider Url */
IdentityPoolProviderUrl.DIGITS = new IdentityPoolProviderUrl(IdentityPoolProviderType.DIGITS, 'www.digits.com');
/**
 * Define a Cognito Identity Pool
 *
 *  @resource AWS::Cognito::IdentityPool
 */
class IdentityPool extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.identityPoolName,
        });
        /**
         * List of Identity Providers added in constructor for use with property overrides
         */
        this.cognitoIdentityProviders = [];
        /**
         * Running count of added role attachments
         */
        this.roleAttachmentCount = 0;
        jsiiDeprecationWarnings.monocdk_aws_cognito_identitypool_IdentityPoolProps(props);
        const authProviders = props.authenticationProviders || {};
        const providers = authProviders.userPools ? authProviders.userPools.map(userPool => userPool.bind(this, this)) : undefined;
        if (providers && providers.length)
            this.cognitoIdentityProviders = providers;
        const openIdConnectProviderArns = authProviders.openIdConnectProviders ?
            authProviders.openIdConnectProviders.map(openIdProvider => openIdProvider.openIdConnectProviderArn) : undefined;
        const samlProviderArns = authProviders.samlProviders ?
            authProviders.samlProviders.map(samlProvider => samlProvider.samlProviderArn) : undefined;
        let supportedLoginProviders = {};
        if (authProviders.amazon)
            supportedLoginProviders[IdentityPoolProviderUrl.AMAZON.value] = authProviders.amazon.appId;
        if (authProviders.facebook)
            supportedLoginProviders[IdentityPoolProviderUrl.FACEBOOK.value] = authProviders.facebook.appId;
        if (authProviders.google)
            supportedLoginProviders[IdentityPoolProviderUrl.GOOGLE.value] = authProviders.google.clientId;
        if (authProviders.apple)
            supportedLoginProviders[IdentityPoolProviderUrl.APPLE.value] = authProviders.apple.servicesId;
        if (authProviders.twitter)
            supportedLoginProviders[IdentityPoolProviderUrl.TWITTER.value] = `${authProviders.twitter.consumerKey};${authProviders.twitter.consumerSecret}`;
        if (authProviders.digits)
            supportedLoginProviders[IdentityPoolProviderUrl.DIGITS.value] = `${authProviders.digits.consumerKey};${authProviders.digits.consumerSecret}`;
        if (!Object.keys(supportedLoginProviders).length)
            supportedLoginProviders = undefined;
        const cfnIdentityPool = new aws_cognito_1.CfnIdentityPool(this, 'Resource', {
            allowUnauthenticatedIdentities: props.allowUnauthenticatedIdentities ? true : false,
            allowClassicFlow: props.allowClassicFlow,
            identityPoolName: this.physicalName,
            developerProviderName: authProviders.customProvider,
            openIdConnectProviderArns,
            samlProviderArns,
            supportedLoginProviders,
            cognitoIdentityProviders: core_1.Lazy.any({ produce: () => this.cognitoIdentityProviders }),
        });
        this.identityPoolName = cfnIdentityPool.attrName;
        this.identityPoolId = cfnIdentityPool.ref;
        this.identityPoolArn = core_1.Stack.of(scope).formatArn({
            service: 'cognito-identity',
            resource: 'identitypool',
            resourceName: this.identityPoolId,
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
        });
        this.authenticatedRole = props.authenticatedRole ? props.authenticatedRole : this.configureDefaultRole('Authenticated');
        this.unauthenticatedRole = props.unauthenticatedRole ? props.unauthenticatedRole : this.configureDefaultRole('Unauthenticated');
        const attachment = new identitypool_role_attachment_1.IdentityPoolRoleAttachment(this, 'DefaultRoleAttachment', {
            identityPool: this,
            authenticatedRole: this.authenticatedRole,
            unauthenticatedRole: this.unauthenticatedRole,
            roleMappings: props.roleMappings,
        });
        // This added by the original author, but it's causing cyclic dependencies.
        // Don't know why this was added in the first place, but I'm disabling it for now and if
        // no complaints come from this, we're probably safe to remove it altogether.
        // attachment.node.addDependency(this);
        Array.isArray(attachment);
    }
    /**
     * Import an existing Identity Pool from its id
     */
    static fromIdentityPoolId(scope, id, identityPoolId) {
        const identityPoolArn = core_1.Stack.of(scope).formatArn({
            service: 'cognito-identity',
            resource: 'identitypool',
            resourceName: identityPoolId,
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
        });
        return IdentityPool.fromIdentityPoolArn(scope, id, identityPoolArn);
    }
    /**
     * Import an existing Identity Pool from its Arn
     */
    static fromIdentityPoolArn(scope, id, identityPoolArn) {
        const pool = core_1.Stack.of(scope).splitArn(identityPoolArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        const res = pool.resourceName || '';
        if (!res) {
            throw new Error('Invalid Identity Pool ARN');
        }
        const idParts = res.split(':');
        if (!(idParts.length === 2))
            throw new Error('Invalid Identity Pool Id: Identity Pool Ids must follow the format <region>:<id>');
        if (idParts[0] !== pool.region)
            throw new Error('Invalid Identity Pool Id: Region in Identity Pool Id must match stack region');
        class ImportedIdentityPool extends core_1.Resource {
            constructor() {
                super(scope, id, {
                    account: pool.account,
                    region: pool.region,
                });
                this.identityPoolId = res;
                this.identityPoolArn = identityPoolArn;
                this.identityPoolName = this.physicalName;
            }
        }
        return new ImportedIdentityPool();
    }
    /**
     * Add a User Pool to the IdentityPool and configure User Pool Client to handle identities
     */
    addUserPoolAuthentication(userPool) {
        jsiiDeprecationWarnings.monocdk_aws_cognito_identitypool_IUserPoolAuthenticationProvider(userPool);
        const providers = userPool.bind(this, this);
        this.cognitoIdentityProviders = this.cognitoIdentityProviders.concat(providers);
    }
    /**
     * Adds Role Mappings to Identity Pool
    */
    addRoleMappings(...roleMappings) {
        jsiiDeprecationWarnings.monocdk_aws_cognito_identitypool_IdentityPoolRoleMapping(roleMappings);
        if (!roleMappings || !roleMappings.length)
            return;
        this.roleAttachmentCount++;
        const name = 'RoleMappingAttachment' + this.roleAttachmentCount.toString();
        const attachment = new identitypool_role_attachment_1.IdentityPoolRoleAttachment(this, name, {
            identityPool: this,
            authenticatedRole: this.authenticatedRole,
            unauthenticatedRole: this.unauthenticatedRole,
            roleMappings,
        });
        // This added by the original author, but it's causing cyclic dependencies.
        // Don't know why this was added in the first place, but I'm disabling it for now and if
        // no complaints come from this, we're probably safe to remove it altogether.
        // attachment.node.addDependency(this);
        Array.isArray(attachment);
    }
    /**
     * Configure Default Roles For Identity Pool
     */
    configureDefaultRole(type) {
        const assumedBy = this.configureDefaultGrantPrincipal(type.toLowerCase());
        const role = new aws_iam_1.Role(this, `${type}Role`, {
            description: `Default ${type} Role for Identity Pool ${this.identityPoolName}`,
            assumedBy,
        });
        return role;
    }
    configureDefaultGrantPrincipal(type) {
        return new aws_iam_1.FederatedPrincipal('cognito-identity.amazonaws.com', {
            'StringEquals': {
                'cognito-identity.amazonaws.com:aud': this.identityPoolId,
            },
            'ForAnyValue:StringLike': {
                'cognito-identity.amazonaws.com:amr': type,
            },
        }, 'sts:AssumeRoleWithWebIdentity');
    }
}
exports.IdentityPool = IdentityPool;
_b = JSII_RTTI_SYMBOL_1;
IdentityPool[_b] = { fqn: "monocdk.aws_cognito_identitypool.IdentityPool", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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