import { IOpenIdConnectProvider, ISamlProvider, IRole } from '../../aws-iam';
import { Resource, IResource } from '../../core';
import { Construct } from 'constructs';
import { IdentityPoolRoleMapping } from './identitypool-role-attachment';
import { IUserPoolAuthenticationProvider } from './identitypool-user-pool-authentication-provider';
/**
 * Represents a Cognito IdentityPool
 */
export interface IIdentityPool extends IResource {
    /**
     * The id of the Identity Pool in the format REGION:GUID
     * @attribute
     */
    readonly identityPoolId: string;
    /**
     * The ARN of the Identity Pool
     * @attribute
     */
    readonly identityPoolArn: string;
    /**
     * Name of the Identity Pool
     * @attribute
     */
    readonly identityPoolName: string;
}
/**
 * Props for the IdentityPool construct
 */
export interface IdentityPoolProps {
    /**
     * The name of the Identity Pool
     * @default - automatically generated name by CloudFormation at deploy time
     */
    readonly identityPoolName?: string;
    /**
     * The Default Role to be assumed by Authenticated Users
     * @default - A Default Authenticated Role will be added
     */
    readonly authenticatedRole?: IRole;
    /**
     * The Default Role to be assumed by Unauthenticated Users
     * @default - A Default Unauthenticated Role will be added
     */
    readonly unauthenticatedRole?: IRole;
    /**
     * Wwhether the identity pool supports unauthenticated logins
     * @default - false
     */
    readonly allowUnauthenticatedIdentities?: boolean;
    /**
     * Rules for mapping roles to users
     * @default - no Role Mappings
     */
    readonly roleMappings?: IdentityPoolRoleMapping[];
    /**
     * Enables the Basic (Classic) authentication flow
     * @default - Classic Flow not allowed
     */
    readonly allowClassicFlow?: boolean;
    /**
     * Authentication providers for using in identity pool.
     * @default - No Authentication Providers passed directly to Identity Pool
     */
    readonly authenticationProviders?: IdentityPoolAuthenticationProviders;
}
/**
 * Types of Identity Pool Login Providers
 */
export declare enum IdentityPoolProviderType {
    /** Facebook Provider type */
    FACEBOOK = "Facebook",
    /** Google Provider Type */
    GOOGLE = "Google",
    /** Amazon Provider Type */
    AMAZON = "Amazon",
    /** Apple Provider Type */
    APPLE = "Apple",
    /** Twitter Provider Type */
    TWITTER = "Twitter",
    /** Digits Provider Type */
    DIGITS = "Digits",
    /** Open Id Provider Type */
    OPEN_ID = "OpenId",
    /** Saml Provider Type */
    SAML = "Saml",
    /** User Pool Provider Type */
    USER_POOL = "UserPool",
    /** Custom Provider Type */
    CUSTOM = "Custom"
}
/**
 * Keys for Login Providers - correspond to client id's of respective federation identity providers
 */
export declare class IdentityPoolProviderUrl {
    /** type of Provider Url */
    readonly type: IdentityPoolProviderType;
    /** value of Provider Url */
    readonly value: string;
    /** Facebook Provider Url */
    static readonly FACEBOOK: IdentityPoolProviderUrl;
    /** Google Provider Url */
    static readonly GOOGLE: IdentityPoolProviderUrl;
    /** Amazon Provider Url */
    static readonly AMAZON: IdentityPoolProviderUrl;
    /** Apple Provider Url */
    static readonly APPLE: IdentityPoolProviderUrl;
    /** Twitter Provider Url */
    static readonly TWITTER: IdentityPoolProviderUrl;
    /** Digits Provider Url */
    static readonly DIGITS: IdentityPoolProviderUrl;
    /** OpenId Provider Url */
    static openId(url: string): IdentityPoolProviderUrl;
    /** Saml Provider Url */
    static saml(url: string): IdentityPoolProviderUrl;
    /** User Pool Provider Url */
    static userPool(url: string): IdentityPoolProviderUrl;
    /** Custom Provider Url */
    static custom(url: string): IdentityPoolProviderUrl;
    constructor(
    /** type of Provider Url */
    type: IdentityPoolProviderType, 
    /** value of Provider Url */
    value: string);
}
/**
 * Login Provider for Identity Federation using Amazon Credentials
 */
export interface IdentityPoolAmazonLoginProvider {
    /**
     * App Id for Amazon Identity Federation
     */
    readonly appId: string;
}
/**
 * Login Provider for Identity Federation using Facebook Credentials
 */
export interface IdentityPoolFacebookLoginProvider {
    /**
     * App Id for Facebook Identity Federation
     */
    readonly appId: string;
}
/**
 * Login Provider for Identity Federation using Apple Credentials
*/
export interface IdentityPoolAppleLoginProvider {
    /**
     * App Id for Apple Identity Federation
    */
    readonly servicesId: string;
}
/**
 * Login Provider for Identity Federation using Google Credentials
 */
export interface IdentityPoolGoogleLoginProvider {
    /**
     * App Id for Google Identity Federation
     */
    readonly clientId: string;
}
/**
 * Login Provider for Identity Federation using Twitter Credentials
 */
export interface IdentityPoolTwitterLoginProvider {
    /**
     * App Id for Twitter Identity Federation
     */
    readonly consumerKey: string;
    /**
     * App Secret for Twitter Identity Federation
     */
    readonly consumerSecret: string;
}
/**
 * Login Provider for Identity Federation using Digits Credentials
 */
export interface IdentityPoolDigitsLoginProvider extends IdentityPoolTwitterLoginProvider {
}
/**
 * External Identity Providers To Connect to User Pools and Identity Pools
 */
export interface IdentityPoolProviders {
    /** App Id for Facebook Identity Federation
     * @default - No Facebook Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly facebook?: IdentityPoolFacebookLoginProvider;
    /** Client Id for Google Identity Federation
     * @default - No Google Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly google?: IdentityPoolGoogleLoginProvider;
    /** App Id for Amazon Identity Federation
     * @default -  No Amazon Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly amazon?: IdentityPoolAmazonLoginProvider;
    /** Services Id for Apple Identity Federation
     * @default - No Apple Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly apple?: IdentityPoolAppleLoginProvider;
    /** Consumer Key and Secret for Twitter Identity Federation
     * @default - No Twitter Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly twitter?: IdentityPoolTwitterLoginProvider;
    /** Consumer Key and Secret for Digits Identity Federation
     * @default - No Digits Authentication Provider used without OpenIdConnect or a User Pool
     */
    readonly digits?: IdentityPoolDigitsLoginProvider;
}
/**
* Authentication providers for using in identity pool.
* @see https://docs.aws.amazon.com/cognito/latest/developerguide/external-identity-providers.html
*/
export interface IdentityPoolAuthenticationProviders extends IdentityPoolProviders {
    /**
     * The User Pool Authentication Providers associated with this Identity Pool
     * @default - no User Pools Associated
     */
    readonly userPools?: IUserPoolAuthenticationProvider[];
    /**
     * The OpenIdConnect Provider associated with this Identity Pool
     * @default - no OpenIdConnectProvider
    */
    readonly openIdConnectProviders?: IOpenIdConnectProvider[];
    /**
     * The Security Assertion Markup Language Provider associated with this Identity Pool
     * @default - no SamlProvider
    */
    readonly samlProviders?: ISamlProvider[];
    /**
     * The Developer Provider Name to associate with this Identity Pool
     * @default - no Custom Provider
    */
    readonly customProvider?: string;
}
/**
 * Define a Cognito Identity Pool
 *
 *  @resource AWS::Cognito::IdentityPool
 */
export declare class IdentityPool extends Resource implements IIdentityPool {
    /**
     * Import an existing Identity Pool from its id
     */
    static fromIdentityPoolId(scope: Construct, id: string, identityPoolId: string): IIdentityPool;
    /**
     * Import an existing Identity Pool from its Arn
     */
    static fromIdentityPoolArn(scope: Construct, id: string, identityPoolArn: string): IIdentityPool;
    /**
     * The id of the Identity Pool in the format REGION:GUID
     * @attribute
     */
    readonly identityPoolId: string;
    /**
     * The ARN of the Identity Pool
     * @attribute
     */
    readonly identityPoolArn: string;
    /**
     * The name of the Identity Pool
     * @attribute
     */
    readonly identityPoolName: string;
    /**
     * Default role for authenticated users
     */
    readonly authenticatedRole: IRole;
    /**
      * Default role for unauthenticated users
      */
    readonly unauthenticatedRole: IRole;
    /**
     * List of Identity Providers added in constructor for use with property overrides
     */
    private cognitoIdentityProviders;
    /**
     * Running count of added role attachments
     */
    private roleAttachmentCount;
    constructor(scope: Construct, id: string, props?: IdentityPoolProps);
    /**
     * Add a User Pool to the IdentityPool and configure User Pool Client to handle identities
     */
    addUserPoolAuthentication(userPool: IUserPoolAuthenticationProvider): void;
    /**
     * Adds Role Mappings to Identity Pool
    */
    addRoleMappings(...roleMappings: IdentityPoolRoleMapping[]): void;
    /**
     * Configure Default Roles For Identity Pool
     */
    private configureDefaultRole;
    private configureDefaultGrantPrincipal;
}
