"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("../../../aws-events");
const cdk = require("../../../core");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        //
        // It may be that a construct already exists with the given action name (CDK Pipelines
        // may do this to maintain construct tree compatibility between versions).
        //
        // If so, we simply reuse it.
        let actionScope = constructs_1.Node.of(this.scope).tryFindChild(action.actionProperties.actionName);
        if (!actionScope) {
            let id = action.actionProperties.actionName;
            if (core_1.Token.isUnresolved(id)) {
                id = findUniqueConstructId(this.scope, action.actionProperties.provider);
            }
            actionScope = new cdk.Construct(this.scope, id);
        }
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.any({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.any({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
function findUniqueConstructId(scope, prefix) {
    let current = prefix;
    let ctr = 1;
    while (constructs_1.Node.of(scope).tryFindChild(current) !== undefined) {
        current = `${prefix}${++ctr}`;
    }
    return current;
}
//# sourceMappingURL=data:application/json;base64,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