"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Pipeline = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const notifications = require("../../aws-codestarnotifications");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const s3 = require("../../aws-s3");
const core_1 = require("../../core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./private/cross-region-support-stack");
const full_action_descriptor_1 = require("./private/full-action-descriptor");
const rich_action_1 = require("./private/rich-action");
const stage_1 = require("./private/stage");
const validation_1 = require("./private/validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.pipelineArn,
        };
    }
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    notifyOnExecutionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUCCEEDED,
                action_1.PipelineNotificationEvents.PIPELINE_EXECUTION_SUPERSEDED,
            ],
        });
    }
    notifyOnAnyStageStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_RESUMED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.STAGE_EXECUTION_SUCCEEDED,
            ],
        });
    }
    notifyOnAnyActionStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_CANCELED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_FAILED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_STARTED,
                action_1.PipelineNotificationEvents.ACTION_EXECUTION_SUCCEEDED,
            ],
        });
    }
    notifyOnAnyManualApprovalStateChange(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_FAILED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_NEEDED,
                action_1.PipelineNotificationEvents.MANUAL_APPROVAL_SUCCEEDED,
            ],
        });
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * import * as codecommit from '@aws-cdk/aws-codecommit';
 *
 * const pipeline = new codepipeline.Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ stageName: 'Source' });
 *
 * // add a source action to the stage
 * declare const repo: codecommit.Repository;
 * declare const sourceArtifact: codepipeline.Artifact;
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   output: sourceArtifact,
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_PipelineProps(props);
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // @deprecated(v2): switch to default false
        this.crossAccountKeys = (_b = props.crossAccountKeys) !== null && _b !== void 0 ? _b : true;
        this.enableKeyRotation = props.enableKeyRotation;
        // Cross account keys must be set for key rotation to be enabled
        if (this.enableKeyRotation && !this.crossAccountKeys) {
            throw new Error("Setting 'enableKeyRotation' to true also requires 'crossAccountKeys' to be enabled");
        }
        this.reuseCrossRegionSupportStacks = (_c = props.reuseCrossRegionSupportStacks) !== null && _c !== void 0 ? _c : true;
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            let encryptionKey;
            if (this.crossAccountKeys) {
                encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                    // remove the key - there is a grace period of a few days before it's gone for good,
                    // that should be enough for any emergency access to the bucket artifacts
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                    enableKeyRotation: this.enableKeyRotation,
                });
                // add an alias to make finding the key in the console easier
                new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                    aliasName: this.generateNameForDefaultBucketKeyAlias(),
                    targetKey: encryptionKey,
                    removalPolicy: core_1.RemovalPolicy.DESTROY,
                });
            }
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: encryptionKey ? s3.BucketEncryption.KMS : s3.BucketEncryption.KMS_MANAGED,
                enforceSSL: true,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.any({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.any({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.any({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).splitArn(pipelineArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_StageOptions(props);
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Access one of the pipeline's stages by stage name
     */
    stage(stageName) {
        for (const stage of this._stages) {
            if (stage.stageName === stageName) {
                return stage;
            }
        }
        throw new Error(`Pipeline does not contain a stage named '${stageName}'. Available stages: ${this._stages.map(s => s.stageName).join(', ')}`);
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        const richAction = new rich_action_1.RichAction(action, this);
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(richAction);
        // get the role for the given action, handling if it's cross-account
        const actionRole = this.getRoleForAction(stage, richAction, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(richAction.actionProperties.variablesNamespace);
        // bind the Action (type h4x)
        const actionConfig = richAction.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            // must be 'action', not 'richAction',
            // as those are returned by the IStage.actions property,
            // and it's important customers of Pipeline get the same instance
            // back as they added to the pipeline
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        if (!action.isCrossRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // The action has a specific region,
        // require the pipeline to have a known region as well.
        this.requireRegion();
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        const crossRegionSupport = this.obtainCrossRegionSupportFor(action);
        // the stack containing the replication bucket must be deployed before the pipeline
        core_1.Stack.of(this).addDependency(crossRegionSupport.stack);
        // The Pipeline role must be able to replicate to that bucket
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: action.effectiveRegion,
        };
    }
    /**
     * Get or create the cross-region support construct for the given action
     */
    obtainCrossRegionSupportFor(action) {
        // this method is never called for non cross-region actions
        const actionRegion = action.effectiveRegion;
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            const otherStack = action.resourceStack;
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        return crossRegionSupport;
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id, {
                    createKmsKey: this.crossAccountKeys,
                    enableKeyRotation: this.enableKeyRotation,
                });
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.supportScope();
        const supportStackId = `cross-region-stack-${this.reuseCrossRegionSupportStacks ? pipelineAccount : pipelineStack.stackName}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
                createKmsKey: this.crossAccountKeys,
                enableKeyRotation: this.enableKeyRotation,
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = core_1.Names.uniqueId(this);
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        // (otherwise we can't configure cross-account trust policies)
        if (action.isCrossAccount) {
            const artifactBucket = this.ensureReplicationResourcesExistFor(action).artifactBucket;
            if (!artifactBucket.encryptionKey) {
                throw new Error(`Artifact Bucket must have a KMS Key to add cross-account action '${action.actionProperties.actionName}' ` +
                    `(pipeline account: '${renderEnvDimension(this.env.account)}', action account: '${renderEnvDimension(action.effectiveAccount)}'). ` +
                    'Create Pipeline with \'crossAccountKeys: true\' (or pass an existing Bucket with a key)');
            }
        }
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${core_1.Names.uniqueId(this)}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const targetAccount = action.actionProperties.resource
            ? action.actionProperties.resource.env.account
            : action.actionProperties.account;
        if (targetAccount === undefined) {
            // if the account of the Action is not specified,
            // then it defaults to the same account the pipeline itself is in
            return undefined;
        }
        // check whether the action's account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            if (core_1.Token.isUnresolved(this.env.account)) {
                // the pipeline is also env-agnostic, so that's fine
                return undefined;
            }
            else {
                throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
            }
        }
        // At this point, we know that the action's account is a static string.
        // In this case, the pipeline's account must also be a static string.
        if (core_1.Token.isUnresolved(this.env.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        // at this point, we know that both the Pipeline's account,
        // and the action-backing resource's account are static strings
        // if they are identical - nothing to do (the action is not cross-account)
        if (this.env.account === targetAccount) {
            return undefined;
        }
        // at this point, we know that the action is certainly cross-account,
        // so we need to return a Stack in its account to create the helper Role in
        const candidateActionResourceStack = action.actionProperties.resource
            ? core_1.Stack.of(action.actionProperties.resource)
            : undefined;
        if ((candidateActionResourceStack === null || candidateActionResourceStack === void 0 ? void 0 : candidateActionResourceStack.account) === targetAccount) {
            // we always use the "latest" action-backing resource's Stack for this account,
            // even if a different one was used earlier
            this._crossAccountSupport[targetAccount] = candidateActionResourceStack;
            return candidateActionResourceStack;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.supportScope();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                const actionRegion = action.actionProperties.resource
                    ? action.actionProperties.resource.env.region
                    : action.actionProperties.region;
                const pipelineStack = core_1.Stack.of(this);
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: actionRegion !== null && actionRegion !== void 0 ? actionRegion : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = this.env.region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    supportScope() {
        const scope = core_1.Stage.of(this);
        if (!scope) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK App or Stage');
        }
        return scope;
    }
}
exports.Pipeline = Pipeline;
_a = JSII_RTTI_SYMBOL_1;
Pipeline[_a] = { fqn: "monocdk.aws_codepipeline.Pipeline", version: "1.149.0" };
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
/**
 * Render an env dimension without showing the ugly stringified tokens
 */
function renderEnvDimension(s) {
    return core_1.Token.isUnresolved(s) ? '(current)' : s;
}
//# sourceMappingURL=data:application/json;base64,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