"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const codepipeline = require("../../../aws-codepipeline");
const targets = require("../../../aws-events-targets");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    CodeCommitTrigger["NONE"] = "None";
    /**
     * CodePipeline will poll the repository to detect changes.
     */
    CodeCommitTrigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * This is the default method of detecting changes.
     */
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 *
 * If the CodeCommit repository is in a different account, you must use
 * `CodeCommitTrigger.EVENTS` to trigger the pipeline.
 *
 * (That is because the Pipeline structure normally only has a `RepositoryName`
 * field, and that is not enough for the pipeline to locate the repository's
 * source account. However, if the pipeline is triggered via an EventBridge
 * event, the event itself has the full repository ARN in there, allowing the
 * pipeline to locate the repository).
 */
class CodeCommitSourceAction extends action_1.Action {
    constructor(props) {
        var _b;
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_actions_CodeCommitSourceActionProps(props);
        const branch = (_b = props.branch) !== null && _b !== void 0 ? _b : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        if (props.codeBuildCloneOutput === true) {
            props.output.setMetadata(CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY, props.repository.repositoryArn);
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    bound(_scope, stage, options) {
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_IStage(stage);
        jsiiDeprecationWarnings.monocdk_aws_codepipeline_ActionBindOptions(options);
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline, {
                    eventRole: this.props.eventRole,
                }),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // when this action is cross-account,
        // the Role needs the s3:PutObjectAcl permission for some not yet fully understood reason
        if (core_1.Token.compareStrings(this.props.repository.env.account, core_1.Stack.of(stage.pipeline).account) === core_1.TokenComparison.DIFFERENT) {
            options.bucket.grantPutAcl(options.role);
        }
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
                ...(this.props.codeBuildCloneOutput === true ? ['codecommit:GetRepository'] : []),
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
                OutputArtifactFormat: this.props.codeBuildCloneOutput === true
                    ? 'CODEBUILD_CLONE_REF'
                    : undefined,
            },
        };
    }
    generateEventId(stage) {
        const baseId = core_1.Names.nodeUniqueId(stage.pipeline.node);
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : `-${this.branch}-`;
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
_a = JSII_RTTI_SYMBOL_1;
CodeCommitSourceAction[_a] = { fqn: "monocdk.aws_codepipeline_actions.CodeCommitSourceAction", version: "1.149.0" };
/**
 * The name of the property that holds the ARN of the CodeCommit Repository
 * inside of the CodePipeline Artifact's metadata.
 *
 * @internal
 */
CodeCommitSourceAction._FULL_CLONE_ARN_PROPERTY = 'CodeCommitCloneRepositoryArn';
//# sourceMappingURL=data:application/json;base64,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