"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 */
class Source {
    constructor(props) {
        this.badgeSupported = false;
        jsiiDeprecationWarnings.monocdk_aws_codebuild_SourceProps(props);
        this.identifier = props.identifier;
    }
    static s3(props) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_S3SourceProps(props);
        return new S3Source(props);
    }
    static codeCommit(props) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_CodeCommitSourceProps(props);
        return new CodeCommitSource(props);
    }
    static gitHub(props) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_GitHubSourceProps(props);
        return new GitHubSource(props);
    }
    static gitHubEnterprise(props) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_GitHubEnterpriseSourceProps(props);
        return new GitHubEnterpriseSource(props);
    }
    static bitBucket(props) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_BitBucketSourceProps(props);
        return new BitBucketSource(props);
    }
    /**
     * Called by the project when the source is added so that the source can perform
     * binding operations on the source. For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     */
    bind(_scope, _project) {
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_IProject(_project);
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
_a = JSII_RTTI_SYMBOL_1;
Source[_a] = { fqn: "monocdk.aws_codebuild.Source", version: "1.149.0" };
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
        this.fetchSubmodules = props.fetchSubmodules;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
                gitSubmodulesConfig: this.fetchSubmodules ? {
                    fetchSubmodules: this.fetchSubmodules,
                } : undefined,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 */
var EventAction;
(function (EventAction) {
    /**
     * A push (of a branch, or a tag) to the repository.
     */
    EventAction["PUSH"] = "PUSH";
    /**
     * Creating a Pull Request.
     */
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    /**
     * Updating a Pull Request.
     */
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    /**
     * Merging a Pull Request.
     */
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    /**
     * Re-opening a previously closed Pull Request.
     * Note that this event is only supported for GitHub and GitHubEnterprise sources.
     */
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
var WebhookFilterTypes;
(function (WebhookFilterTypes) {
    WebhookFilterTypes["FILE_PATH"] = "FILE_PATH";
    WebhookFilterTypes["COMMIT_MESSAGE"] = "COMMIT_MESSAGE";
    WebhookFilterTypes["HEAD_REF"] = "HEAD_REF";
    WebhookFilterTypes["ACTOR_ACCOUNT_ID"] = "ACTOR_ACCOUNT_ID";
    WebhookFilterTypes["BASE_REF"] = "BASE_REF";
})(WebhookFilterTypes || (WebhookFilterTypes = {}));
/**
 * An object that represents a group of filter conditions for a webhook.
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on
     */
    static inEventOf(...actions) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_EventAction(actions);
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression)
     */
    andCommitMessageIs(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a head commit with the given message.
     *
     * @param commitMessage the commit message (can be a regular expression)
     */
    andCommitMessageIsNot(commitMessage) {
        return this.addCommitMessageFilter(commitMessage, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub, Bitbucket and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must not affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub, Bitbucket and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addCommitMessageFilter(commitMessage, include) {
        return this.addFilter(WebhookFilterTypes.COMMIT_MESSAGE, commitMessage, include);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter(WebhookFilterTypes.HEAD_REF, refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter(WebhookFilterTypes.ACTOR_ACCOUNT_ID, accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter(WebhookFilterTypes.BASE_REF, refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(WebhookFilterTypes.FILE_PATH, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
_b = JSII_RTTI_SYMBOL_1;
FilterGroup[_b] = { fqn: "monocdk.aws_codebuild.FilterGroup", version: "1.149.0" };
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        var _c;
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = (_c = props.reportBuildStatus) !== null && _c !== void 0 ? _c : true;
        this.webhookFilters = props.webhookFilters || [];
        this.webhookTriggersBatchBuild = props.webhookTriggersBatchBuild;
    }
    bind(_scope, project) {
        var _c;
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = (_c = this.webhook) !== null && _c !== void 0 ? _c : (anyFilterGroupsProvided ? true : undefined);
        if (!webhook && anyFilterGroupsProvided) {
            throw new Error('`webhookFilters` cannot be used when `webhook` is `false`');
        }
        if (!webhook && this.webhookTriggersBatchBuild) {
            throw new Error('`webhookTriggersBatchBuild` cannot be used when `webhook` is `false`');
        }
        const superConfig = super.bind(_scope, project);
        if (this.webhookTriggersBatchBuild) {
            project.enableBatchBuilds();
        }
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                buildType: this.webhookTriggersBatchBuild ? 'BUILD_BATCH' : undefined,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project, this.path);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        if (this.hasCommitMessageFilterAndPrEvent()) {
            throw new Error('COMMIT_MESSAGE filters cannot be used with GitHub Enterprise Server pull request events');
        }
        if (this.hasFilePathFilterAndPrEvent()) {
            throw new Error('FILE_PATH filters cannot be used with GitHub Enterprise Server pull request events');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    hasCommitMessageFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.COMMIT_MESSAGE) &&
            this.hasPrEvent(fg._actions)));
    }
    hasFilePathFilterAndPrEvent() {
        return this.webhookFilters.some(fg => (fg._filters.some(fp => fp.type === WebhookFilterTypes.FILE_PATH) &&
            this.hasPrEvent(fg._actions)));
    }
    hasPrEvent(actions) {
        return actions.includes(EventAction.PULL_REQUEST_CREATED ||
            EventAction.PULL_REQUEST_MERGED ||
            EventAction.PULL_REQUEST_REOPENED ||
            EventAction.PULL_REQUEST_UPDATED);
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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