"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProjectNotificationEvents = exports.BuildEnvironmentVariableType = exports.WindowsBuildImage = exports.WindowsImageType = exports.LinuxBuildImage = exports.ImagePullPrincipalType = exports.ComputeType = exports.Project = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("../../aws-cloudwatch");
const notifications = require("../../aws-codestarnotifications");
const ec2 = require("../../aws-ec2");
const aws_ecr_assets_1 = require("../../aws-ecr-assets");
const events = require("../../aws-events");
const iam = require("../../aws-iam");
const kms = require("../../aws-kms");
const core_1 = require("../../core");
const build_spec_1 = require("./build-spec");
const cache_1 = require("./cache");
const codebuild_canned_metrics_generated_1 = require("./codebuild-canned-metrics.generated");
const codebuild_generated_1 = require("./codebuild.generated");
const codepipeline_artifacts_1 = require("./codepipeline-artifacts");
const no_artifacts_1 = require("./no-artifacts");
const no_source_1 = require("./no-source");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const report_group_utils_1 = require("./report-group-utils");
const source_types_1 = require("./source-types");
const VPC_POLICY_SYM = Symbol.for('@aws-cdk/aws-codebuild.roleVpcPolicy');
/**
 * Represents a reference to a CodeBuild Project.
 *
 * If you're managing the Project alongside the rest of your CDK resources,
 * use the {@link Project} class.
 *
 * If you want to reference an already existing Project
 * (or one defined in a different CDK Stack),
 * use the {@link import} method.
 */
class ProjectBase extends core_1.Resource {
    /**
     * Access the Connections object.
     * Will fail if this Project does not have a VPC set.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('Only VPC-associated Projects have security groups to manage. Supply the "vpc" parameter when creating the Project');
        }
        return this._connections;
    }
    enableBatchBuilds() {
        return undefined;
    }
    /**
     * Add a permission only if there's a policy attached.
     * @param statement The permissions statement to add
     */
    addToRolePolicy(statement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(statement);
        }
    }
    /**
     * Defines a CloudWatch event rule triggered when something happens with this project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codebuild'],
            detail: {
                'project-name': [this.projectName],
            },
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule triggered when the build project state
     * changes. You can filter specific build status events using an event
     * pattern filter on the `build-status` detail field:
     *
     *    const rule = project.onStateChange('OnBuildStarted', { target });
     *    rule.addEventPattern({
     *      detail: {
     *        'build-status': [
     *          "IN_PROGRESS",
     *          "SUCCEEDED",
     *          "FAILED",
     *          "STOPPED"
     *        ]
     *      }
     *    });
     *
     * You can also use the methods `onBuildFailed` and `onBuildSucceeded` to define rules for
     * these specific state changes.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build State Change'],
        });
        return rule;
    }
    /**
     * Defines a CloudWatch event rule that triggers upon phase change of this
     * build project.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-build-notifications.html
     */
    onPhaseChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodeBuild Build Phase Change'],
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build starts.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildStarted(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['IN_PROGRESS'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build fails.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildFailed(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['FAILED'],
            },
        });
        return rule;
    }
    /**
     * Defines an event rule which triggers when a build completes successfully.
     *
     * To access fields from the event in the event target input,
     * use the static fields on the `StateChangeEvent` class.
     */
    onBuildSucceeded(id, options = {}) {
        const rule = this.onStateChange(id, options);
        rule.addEventPattern({
            detail: {
                'build-status': ['SUCCEEDED'],
            },
        });
        return rule;
    }
    /**
     * @returns a CloudWatch metric associated with this build project.
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/CodeBuild',
            metricName,
            dimensionsMap: { ProjectName: this.projectName },
            ...props,
        }).attachTo(this);
    }
    /**
     * Measures the number of builds triggered.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.buildsSum, props);
    }
    /**
     * Measures the duration of all builds over time.
     *
     * Units: Seconds
     *
     * Valid CloudWatch statistics: Average (recommended), Maximum, Minimum
     *
     * @default average over 5 minutes
     */
    metricDuration(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.durationAverage, props);
    }
    /**
     * Measures the number of successful builds.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricSucceededBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.succeededBuildsSum, props);
    }
    /**
     * Measures the number of builds that failed because of client error or
     * because of a timeout.
     *
     * Units: Count
     *
     * Valid CloudWatch statistics: Sum
     *
     * @default sum over 5 minutes
     */
    metricFailedBuilds(props) {
        return this.cannedMetric(codebuild_canned_metrics_generated_1.CodeBuildMetrics.failedBuildsSum, props);
    }
    notifyOn(id, target, options) {
        return new notifications.NotificationRule(this, id, {
            ...options,
            source: this,
            targets: [target],
        });
    }
    notifyOnBuildSucceeded(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [ProjectNotificationEvents.BUILD_SUCCEEDED],
        });
    }
    notifyOnBuildFailed(id, target, options) {
        return this.notifyOn(id, target, {
            ...options,
            events: [ProjectNotificationEvents.BUILD_FAILED],
        });
    }
    bindAsNotificationRuleSource(_scope) {
        return {
            sourceArn: this.projectArn,
        };
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ ProjectName: this.projectName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * A representation of a CodeBuild Project.
 */
class Project extends ProjectBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.projectName,
        });
        jsiiDeprecationWarnings.monocdk_aws_codebuild_ProjectProps(props);
        this.role = props.role || new iam.Role(this, 'Role', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        this.buildImage = (props.environment && props.environment.buildImage) || LinuxBuildImage.STANDARD_1_0;
        // let source "bind" to the project. this usually involves granting permissions
        // for the code build role to interact with the source.
        this.source = props.source || new no_source_1.NoSource();
        const sourceConfig = this.source.bind(this, this);
        if (props.badge && !this.source.badgeSupported) {
            throw new Error(`Badge is not supported for source type ${this.source.type}`);
        }
        const artifacts = props.artifacts
            ? props.artifacts
            : (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE
                ? new codepipeline_artifacts_1.CodePipelineArtifacts()
                : new no_artifacts_1.NoArtifacts());
        const artifactsConfig = artifacts.bind(this, this);
        const cache = props.cache || cache_1.Cache.none();
        // give the caching strategy the option to grant permissions to any required resources
        cache._bind(this);
        // Inject download commands for asset if requested
        const environmentVariables = props.environmentVariables || {};
        const buildSpec = props.buildSpec;
        if (this.source.type === source_types_1.NO_SOURCE_TYPE && (buildSpec === undefined || !buildSpec.isImmediate)) {
            throw new Error("If the Project's source is NoSource, you need to provide a concrete buildSpec");
        }
        this._secondarySources = [];
        this._secondarySourceVersions = [];
        this._fileSystemLocations = [];
        for (const secondarySource of props.secondarySources || []) {
            this.addSecondarySource(secondarySource);
        }
        this._secondaryArtifacts = [];
        for (const secondaryArtifact of props.secondaryArtifacts || []) {
            this.addSecondaryArtifact(secondaryArtifact);
        }
        this.validateCodePipelineSettings(artifacts);
        for (const fileSystemLocation of props.fileSystemLocations || []) {
            this.addFileSystemLocation(fileSystemLocation);
        }
        const resource = new codebuild_generated_1.CfnProject(this, 'Resource', {
            description: props.description,
            source: {
                ...sourceConfig.sourceProperty,
                buildSpec: buildSpec && buildSpec.toBuildSpec(),
            },
            artifacts: artifactsConfig.artifactsProperty,
            serviceRole: this.role.roleArn,
            environment: this.renderEnvironment(props, environmentVariables),
            fileSystemLocations: core_1.Lazy.any({ produce: () => this.renderFileSystemLocations() }),
            // lazy, because we have a setter for it in setEncryptionKey
            // The 'alias/aws/s3' default is necessary because leaving the `encryptionKey` field
            // empty will not remove existing encryptionKeys during an update (ref. t/D17810523)
            encryptionKey: core_1.Lazy.string({ produce: () => this._encryptionKey ? this._encryptionKey.keyArn : 'alias/aws/s3' }),
            badgeEnabled: props.badge,
            cache: cache._toCloudFormation(),
            name: this.physicalName,
            timeoutInMinutes: props.timeout && props.timeout.toMinutes(),
            queuedTimeoutInMinutes: props.queuedTimeout && props.queuedTimeout.toMinutes(),
            concurrentBuildLimit: props.concurrentBuildLimit,
            secondarySources: core_1.Lazy.any({ produce: () => this.renderSecondarySources() }),
            secondarySourceVersions: core_1.Lazy.any({ produce: () => this.renderSecondarySourceVersions() }),
            secondaryArtifacts: core_1.Lazy.any({ produce: () => this.renderSecondaryArtifacts() }),
            triggers: sourceConfig.buildTriggers,
            sourceVersion: sourceConfig.sourceVersion,
            vpcConfig: this.configureVpc(props),
            logsConfig: this.renderLoggingConfiguration(props.logging),
            buildBatchConfig: core_1.Lazy.any({
                produce: () => {
                    const config = this._batchServiceRole ? {
                        serviceRole: this._batchServiceRole.roleArn,
                    } : undefined;
                    return config;
                },
            }),
        });
        this.addVpcRequiredPermissions(props, resource);
        this.projectArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'codebuild',
            resource: 'project',
            resourceName: this.physicalName,
        });
        this.projectName = this.getResourceNameAttribute(resource.ref);
        this.addToRolePolicy(this.createLoggingPermission());
        // add permissions to create and use test report groups
        // with names starting with the project's name,
        // unless the customer explicitly opts out of it
        if (props.grantReportGroupPermissions !== false) {
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: [
                    'codebuild:CreateReportGroup',
                    'codebuild:CreateReport',
                    'codebuild:UpdateReport',
                    'codebuild:BatchPutTestCases',
                    'codebuild:BatchPutCodeCoverages',
                ],
                resources: [report_group_utils_1.renderReportGroupArn(this, `${this.projectName}-*`)],
            }));
        }
        if (props.encryptionKey) {
            this.encryptionKey = props.encryptionKey;
        }
        // bind
        if (isBindableBuildImage(this.buildImage)) {
            this.buildImage.bind(this, this, {});
        }
    }
    static fromProjectArn(scope, id, projectArn) {
        const parsedArn = core_1.Stack.of(scope).splitArn(projectArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i, {
                    account: parsedArn.account,
                    region: parsedArn.region,
                });
                this.projectArn = projectArn;
                this.projectName = parsedArn.resourceName;
                this.role = undefined;
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Import a Project defined either outside the CDK,
     * or in a different CDK Stack
     * (and exported using the {@link export} method).
     *
     * @note if you're importing a CodeBuild Project for use
     *   in a CodePipeline, make sure the existing Project
     *   has permissions to access the S3 Bucket of that Pipeline -
     *   otherwise, builds in that Pipeline will always fail.
     *
     * @param scope the parent Construct for this Construct
     * @param id the logical name of this Construct
     * @param projectName the name of the project to import
     * @returns a reference to the existing Project
     */
    static fromProjectName(scope, id, projectName) {
        class Import extends ProjectBase {
            constructor(s, i) {
                super(s, i);
                this.role = undefined;
                this.projectArn = core_1.Stack.of(this).formatArn({
                    service: 'codebuild',
                    resource: 'project',
                    resourceName: projectName,
                });
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
                this.projectName = projectName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Convert the environment variables map of string to {@link BuildEnvironmentVariable},
     * which is the customer-facing type, to a list of {@link CfnProject.EnvironmentVariableProperty},
     * which is the representation of environment variables in CloudFormation.
     *
     * @param environmentVariables the map of string to environment variables
     * @param validateNoPlainTextSecrets whether to throw an exception
     *   if any of the plain text environment variables contain secrets, defaults to 'false'
     * @returns an array of {@link CfnProject.EnvironmentVariableProperty} instances
     */
    static serializeEnvVariables(environmentVariables, validateNoPlainTextSecrets = false, principal) {
        var _d;
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(principal);
        const ret = new Array();
        const ssmIamResources = new Array();
        const secretsManagerIamResources = new Set();
        const kmsIamResources = new Set();
        for (const [name, envVariable] of Object.entries(environmentVariables)) {
            const envVariableValue = (_d = envVariable.value) === null || _d === void 0 ? void 0 : _d.toString();
            const cfnEnvVariable = {
                name,
                type: envVariable.type || BuildEnvironmentVariableType.PLAINTEXT,
                value: envVariableValue,
            };
            ret.push(cfnEnvVariable);
            // validate that the plain-text environment variables don't contain any secrets in them
            if (validateNoPlainTextSecrets && cfnEnvVariable.type === BuildEnvironmentVariableType.PLAINTEXT) {
                const fragments = core_1.Tokenization.reverseString(cfnEnvVariable.value);
                for (const token of fragments.tokens) {
                    if (token instanceof core_1.SecretValue) {
                        throw new Error(`Plaintext environment variable '${name}' contains a secret value! ` +
                            'This means the value of this variable will be visible in plain text in the AWS Console. ' +
                            "Please consider using CodeBuild's SecretsManager environment variables feature instead. " +
                            "If you'd like to continue with having this secret in the plaintext environment variables, " +
                            'please set the checkSecretsInPlainTextEnvVariables property to false');
                    }
                }
            }
            if (principal) {
                const stack = core_1.Stack.of(principal);
                // save the SSM env variables
                if (envVariable.type === BuildEnvironmentVariableType.PARAMETER_STORE) {
                    ssmIamResources.push(stack.formatArn({
                        service: 'ssm',
                        resource: 'parameter',
                        // If the parameter name starts with / the resource name is not separated with a double '/'
                        // arn:aws:ssm:region:1111111111:parameter/PARAM_NAME
                        resourceName: envVariableValue.startsWith('/')
                            ? envVariableValue.substr(1)
                            : envVariableValue,
                    }));
                }
                // save SecretsManager env variables
                if (envVariable.type === BuildEnvironmentVariableType.SECRETS_MANAGER) {
                    // We have 3 basic cases here of what envVariableValue can be:
                    // 1. A string that starts with 'arn:' (and might contain Token fragments).
                    // 2. A Token.
                    // 3. A simple value, like 'secret-id'.
                    if (envVariableValue.startsWith('arn:')) {
                        const parsedArn = stack.splitArn(envVariableValue, core_1.ArnFormat.COLON_RESOURCE_NAME);
                        if (!parsedArn.resourceName) {
                            throw new Error('SecretManager ARN is missing the name of the secret: ' + envVariableValue);
                        }
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = parsedArn.resourceName.split(':')[0];
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            // since we don't know whether the ARN was full, or partial
                            // (CodeBuild supports both),
                            // stick a "*" at the end, which makes it work for both
                            resourceName: `${secretName}*`,
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                            partition: parsedArn.partition,
                            account: parsedArn.account,
                            region: parsedArn.region,
                        }));
                        // if secret comes from another account, SecretsManager will need to access
                        // KMS on the other account as well to be able to get the secret
                        if (parsedArn.account && core_1.Token.compareStrings(parsedArn.account, stack.account) === core_1.TokenComparison.DIFFERENT) {
                            kmsIamResources.add(stack.formatArn({
                                service: 'kms',
                                resource: 'key',
                                // We do not know the ID of the key, but since this is a cross-account access,
                                // the key policies have to allow this access, so a wildcard is safe here
                                resourceName: '*',
                                arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                                partition: parsedArn.partition,
                                account: parsedArn.account,
                                region: parsedArn.region,
                            }));
                        }
                    }
                    else if (core_1.Token.isUnresolved(envVariableValue)) {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        let secretArn = envVariableValue.split(':')[0];
                        // parse the Token, and see if it represents a single resource
                        // (we will assume it's a Secret from SecretsManager)
                        const fragments = core_1.Tokenization.reverseString(envVariableValue);
                        if (fragments.tokens.length === 1) {
                            const resolvable = fragments.tokens[0];
                            if (core_1.Reference.isReference(resolvable)) {
                                // check the Stack the resource owning the reference belongs to
                                const resourceStack = core_1.Stack.of(resolvable.target);
                                if (core_1.Token.compareStrings(stack.account, resourceStack.account) === core_1.TokenComparison.DIFFERENT) {
                                    // since this is a cross-account access,
                                    // add the appropriate KMS permissions
                                    kmsIamResources.add(stack.formatArn({
                                        service: 'kms',
                                        resource: 'key',
                                        // We do not know the ID of the key, but since this is a cross-account access,
                                        // the key policies have to allow this access, so a wildcard is safe here
                                        resourceName: '*',
                                        arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
                                        partition: resourceStack.partition,
                                        account: resourceStack.account,
                                        region: resourceStack.region,
                                    }));
                                    // Work around a bug in SecretsManager -
                                    // when the access is cross-environment,
                                    // Secret.secretArn returns a partial ARN!
                                    // So add a "*" at the end, so that the permissions work
                                    secretArn = `${secretArn}-??????`;
                                }
                            }
                        }
                        // if we are passed a Token, we should assume it's the ARN of the Secret
                        // (as the name would not work anyway, because it would be the full name, which CodeBuild does not support)
                        secretsManagerIamResources.add(secretArn);
                    }
                    else {
                        // the value of the property can be a complex string, separated by ':';
                        // see https://docs.aws.amazon.com/codebuild/latest/userguide/build-spec-ref.html#build-spec.env.secrets-manager
                        const secretName = envVariableValue.split(':')[0];
                        secretsManagerIamResources.add(stack.formatArn({
                            service: 'secretsmanager',
                            resource: 'secret',
                            resourceName: `${secretName}-??????`,
                            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                        }));
                    }
                }
            }
        }
        if (ssmIamResources.length !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['ssm:GetParameters'],
                resources: ssmIamResources,
            }));
        }
        if (secretsManagerIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['secretsmanager:GetSecretValue'],
                resources: Array.from(secretsManagerIamResources),
            }));
        }
        if (kmsIamResources.size !== 0) {
            principal === null || principal === void 0 ? void 0 : principal.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: ['kms:Decrypt'],
                resources: Array.from(kmsIamResources),
            }));
        }
        return ret;
    }
    enableBatchBuilds() {
        if (!this._batchServiceRole) {
            this._batchServiceRole = new iam.Role(this, 'BatchServiceRole', {
                assumedBy: new iam.ServicePrincipal('codebuild.amazonaws.com'),
            });
            this._batchServiceRole.addToPrincipalPolicy(new iam.PolicyStatement({
                resources: [core_1.Lazy.string({
                        produce: () => this.projectArn,
                    })],
                actions: [
                    'codebuild:StartBuild',
                    'codebuild:StopBuild',
                    'codebuild:RetryBuild',
                ],
            }));
        }
        return {
            role: this._batchServiceRole,
        };
    }
    /**
     * Adds a secondary source to the Project.
     *
     * @param secondarySource the source to add as a secondary source
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondarySource(secondarySource) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_ISource(secondarySource);
        if (!secondarySource.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary sources');
        }
        const secondarySourceConfig = secondarySource.bind(this, this);
        this._secondarySources.push(secondarySourceConfig.sourceProperty);
        if (secondarySourceConfig.sourceVersion) {
            this._secondarySourceVersions.push({
                sourceIdentifier: secondarySource.identifier,
                sourceVersion: secondarySourceConfig.sourceVersion,
            });
        }
    }
    /**
     * Adds a fileSystemLocation to the Project.
     *
     * @param fileSystemLocation the fileSystemLocation to add
     */
    addFileSystemLocation(fileSystemLocation) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_IFileSystemLocation(fileSystemLocation);
        const fileSystemConfig = fileSystemLocation.bind(this, this);
        this._fileSystemLocations.push(fileSystemConfig.location);
    }
    /**
     * Adds a secondary artifact to the Project.
     *
     * @param secondaryArtifact the artifact to add as a secondary artifact
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-multi-in-out.html
     */
    addSecondaryArtifact(secondaryArtifact) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_IArtifacts(secondaryArtifact);
        if (!secondaryArtifact.identifier) {
            throw new Error('The identifier attribute is mandatory for secondary artifacts');
        }
        this._secondaryArtifacts.push(secondaryArtifact.bind(this, this).artifactsProperty);
    }
    /**
     * A callback invoked when the given project is added to a CodePipeline.
     *
     * @param _scope the construct the binding is taking place in
     * @param options additional options for the binding
     */
    bindToCodePipeline(_scope, options) {
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_BindToCodePipelineOptions(options);
        // work around a bug in CodeBuild: it ignores the KMS key set on the pipeline,
        // and always uses its own, project-level key
        if (options.artifactBucket.encryptionKey && !this._encryptionKey) {
            // we cannot safely do this assignment if the key is of type kms.Key,
            // and belongs to a stack in a different account or region than the project
            // (that would cause an illegal reference, as KMS keys don't have physical names)
            const keyStack = core_1.Stack.of(options.artifactBucket.encryptionKey);
            const projectStack = core_1.Stack.of(this);
            if (!(options.artifactBucket.encryptionKey instanceof kms.Key &&
                (keyStack.account !== projectStack.account || keyStack.region !== projectStack.region))) {
                this.encryptionKey = options.artifactBucket.encryptionKey;
            }
        }
    }
    /**
     * @override
     */
    validate() {
        const ret = new Array();
        if (this.source.type === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) {
            if (this._secondarySources.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary sources. ' +
                    "Use the CodeBuild Pipeline Actions' `extraInputs` property instead");
            }
            if (this._secondaryArtifacts.length > 0) {
                ret.push('A Project with a CodePipeline Source cannot have secondary artifacts. ' +
                    "Use the CodeBuild Pipeline Actions' `outputs` property instead");
            }
        }
        return ret;
    }
    set encryptionKey(encryptionKey) {
        this._encryptionKey = encryptionKey;
        encryptionKey.grantEncryptDecrypt(this);
    }
    createLoggingPermission() {
        const logGroupArn = core_1.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
            resourceName: `/aws/codebuild/${this.projectName}`,
        });
        const logGroupStarArn = `${logGroupArn}:*`;
        return new iam.PolicyStatement({
            resources: [logGroupArn, logGroupStarArn],
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
        });
    }
    renderEnvironment(props, projectVars = {}) {
        var _d, _e, _f, _g, _h;
        const env = (_d = props.environment) !== null && _d !== void 0 ? _d : {};
        const vars = {};
        const containerVars = env.environmentVariables || {};
        // first apply environment variables from the container definition
        for (const name of Object.keys(containerVars)) {
            vars[name] = containerVars[name];
        }
        // now apply project-level vars
        for (const name of Object.keys(projectVars)) {
            vars[name] = projectVars[name];
        }
        const hasEnvironmentVars = Object.keys(vars).length > 0;
        const errors = this.buildImage.validate(env);
        if (errors.length > 0) {
            throw new Error('Invalid CodeBuild environment: ' + errors.join('\n'));
        }
        const imagePullPrincipalType = this.buildImage.imagePullPrincipalType === ImagePullPrincipalType.CODEBUILD
            ? ImagePullPrincipalType.CODEBUILD
            : ImagePullPrincipalType.SERVICE_ROLE;
        if (this.buildImage.repository) {
            if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
                this.buildImage.repository.grantPull(this);
            }
            else {
                const statement = new iam.PolicyStatement({
                    principals: [new iam.ServicePrincipal('codebuild.amazonaws.com')],
                    actions: ['ecr:GetDownloadUrlForLayer', 'ecr:BatchGetImage', 'ecr:BatchCheckLayerAvailability'],
                });
                statement.sid = 'CodeBuild';
                this.buildImage.repository.addToResourcePolicy(statement);
            }
        }
        if (imagePullPrincipalType === ImagePullPrincipalType.SERVICE_ROLE) {
            (_e = this.buildImage.secretsManagerCredentials) === null || _e === void 0 ? void 0 : _e.grantRead(this);
        }
        const secret = this.buildImage.secretsManagerCredentials;
        return {
            type: this.buildImage.type,
            image: this.buildImage.imageId,
            imagePullCredentialsType: imagePullPrincipalType,
            registryCredential: secret
                ? {
                    credentialProvider: 'SECRETS_MANAGER',
                    // Secrets must be referenced by either the full ARN (with SecretsManager suffix), or by name.
                    // "Partial" ARNs (without the suffix) will fail a validation regex at deploy-time.
                    credential: (_f = secret.secretFullArn) !== null && _f !== void 0 ? _f : secret.secretName,
                }
                : undefined,
            certificate: (_g = env.certificate) === null || _g === void 0 ? void 0 : _g.bucket.arnForObjects(env.certificate.objectKey),
            privilegedMode: env.privileged || false,
            computeType: env.computeType || this.buildImage.defaultComputeType,
            environmentVariables: hasEnvironmentVars
                ? Project.serializeEnvVariables(vars, (_h = props.checkSecretsInPlainTextEnvVariables) !== null && _h !== void 0 ? _h : true, this)
                : undefined,
        };
    }
    renderFileSystemLocations() {
        return this._fileSystemLocations.length === 0
            ? undefined
            : this._fileSystemLocations;
    }
    renderSecondarySources() {
        return this._secondarySources.length === 0
            ? undefined
            : this._secondarySources;
    }
    renderSecondarySourceVersions() {
        return this._secondarySourceVersions.length === 0
            ? undefined
            : this._secondarySourceVersions;
    }
    renderSecondaryArtifacts() {
        return this._secondaryArtifacts.length === 0
            ? undefined
            : this._secondaryArtifacts;
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * codebuild creation properties.
     */
    configureVpc(props) {
        if ((props.securityGroups || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if ((props.securityGroups && props.securityGroups.length > 0) && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroups && props.securityGroups.length > 0) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic generated security group for CodeBuild ' + core_1.Names.uniqueId(this),
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        return {
            vpcId: props.vpc.vpcId,
            subnets: props.vpc.selectSubnets(props.subnetSelection).subnetIds,
            securityGroupIds: this.connections.securityGroups.map(s => s.securityGroupId),
        };
    }
    renderLoggingConfiguration(props) {
        var _d, _e, _f, _g, _h;
        if (props === undefined) {
            return undefined;
        }
        let s3Config = undefined;
        let cloudwatchConfig = undefined;
        if (props.s3) {
            const s3Logs = props.s3;
            s3Config = {
                status: ((_d = s3Logs.enabled) !== null && _d !== void 0 ? _d : true) ? 'ENABLED' : 'DISABLED',
                location: `${s3Logs.bucket.bucketName}` + (s3Logs.prefix ? `/${s3Logs.prefix}` : ''),
                encryptionDisabled: s3Logs.encrypted,
            };
            (_e = s3Logs.bucket) === null || _e === void 0 ? void 0 : _e.grantWrite(this);
        }
        if (props.cloudWatch) {
            const cloudWatchLogs = props.cloudWatch;
            const status = ((_f = cloudWatchLogs.enabled) !== null && _f !== void 0 ? _f : true) ? 'ENABLED' : 'DISABLED';
            if (status === 'ENABLED' && !(cloudWatchLogs.logGroup)) {
                throw new Error('Specifying a LogGroup is required if CloudWatch logging for CodeBuild is enabled');
            }
            (_g = cloudWatchLogs.logGroup) === null || _g === void 0 ? void 0 : _g.grantWrite(this);
            cloudwatchConfig = {
                status,
                groupName: (_h = cloudWatchLogs.logGroup) === null || _h === void 0 ? void 0 : _h.logGroupName,
                streamName: cloudWatchLogs.prefix,
            };
        }
        return {
            s3Logs: s3Config,
            cloudWatchLogs: cloudwatchConfig,
        };
    }
    addVpcRequiredPermissions(props, project) {
        if (!props.vpc || !this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            resources: [`arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:network-interface/*`],
            actions: ['ec2:CreateNetworkInterfacePermission'],
            conditions: {
                StringEquals: {
                    'ec2:Subnet': props.vpc
                        .selectSubnets(props.subnetSelection).subnetIds
                        .map(si => `arn:${core_1.Aws.PARTITION}:ec2:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:subnet/${si}`),
                    'ec2:AuthorizedService': 'codebuild.amazonaws.com',
                },
            },
        }));
        // If the same Role is used for multiple Projects, always creating a new `iam.Policy`
        // will attach the same policy multiple times, probably exceeding the maximum size of the
        // Role policy. Make sure we only do it once for the same role.
        //
        // This deduplication could be a feature of the Role itself, but that feels risky and
        // is hard to implement (what with Tokens and all). Safer to fix it locally for now.
        let policy = this.role[VPC_POLICY_SYM];
        if (!policy) {
            policy = new iam.Policy(this, 'PolicyDocument', {
                statements: [
                    new iam.PolicyStatement({
                        resources: ['*'],
                        actions: [
                            'ec2:CreateNetworkInterface',
                            'ec2:DescribeNetworkInterfaces',
                            'ec2:DeleteNetworkInterface',
                            'ec2:DescribeSubnets',
                            'ec2:DescribeSecurityGroups',
                            'ec2:DescribeDhcpOptions',
                            'ec2:DescribeVpcs',
                        ],
                    }),
                ],
            });
            this.role.attachInlinePolicy(policy);
            this.role[VPC_POLICY_SYM] = policy;
        }
        // add an explicit dependency between the EC2 Policy and this Project -
        // otherwise, creating the Project fails, as it requires these permissions
        // to be already attached to the Project's Role
        project.node.addDependency(policy);
    }
    validateCodePipelineSettings(artifacts) {
        const sourceType = this.source.type;
        const artifactsType = artifacts.type;
        if ((sourceType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE ||
            artifactsType === source_types_1.CODEPIPELINE_SOURCE_ARTIFACTS_TYPE) &&
            (sourceType !== artifactsType)) {
            throw new Error('Both source and artifacts must be set to CodePipeline');
        }
    }
}
exports.Project = Project;
_a = JSII_RTTI_SYMBOL_1;
Project[_a] = { fqn: "monocdk.aws_codebuild.Project", version: "1.149.0" };
/**
 * Build machine compute type.
 */
var ComputeType;
(function (ComputeType) {
    ComputeType["SMALL"] = "BUILD_GENERAL1_SMALL";
    ComputeType["MEDIUM"] = "BUILD_GENERAL1_MEDIUM";
    ComputeType["LARGE"] = "BUILD_GENERAL1_LARGE";
    ComputeType["X2_LARGE"] = "BUILD_GENERAL1_2XLARGE";
})(ComputeType = exports.ComputeType || (exports.ComputeType = {}));
/**
 * The type of principal CodeBuild will use to pull your build Docker image.
 */
var ImagePullPrincipalType;
(function (ImagePullPrincipalType) {
    /**
     * CODEBUILD specifies that CodeBuild uses its own identity when pulling the image.
     * This means the resource policy of the ECR repository that hosts the image will be modified to trust
     * CodeBuild's service principal.
     * This is the required principal type when using CodeBuild's pre-defined images.
     */
    ImagePullPrincipalType["CODEBUILD"] = "CODEBUILD";
    /**
     * SERVICE_ROLE specifies that AWS CodeBuild uses the project's role when pulling the image.
     * The role will be granted pull permissions on the ECR repository hosting the image.
     */
    ImagePullPrincipalType["SERVICE_ROLE"] = "SERVICE_ROLE";
})(ImagePullPrincipalType = exports.ImagePullPrincipalType || (exports.ImagePullPrincipalType = {}));
// Keep around to resolve a circular dependency until removing deprecated ARM image constants from LinuxBuildImage
// eslint-disable-next-line no-duplicate-imports, import/order
const linux_arm_build_image_1 = require("./linux-arm-build-image");
/**
 * A CodeBuild image running x86-64 Linux.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])
 * - LinuxBuildImage.fromEcrRepository(repo[, tag])
 * - LinuxBuildImage.fromAsset(parent, id, props)
 *
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class LinuxBuildImage {
    constructor(props) {
        this.type = 'LINUX_CONTAINER';
        this.defaultComputeType = ComputeType.SMALL;
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a x86-64 Linux build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_DockerImageOptions(options);
        return new LinuxBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
        });
    }
    /**
     * @returns A x86-64 Linux build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        jsiiDeprecationWarnings.monocdk_aws_ecr_IRepository(repository);
        return new LinuxBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a x86-64 Linux build image.
     */
    static fromAsset(scope, id, props) {
        jsiiDeprecationWarnings.monocdk_aws_ecr_assets_DockerImageAssetProps(props);
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new LinuxBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            repository: asset.repository,
        });
    }
    /**
     * Uses a Docker image provided by CodeBuild.
     *
     * @returns A Docker image provided by CodeBuild.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
     *
     * @param id The image identifier
     * @example 'aws/codebuild/standard:4.0'
     */
    static fromCodeBuildImageId(id) {
        return LinuxBuildImage.codeBuildImage(id);
    }
    static codeBuildImage(name) {
        return new LinuxBuildImage({
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
        });
    }
    validate(_) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_BuildEnvironment(_);
        return [];
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxBuildImage = LinuxBuildImage;
_b = JSII_RTTI_SYMBOL_1;
LinuxBuildImage[_b] = { fqn: "monocdk.aws_codebuild.LinuxBuildImage", version: "1.149.0" };
LinuxBuildImage.STANDARD_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:1.0');
LinuxBuildImage.STANDARD_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:2.0');
LinuxBuildImage.STANDARD_3_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:3.0');
/** The `aws/codebuild/standard:4.0` build image. */
LinuxBuildImage.STANDARD_4_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:4.0');
/** The `aws/codebuild/standard:5.0` build image. */
LinuxBuildImage.STANDARD_5_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/standard:5.0');
LinuxBuildImage.AMAZON_LINUX_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:1.0');
LinuxBuildImage.AMAZON_LINUX_2_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:2.0');
/** The Amazon Linux 2 x86_64 standard image, version `3.0`. */
LinuxBuildImage.AMAZON_LINUX_2_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/amazonlinux2-x86_64-standard:3.0');
/** @deprecated Use LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_1_0 instead. */
LinuxBuildImage.AMAZON_LINUX_2_ARM = linux_arm_build_image_1.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_1_0;
/**
 * Image "aws/codebuild/amazonlinux2-aarch64-standard:2.0".
 * @deprecated Use LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0 instead.
 * */
LinuxBuildImage.AMAZON_LINUX_2_ARM_2 = linux_arm_build_image_1.LinuxArmBuildImage.AMAZON_LINUX_2_STANDARD_2_0;
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_BASE = LinuxBuildImage.codeBuildImage('aws/codebuild/ubuntu-base:14.04');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_24_4_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:24.4.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_ANDROID_JAVA8_26_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/android-java-8:26.1.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_17_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:17.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOCKER_18_09_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/docker:18.09.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_10 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.10');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_GOLANG_1_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/golang:1.11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_8 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-8');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_9 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-9');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_OPEN_JDK_11 = LinuxBuildImage.codeBuildImage('aws/codebuild/java:openjdk-11');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_14_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.14.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_10_1_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:10.1.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_8_11_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:8.11.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_NODEJS_6_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/nodejs:6.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_5_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:5.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PHP_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/php:7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_7_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.7.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_6_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.6.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_5_2 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.5.2');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_4_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.4.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_3_3_6 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:3.3.6');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_PYTHON_2_7_12 = LinuxBuildImage.codeBuildImage('aws/codebuild/python:2.7.12');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_3 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.3');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_5_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.5.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_3_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.3.1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_RUBY_2_2_5 = LinuxBuildImage.codeBuildImage('aws/codebuild/ruby:2.2.5');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_1_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-1');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_0 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.0');
/** @deprecated Use {@link STANDARD_2_0} and specify runtime in buildspec runtime-versions section */
LinuxBuildImage.UBUNTU_14_04_DOTNET_CORE_2_1 = LinuxBuildImage.codeBuildImage('aws/codebuild/dot-net:core-2.1');
/**
 * Environment type for Windows Docker images
 */
var WindowsImageType;
(function (WindowsImageType) {
    /**
     * The standard environment type, WINDOWS_CONTAINER
     */
    WindowsImageType["STANDARD"] = "WINDOWS_CONTAINER";
    /**
     * The WINDOWS_SERVER_2019_CONTAINER environment type
     */
    WindowsImageType["SERVER_2019"] = "WINDOWS_SERVER_2019_CONTAINER";
})(WindowsImageType = exports.WindowsImageType || (exports.WindowsImageType = {}));
/**
 * A CodeBuild image running Windows.
 *
 * This class has a bunch of public constants that represent the most popular images.
 *
 * You can also specify a custom image using one of the static methods:
 *
 * - WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])
 * - WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])
 * - WindowsBuildImage.fromAsset(parent, id, props, [, imageType])
 *
 * @see https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-available.html
 */
class WindowsBuildImage {
    constructor(props) {
        var _d;
        this.defaultComputeType = ComputeType.MEDIUM;
        this.type = ((_d = props.imageType) !== null && _d !== void 0 ? _d : WindowsImageType.STANDARD).toString();
        this.imageId = props.imageId;
        this.imagePullPrincipalType = props.imagePullPrincipalType;
        this.secretsManagerCredentials = props.secretsManagerCredentials;
        this.repository = props.repository;
    }
    /**
     * @returns a Windows build image from a Docker Hub image.
     */
    static fromDockerRegistry(name, options = {}, imageType = WindowsImageType.STANDARD) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_DockerImageOptions(options);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_WindowsImageType(imageType);
        return new WindowsBuildImage({
            ...options,
            imageId: name,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
        });
    }
    /**
     * @returns A Windows build image from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest', imageType = WindowsImageType.STANDARD) {
        jsiiDeprecationWarnings.monocdk_aws_ecr_IRepository(repository);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_WindowsImageType(imageType);
        return new WindowsBuildImage({
            imageId: repository.repositoryUriForTag(tag),
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository,
        });
    }
    /**
     * Uses an Docker image asset as a Windows build image.
     */
    static fromAsset(scope, id, props, imageType = WindowsImageType.STANDARD) {
        jsiiDeprecationWarnings.monocdk_aws_ecr_assets_DockerImageAssetProps(props);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_WindowsImageType(imageType);
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new WindowsBuildImage({
            imageId: asset.imageUri,
            imagePullPrincipalType: ImagePullPrincipalType.SERVICE_ROLE,
            imageType,
            repository: asset.repository,
        });
    }
    validate(buildEnvironment) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_BuildEnvironment(buildEnvironment);
        const ret = [];
        if (buildEnvironment.computeType === ComputeType.SMALL) {
            ret.push('Windows images do not support the Small ComputeType');
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return build_spec_1.BuildSpec.fromObject({
            version: '0.2',
            phases: {
                pre_build: {
                    // Would love to do downloading here and executing in the next step,
                    // but I don't know how to propagate the value of $TEMPDIR.
                    //
                    // Punting for someone who knows PowerShell well enough.
                    commands: [],
                },
                build: {
                    commands: [
                        'Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName',
                        `aws s3 cp s3://$env:${run_script_linux_build_spec_1.S3_BUCKET_ENV}/$env:${run_script_linux_build_spec_1.S3_KEY_ENV} $TEMPDIR\\scripts.zip`,
                        'New-Item -ItemType Directory -Path $TEMPDIR\\scriptdir',
                        'Expand-Archive -Path $TEMPDIR/scripts.zip -DestinationPath $TEMPDIR\\scriptdir',
                        '$env:SCRIPT_DIR = "$TEMPDIR\\scriptdir"',
                        `& $TEMPDIR\\scriptdir\\${entrypoint}`,
                    ],
                },
            },
        });
    }
}
exports.WindowsBuildImage = WindowsBuildImage;
_c = JSII_RTTI_SYMBOL_1;
WindowsBuildImage[_c] = { fqn: "monocdk.aws_codebuild.WindowsBuildImage", version: "1.149.0" };
/**
 * Corresponds to the standard CodeBuild image `aws/codebuild/windows-base:1.0`.
 *
 * @deprecated `WindowsBuildImage.WINDOWS_BASE_2_0` should be used instead.
 */
WindowsBuildImage.WIN_SERVER_CORE_2016_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2.0`, which is
 * based off Windows Server Core 2016.
 */
WindowsBuildImage.WINDOWS_BASE_2_0 = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
});
/**
 * The standard CodeBuild image `aws/codebuild/windows-base:2019-1.0`, which is
 * based off Windows Server Core 2019.
 */
WindowsBuildImage.WIN_SERVER_CORE_2019_BASE = new WindowsBuildImage({
    imageId: 'aws/codebuild/windows-base:2019-1.0',
    imagePullPrincipalType: ImagePullPrincipalType.CODEBUILD,
    imageType: WindowsImageType.SERVER_2019,
});
var BuildEnvironmentVariableType;
(function (BuildEnvironmentVariableType) {
    /**
     * An environment variable in plaintext format.
     */
    BuildEnvironmentVariableType["PLAINTEXT"] = "PLAINTEXT";
    /**
     * An environment variable stored in Systems Manager Parameter Store.
     */
    BuildEnvironmentVariableType["PARAMETER_STORE"] = "PARAMETER_STORE";
    /**
     * An environment variable stored in AWS Secrets Manager.
     */
    BuildEnvironmentVariableType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
})(BuildEnvironmentVariableType = exports.BuildEnvironmentVariableType || (exports.BuildEnvironmentVariableType = {}));
/**
 * The list of event types for AWS Codebuild
 * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-buildproject
 */
var ProjectNotificationEvents;
(function (ProjectNotificationEvents) {
    /**
     * Trigger notification when project build state failed
     */
    ProjectNotificationEvents["BUILD_FAILED"] = "codebuild-project-build-state-failed";
    /**
     * Trigger notification when project build state succeeded
     */
    ProjectNotificationEvents["BUILD_SUCCEEDED"] = "codebuild-project-build-state-succeeded";
    /**
     * Trigger notification when project build state in progress
     */
    ProjectNotificationEvents["BUILD_IN_PROGRESS"] = "codebuild-project-build-state-in-progress";
    /**
     * Trigger notification when project build state stopped
     */
    ProjectNotificationEvents["BUILD_STOPPED"] = "codebuild-project-build-state-stopped";
    /**
     * Trigger notification when project build phase failure
     */
    ProjectNotificationEvents["BUILD_PHASE_FAILED"] = "codebuild-project-build-phase-failure";
    /**
     * Trigger notification when project build phase success
     */
    ProjectNotificationEvents["BUILD_PHASE_SUCCEEDED"] = "codebuild-project-build-phase-success";
})(ProjectNotificationEvents = exports.ProjectNotificationEvents || (exports.ProjectNotificationEvents = {}));
function isBindableBuildImage(x) {
    return typeof x === 'object' && !!x && !!x.bind;
}
//# sourceMappingURL=data:application/json;base64,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