"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LinuxGpuBuildImage = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ecr = require("../../aws-ecr");
const core = require("../../core");
const region_info_1 = require("../../region-info");
const run_script_linux_build_spec_1 = require("./private/run-script-linux-build-spec");
const project_1 = require("./project");
/**
 * A CodeBuild GPU image running Linux.
 *
 * This class has public constants that represent the most popular GPU images from AWS Deep Learning Containers.
 *
 * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
 */
class LinuxGpuBuildImage {
    constructor(repositoryName, tag, account) {
        this.repositoryName = repositoryName;
        this.account = account;
        this.type = 'LINUX_GPU_CONTAINER';
        this.defaultComputeType = project_1.ComputeType.LARGE;
        this.imagePullPrincipalType = project_1.ImagePullPrincipalType.SERVICE_ROLE;
        const imageAccount = account !== null && account !== void 0 ? account : core.Lazy.string({
            produce: () => {
                if (this._imageAccount === undefined) {
                    throw new Error('Make sure this \'LinuxGpuBuildImage\' is used in a CodeBuild Project construct');
                }
                return this._imageAccount;
            },
        });
        // The value of imageId below *should* have been `Lazy.stringValue(() => repository.repositoryUriForTag(this.tag))`,
        // but we can't change that anymore because someone somewhere might at this point have written code
        // to do `image.imageId.includes('pytorch')` and changing this to a full-on token would break them.
        this.imageId = `${imageAccount}.dkr.ecr.${core.Aws.REGION}.${core.Aws.URL_SUFFIX}/${repositoryName}:${tag}`;
    }
    /**
     * Returns a Linux GPU build image from AWS Deep Learning Containers.
     *
     * @param repositoryName the name of the repository,
     *   for example "pytorch-inference"
     * @param tag the tag of the image, for example "1.5.0-gpu-py36-cu101-ubuntu16.04"
     * @param account the AWS account ID where the DLC repository for this region is hosted in.
     *   In many cases, the CDK can infer that for you, but for some newer region our information
     *   might be out of date; in that case, you can specify the region explicitly using this optional parameter
     * @see https://aws.amazon.com/releasenotes/available-deep-learning-containers-images
     */
    static awsDeepLearningContainersImage(repositoryName, tag, account) {
        return new LinuxGpuBuildImage(repositoryName, tag, account);
    }
    /**
     * Returns a GPU image running Linux from an ECR repository.
     *
     * NOTE: if the repository is external (i.e. imported), then we won't be able to add
     * a resource policy statement for it so CodeBuild can pull the image.
     *
     * @see https://docs.aws.amazon.com/codebuild/latest/userguide/sample-ecr.html
     *
     * @param repository The ECR repository
     * @param tag Image tag (default "latest")
     */
    static fromEcrRepository(repository, tag = 'latest') {
        jsiiDeprecationWarnings.monocdk_aws_ecr_IRepository(repository);
        return new LinuxGpuBuildImage(repository.repositoryName, tag, repository.env.account);
    }
    bind(scope, project, _options) {
        var _b;
        jsiiDeprecationWarnings.monocdk_Construct(scope);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_IProject(project);
        jsiiDeprecationWarnings.monocdk_aws_codebuild_BuildImageBindOptions(_options);
        const account = (_b = this.account) !== null && _b !== void 0 ? _b : core.Stack.of(scope).regionalFact(region_info_1.FactName.DLC_REPOSITORY_ACCOUNT);
        const repository = ecr.Repository.fromRepositoryAttributes(scope, 'AwsDlcRepositoryCodeBuild', {
            repositoryName: this.repositoryName,
            repositoryArn: ecr.Repository.arnForLocalRepository(this.repositoryName, scope, account),
        });
        repository.grantPull(project);
        this._imageAccount = account;
        return {};
    }
    validate(buildEnvironment) {
        jsiiDeprecationWarnings.monocdk_aws_codebuild_BuildEnvironment(buildEnvironment);
        const ret = [];
        if (buildEnvironment.computeType &&
            buildEnvironment.computeType !== project_1.ComputeType.LARGE) {
            ret.push(`GPU images only support ComputeType '${project_1.ComputeType.LARGE}' - ` +
                `'${buildEnvironment.computeType}' was given`);
        }
        return ret;
    }
    runScriptBuildspec(entrypoint) {
        return run_script_linux_build_spec_1.runScriptLinuxBuildSpec(entrypoint);
    }
}
exports.LinuxGpuBuildImage = LinuxGpuBuildImage;
_a = JSII_RTTI_SYMBOL_1;
LinuxGpuBuildImage[_a] = { fqn: "monocdk.aws_codebuild.LinuxGpuBuildImage", version: "1.149.0" };
/** Tensorflow 1.14.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_14_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.14.0-gpu-py36-cu100-ubuntu16.04');
/** Tensorflow 1.15.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.0-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 1.15.2 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '1.15.2-gpu-py37-cu100-ubuntu18.04');
/** Tensorflow 1.15.2 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_1_15_2_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '1.15.2-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.0.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.0-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.0.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_0_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.0.1-gpu-py36-cu100-ubuntu18.04');
/** Tensorflow 2.1.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/** Tensorflow 2.1.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-inference', '2.1.0-gpu-py36-cu101-ubuntu18.04');
/** Tensorflow 2.2.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_TENSORFLOW_2_2_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('tensorflow-training', '2.2.0-gpu-py37-cu101-ubuntu18.04');
/** PyTorch 1.2.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_2_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.2.0-gpu-py36-cu100-ubuntu16.04');
/** PyTorch 1.3.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_3_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.3.1-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.4.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.4.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_4_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.4.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.5.0 GPU training image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_TRAINING = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-training', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/** PyTorch 1.5.0 GPU inference image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_PYTORCH_1_5_0_INFERENCE = LinuxGpuBuildImage.awsDeepLearningContainersImage('pytorch-inference', '1.5.0-gpu-py36-cu101-ubuntu16.04');
/** MXNet 1.4.1 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_MXNET_1_4_1 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.4.1-gpu-py36-cu100-ubuntu16.04');
/** MXNet 1.6.0 GPU image from AWS Deep Learning Containers. */
LinuxGpuBuildImage.DLC_MXNET_1_6_0 = LinuxGpuBuildImage.awsDeepLearningContainersImage('mxnet-training', '1.6.0-gpu-py36-cu101-ubuntu16.04');
//# sourceMappingURL=data:application/json;base64,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