"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.ViewerCertificate = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginSslPolicy = exports.FailoverStatusCode = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const certificatemanager = require("../../aws-certificatemanager");
const iam = require("../../aws-iam");
const s3 = require("../../aws-s3");
const cdk = require("../../core");
const cloudfront_generated_1 = require("./cloudfront.generated");
const distribution_1 = require("./distribution");
/**
 * HTTP status code to failover to second origin
 */
var FailoverStatusCode;
(function (FailoverStatusCode) {
    /**
     * Forbidden (403)
     */
    FailoverStatusCode[FailoverStatusCode["FORBIDDEN"] = 403] = "FORBIDDEN";
    /**
     * Not found (404)
     */
    FailoverStatusCode[FailoverStatusCode["NOT_FOUND"] = 404] = "NOT_FOUND";
    /**
     * Internal Server Error (500)
     */
    FailoverStatusCode[FailoverStatusCode["INTERNAL_SERVER_ERROR"] = 500] = "INTERNAL_SERVER_ERROR";
    /**
     * Bad Gateway (502)
     */
    FailoverStatusCode[FailoverStatusCode["BAD_GATEWAY"] = 502] = "BAD_GATEWAY";
    /**
     * Service Unavailable (503)
     */
    FailoverStatusCode[FailoverStatusCode["SERVICE_UNAVAILABLE"] = 503] = "SERVICE_UNAVAILABLE";
    /**
     * Gateway Timeout (504)
     */
    FailoverStatusCode[FailoverStatusCode["GATEWAY_TIMEOUT"] = 504] = "GATEWAY_TIMEOUT";
})(FailoverStatusCode = exports.FailoverStatusCode || (exports.FailoverStatusCode = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_certificatemanager_ICertificate(certificate);
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_ViewerCertificateOptions(options);
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_ViewerCertificateOptions(options);
        const { sslMethod: sslSupportMethod = distribution_1.SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
_a = JSII_RTTI_SYMBOL_1;
ViewerCertificate[_a] = { fqn: "monocdk.aws_cloudfront.ViewerCertificate", version: "1.149.0" };
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * const sourceBucket = new s3.Bucket(this, 'Bucket');
 *
 * const distribution = new cloudfront.CloudFrontWebDistribution(this, 'MyDistribution', {
 *   originConfigs: [
 *     {
 *       s3OriginSource: {
 *       s3BucketSource: sourceBucket,
 *       },
 *       behaviors : [ {isDefaultBehavior: true}],
 *     },
 *   ],
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 * @resource AWS::CloudFront::Distribution
 */
class CloudFrontWebDistribution extends cdk.Resource {
    constructor(scope, id, props) {
        var _c, _d, _e, _f;
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [distribution_1.SSLMethod.SNI]: [
                distribution_1.SecurityPolicyProtocol.TLS_V1, distribution_1.SecurityPolicyProtocol.TLS_V1_1_2016,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2016, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2018,
                distribution_1.SecurityPolicyProtocol.TLS_V1_2_2019, distribution_1.SecurityPolicyProtocol.TLS_V1_2_2021,
            ],
            [distribution_1.SSLMethod.VIP]: [distribution_1.SecurityPolicyProtocol.SSL_V3, distribution_1.SecurityPolicyProtocol.TLS_V1],
        };
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_CloudFrontWebDistributionProps(props);
        // Comments have an undocumented limit of 128 characters
        const trimmedComment = props.comment && props.comment.length > 128
            ? `${props.comment.substr(0, 128 - 3)}...`
            : props.comment;
        let distributionConfig = {
            comment: trimmedComment,
            enabled: (_c = props.enabled) !== null && _c !== void 0 ? _c : true,
            defaultRootObject: (_d = props.defaultRootObject) !== null && _d !== void 0 ? _d : 'index.html',
            httpVersion: props.httpVersion || distribution_1.HttpVersion.HTTP2,
            priceClass: props.priceClass || distribution_1.PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (_e = props.enableIpV6) !== null && _e !== void 0 ? _e : true,
            // eslint-disable-next-line max-len
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        const originGroups = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            let originId = `origin${originIndex}`;
            const originProperty = this.toOriginProperty(originConfig, originId);
            if (originConfig.failoverCustomOriginSource || originConfig.failoverS3OriginSource) {
                const originSecondaryId = `originSecondary${originIndex}`;
                const originSecondaryProperty = this.toOriginProperty({
                    s3OriginSource: originConfig.failoverS3OriginSource,
                    customOriginSource: originConfig.failoverCustomOriginSource,
                    originPath: originConfig.originPath,
                    originHeaders: originConfig.originHeaders,
                    originShieldRegion: originConfig.originShieldRegion,
                }, originSecondaryId);
                const originGroupsId = `OriginGroup${originIndex}`;
                const failoverCodes = (_f = originConfig.failoverCriteriaStatusCodes) !== null && _f !== void 0 ? _f : [500, 502, 503, 504];
                originGroups.push({
                    id: originGroupsId,
                    members: {
                        items: [{ originId }, { originId: originSecondaryId }],
                        quantity: 2,
                    },
                    failoverCriteria: {
                        statusCodes: {
                            items: failoverCodes,
                            quantity: failoverCodes.length,
                        },
                    },
                });
                originId = originGroupsId;
                origins.push(originSecondaryProperty);
            }
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        const originGroupsDistConfig = originGroups.length > 0
            ? {
                items: originGroups,
                quantity: originGroups.length,
            }
            : undefined;
        distributionConfig = {
            ...distributionConfig,
            origins,
            originGroups: originGroupsDistConfig,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // eslint-disable-next-line max-len
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = {
                ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket', {
                encryption: s3.BucketEncryption.S3_MANAGED,
            });
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionDomainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    /**
     * Creates a construct that represents an external (imported) distribution.
     */
    static fromDistributionAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_CloudFrontWebDistributionAttributes(attrs);
        return new class extends cdk.Resource {
            constructor() {
                super(scope, id);
                this.domainName = attrs.domainName;
                this.distributionDomainName = attrs.domainName;
                this.distributionId = attrs.distributionId;
            }
        }();
    }
    toBehavior(input, protoPolicy) {
        var _c;
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedKeyGroups: (_c = input.trustedKeyGroups) === null || _c === void 0 ? void 0 : _c.map(key => key.keyGroupId),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: input.viewerProtocolPolicy || protoPolicy || distribution_1.ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.functionAssociations) {
            toReturn = Object.assign(toReturn, {
                functionAssociations: input.functionAssociations.map(association => ({
                    functionArn: association.function.functionArn,
                    eventType: association.eventType.toString(),
                })),
            });
        }
        if (input.lambdaFunctionAssociations) {
            const includeBodyEventTypes = [distribution_1.LambdaEdgeEventType.ORIGIN_REQUEST, distribution_1.LambdaEdgeEventType.VIEWER_REQUEST];
            if (input.lambdaFunctionAssociations.some(fna => fna.includeBody && !includeBodyEventTypes.includes(fna.eventType))) {
                throw new Error('\'includeBody\' can only be true for ORIGIN_REQUEST or VIEWER_REQUEST event types.');
            }
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.edgeArn,
                    includeBody: fna.includeBody,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
    toOriginProperty(originConfig, originId) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s;
        if (!originConfig.s3OriginSource &&
            !originConfig.customOriginSource) {
            throw new Error('There must be at least one origin source - either an s3OriginSource, a customOriginSource');
        }
        if (originConfig.customOriginSource && originConfig.s3OriginSource) {
            throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
        }
        if ([
            originConfig.originHeaders,
            (_c = originConfig.s3OriginSource) === null || _c === void 0 ? void 0 : _c.originHeaders,
            (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originHeaders,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originHeaders field allowed across origin and failover origins');
        }
        if ([
            originConfig.originPath,
            (_e = originConfig.s3OriginSource) === null || _e === void 0 ? void 0 : _e.originPath,
            (_f = originConfig.customOriginSource) === null || _f === void 0 ? void 0 : _f.originPath,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originPath field allowed across origin and failover origins');
        }
        if ([
            originConfig.originShieldRegion,
            (_g = originConfig.s3OriginSource) === null || _g === void 0 ? void 0 : _g.originShieldRegion,
            (_h = originConfig.customOriginSource) === null || _h === void 0 ? void 0 : _h.originShieldRegion,
        ].filter(x => x).length > 1) {
            throw new Error('Only one originShieldRegion field allowed across origin and failover origins');
        }
        const headers = (_l = (_j = originConfig.originHeaders) !== null && _j !== void 0 ? _j : (_k = originConfig.s3OriginSource) === null || _k === void 0 ? void 0 : _k.originHeaders) !== null && _l !== void 0 ? _l : (_m = originConfig.customOriginSource) === null || _m === void 0 ? void 0 : _m.originHeaders;
        const originHeaders = [];
        if (headers) {
            Object.keys(headers).forEach((key) => {
                const oHeader = {
                    headerName: key,
                    headerValue: headers[key],
                };
                originHeaders.push(oHeader);
            });
        }
        let s3OriginConfig;
        if (originConfig.s3OriginSource) {
            // first case for backwards compatibility
            if (originConfig.s3OriginSource.originAccessIdentity) {
                // grant CloudFront OriginAccessIdentity read access to S3 bucket
                // Used rather than `grantRead` because `grantRead` will grant overly-permissive policies.
                // Only GetObject is needed to retrieve objects for the distribution.
                // This also excludes KMS permissions; currently, OAI only supports SSE-S3 for buckets.
                // Source: https://aws.amazon.com/blogs/networking-and-content-delivery/serving-sse-kms-encrypted-content-from-s3-using-cloudfront/
                originConfig.s3OriginSource.s3BucketSource.addToResourcePolicy(new iam.PolicyStatement({
                    resources: [originConfig.s3OriginSource.s3BucketSource.arnForObjects('*')],
                    actions: ['s3:GetObject'],
                    principals: [originConfig.s3OriginSource.originAccessIdentity.grantPrincipal],
                }));
                s3OriginConfig = {
                    originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                };
            }
            else {
                s3OriginConfig = {};
            }
        }
        const connectionAttempts = (_o = originConfig.connectionAttempts) !== null && _o !== void 0 ? _o : 3;
        if (connectionAttempts < 1 || 3 < connectionAttempts || !Number.isInteger(connectionAttempts)) {
            throw new Error('connectionAttempts: You can specify 1, 2, or 3 as the number of attempts.');
        }
        const connectionTimeout = (originConfig.connectionTimeout || cdk.Duration.seconds(10)).toSeconds();
        if (connectionTimeout < 1 || 10 < connectionTimeout || !Number.isInteger(connectionTimeout)) {
            throw new Error('connectionTimeout: You can specify a number of seconds between 1 and 10 (inclusive).');
        }
        const originProperty = {
            id: originId,
            domainName: originConfig.s3OriginSource
                ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                : originConfig.customOriginSource.domainName,
            originPath: (_r = (_p = originConfig.originPath) !== null && _p !== void 0 ? _p : (_q = originConfig.customOriginSource) === null || _q === void 0 ? void 0 : _q.originPath) !== null && _r !== void 0 ? _r : (_s = originConfig.s3OriginSource) === null || _s === void 0 ? void 0 : _s.originPath,
            originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
            s3OriginConfig,
            originShield: this.toOriginShieldProperty(originConfig),
            customOriginConfig: originConfig.customOriginSource
                ? {
                    httpPort: originConfig.customOriginSource.httpPort || 80,
                    httpsPort: originConfig.customOriginSource.httpsPort || 443,
                    originKeepaliveTimeout: (originConfig.customOriginSource.originKeepaliveTimeout &&
                        originConfig.customOriginSource.originKeepaliveTimeout.toSeconds()) ||
                        5,
                    originReadTimeout: (originConfig.customOriginSource.originReadTimeout &&
                        originConfig.customOriginSource.originReadTimeout.toSeconds()) ||
                        30,
                    originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy ||
                        distribution_1.OriginProtocolPolicy.HTTPS_ONLY,
                    originSslProtocols: originConfig.customOriginSource
                        .allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                }
                : undefined,
            connectionAttempts,
            connectionTimeout,
        };
        return originProperty;
    }
    /**
     * Takes origin shield region from props and converts to CfnDistribution.OriginShieldProperty
     */
    toOriginShieldProperty(originConfig) {
        var _c, _d, _e, _f;
        const originShieldRegion = (_e = (_c = originConfig.originShieldRegion) !== null && _c !== void 0 ? _c : (_d = originConfig.customOriginSource) === null || _d === void 0 ? void 0 : _d.originShieldRegion) !== null && _e !== void 0 ? _e : (_f = originConfig.s3OriginSource) === null || _f === void 0 ? void 0 : _f.originShieldRegion;
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
_b = JSII_RTTI_SYMBOL_1;
CloudFrontWebDistribution[_b] = { fqn: "monocdk.aws_cloudfront.CloudFrontWebDistribution", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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