"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginBase = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
/**
 * Represents a distribution origin, that describes the Amazon S3 bucket, HTTP server (for example, a web server),
 * Amazon MediaStore, or other server from which CloudFront gets your files.
 */
class OriginBase {
    constructor(domainName, props = {}) {
        var _b;
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_OriginProps(props);
        validateIntInRangeOrUndefined('connectionTimeout', 1, 10, (_b = props.connectionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds());
        validateIntInRangeOrUndefined('connectionAttempts', 1, 3, props.connectionAttempts, false);
        validateCustomHeaders(props.customHeaders);
        this.domainName = domainName;
        this.originPath = this.validateOriginPath(props.originPath);
        this.connectionTimeout = props.connectionTimeout;
        this.connectionAttempts = props.connectionAttempts;
        this.customHeaders = props.customHeaders;
        this.originShieldRegion = props.originShieldRegion;
    }
    /**
     * Binds the origin to the associated Distribution. Can be used to grant permissions, create dependent resources, etc.
     */
    bind(_scope, options) {
        var _b;
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_OriginBindOptions(options);
        const s3OriginConfig = this.renderS3OriginConfig();
        const customOriginConfig = this.renderCustomOriginConfig();
        if (!s3OriginConfig && !customOriginConfig) {
            throw new Error('Subclass must override and provide either s3OriginConfig or customOriginConfig');
        }
        return {
            originProperty: {
                domainName: this.domainName,
                id: options.originId,
                originPath: this.originPath,
                connectionAttempts: this.connectionAttempts,
                connectionTimeout: (_b = this.connectionTimeout) === null || _b === void 0 ? void 0 : _b.toSeconds(),
                originCustomHeaders: this.renderCustomHeaders(),
                s3OriginConfig,
                customOriginConfig,
                originShield: this.renderOriginShield(this.originShieldRegion),
            },
        };
    }
    // Overridden by sub-classes to provide S3 origin config.
    renderS3OriginConfig() {
        return undefined;
    }
    // Overridden by sub-classes to provide custom origin config.
    renderCustomOriginConfig() {
        return undefined;
    }
    renderCustomHeaders() {
        if (!this.customHeaders || Object.entries(this.customHeaders).length === 0) {
            return undefined;
        }
        return Object.entries(this.customHeaders).map(([headerName, headerValue]) => {
            return { headerName, headerValue };
        });
    }
    /**
     * If the path is defined, it must start with a '/' and not end with a '/'.
     * This method takes in the originPath, and returns it back (if undefined) or adds/removes the '/' as appropriate.
     */
    validateOriginPath(originPath) {
        if (core_1.Token.isUnresolved(originPath)) {
            return originPath;
        }
        if (originPath === undefined) {
            return undefined;
        }
        let path = originPath;
        if (!path.startsWith('/')) {
            path = '/' + path;
        }
        if (path.endsWith('/')) {
            path = path.substr(0, path.length - 1);
        }
        return path;
    }
    /**
     * Takes origin shield region and converts to CfnDistribution.OriginShieldProperty
     */
    renderOriginShield(originShieldRegion) {
        return originShieldRegion
            ? { enabled: true, originShieldRegion }
            : undefined;
    }
}
exports.OriginBase = OriginBase;
_a = JSII_RTTI_SYMBOL_1;
OriginBase[_a] = { fqn: "monocdk.aws_cloudfront.OriginBase", version: "1.149.0" };
/**
 * Throws an error if a value is defined and not an integer or not in a range.
 */
function validateIntInRangeOrUndefined(name, min, max, value, isDuration = true) {
    if (value === undefined) {
        return;
    }
    if (!Number.isInteger(value) || value < min || value > max) {
        const seconds = isDuration ? ' seconds' : '';
        throw new Error(`${name}: Must be an int between ${min} and ${max}${seconds} (inclusive); received ${value}.`);
    }
}
/**
 * Throws an error if custom header assignment is prohibited by CloudFront.
 * @link: https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/add-origin-custom-headers.html#add-origin-custom-headers-denylist
 */
function validateCustomHeaders(customHeaders) {
    if (!customHeaders || Object.entries(customHeaders).length === 0) {
        return;
    }
    const customHeaderKeys = Object.keys(customHeaders);
    const prohibitedHeaderKeys = [
        'Cache-Control', 'Connection', 'Content-Length', 'Cookie', 'Host', 'If-Match', 'If-Modified-Since', 'If-None-Match', 'If-Range', 'If-Unmodified-Since',
        'Max-Forwards', 'Pragma', 'Proxy-Authorization', 'Proxy-Connection', 'Range', 'Request-Range', 'TE', 'Trailer', 'Transfer-Encoding', 'Upgrade', 'Via',
        'X-Real-Ip',
    ];
    const prohibitedHeaderKeyPrefixes = [
        'X-Amz-', 'X-Edge-',
    ];
    const prohibitedHeadersKeysMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeys.map((prohibitedKey) => prohibitedKey.toLowerCase()).includes(customKey.toLowerCase());
    });
    const prohibitedHeaderPrefixMatches = customHeaderKeys.filter(customKey => {
        return prohibitedHeaderKeyPrefixes.some(prohibitedKeyPrefix => customKey.toLowerCase().startsWith(prohibitedKeyPrefix.toLowerCase()));
    });
    if (prohibitedHeadersKeysMatches.length !== 0) {
        throw new Error(`The following headers cannot be configured as custom origin headers: ${prohibitedHeadersKeysMatches.join(', ')}`);
    }
    if (prohibitedHeaderPrefixMatches.length !== 0) {
        throw new Error(`The following headers cannot be used as prefixes for custom origin headers: ${prohibitedHeaderPrefixMatches.join(', ')}`);
    }
}
//# sourceMappingURL=data:application/json;base64,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