"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CacheQueryStringBehavior = exports.CacheHeaderBehavior = exports.CacheCookieBehavior = exports.CachePolicy = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const cloudfront_generated_1 = require("./cloudfront.generated");
/**
 * A Cache Policy configuration.
 *
 * @resource AWS::CloudFront::CachePolicy
 * @link https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-managed-cache-policies.html
 */
class CachePolicy extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _e, _f, _g, _h;
        super(scope, id, {
            physicalName: props.cachePolicyName,
        });
        jsiiDeprecationWarnings.monocdk_aws_cloudfront_CachePolicyProps(props);
        const cachePolicyName = (_e = props.cachePolicyName) !== null && _e !== void 0 ? _e : `${core_1.Names.uniqueId(this).slice(0, 110)}-${core_1.Stack.of(this).region}`;
        if (!core_1.Token.isUnresolved(cachePolicyName) && !cachePolicyName.match(/^[\w-]+$/i)) {
            throw new Error(`'cachePolicyName' can only include '-', '_', and alphanumeric characters, got: '${cachePolicyName}'`);
        }
        if (cachePolicyName.length > 128) {
            throw new Error(`'cachePolicyName' cannot be longer than 128 characters, got: '${cachePolicyName.length}'`);
        }
        const minTtl = ((_f = props.minTtl) !== null && _f !== void 0 ? _f : core_1.Duration.seconds(0)).toSeconds();
        const defaultTtl = Math.max(((_g = props.defaultTtl) !== null && _g !== void 0 ? _g : core_1.Duration.days(1)).toSeconds(), minTtl);
        const maxTtl = Math.max(((_h = props.maxTtl) !== null && _h !== void 0 ? _h : core_1.Duration.days(365)).toSeconds(), defaultTtl);
        const resource = new cloudfront_generated_1.CfnCachePolicy(this, 'Resource', {
            cachePolicyConfig: {
                name: cachePolicyName,
                comment: props.comment,
                minTtl,
                maxTtl,
                defaultTtl,
                parametersInCacheKeyAndForwardedToOrigin: this.renderCacheKey(props),
            },
        });
        this.cachePolicyId = resource.ref;
    }
    /** Imports a Cache Policy from its id. */
    static fromCachePolicyId(scope, id, cachePolicyId) {
        return new class extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.cachePolicyId = cachePolicyId;
            }
        }(scope, id);
    }
    /** Use an existing managed cache policy. */
    static fromManagedCachePolicy(managedCachePolicyId) {
        return new class {
            constructor() {
                this.cachePolicyId = managedCachePolicyId;
            }
        }();
    }
    renderCacheKey(props) {
        var _e, _f, _g, _h, _j;
        const cookies = (_e = props.cookieBehavior) !== null && _e !== void 0 ? _e : CacheCookieBehavior.none();
        const headers = (_f = props.headerBehavior) !== null && _f !== void 0 ? _f : CacheHeaderBehavior.none();
        const queryStrings = (_g = props.queryStringBehavior) !== null && _g !== void 0 ? _g : CacheQueryStringBehavior.none();
        return {
            cookiesConfig: {
                cookieBehavior: cookies.behavior,
                cookies: cookies.cookies,
            },
            headersConfig: {
                headerBehavior: headers.behavior,
                headers: headers.headers,
            },
            enableAcceptEncodingGzip: (_h = props.enableAcceptEncodingGzip) !== null && _h !== void 0 ? _h : false,
            enableAcceptEncodingBrotli: (_j = props.enableAcceptEncodingBrotli) !== null && _j !== void 0 ? _j : false,
            queryStringsConfig: {
                queryStringBehavior: queryStrings.behavior,
                queryStrings: queryStrings.queryStrings,
            },
        };
    }
}
exports.CachePolicy = CachePolicy;
_a = JSII_RTTI_SYMBOL_1;
CachePolicy[_a] = { fqn: "monocdk.aws_cloudfront.CachePolicy", version: "1.149.0" };
/**
 * This policy is designed for use with an origin that is an AWS Amplify web app.
 */
CachePolicy.AMPLIFY = CachePolicy.fromManagedCachePolicy('2e54312d-136d-493c-8eb9-b001f22f67d2');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 */
CachePolicy.CACHING_OPTIMIZED = CachePolicy.fromManagedCachePolicy('658327ea-f89d-4fab-a63d-7e88639e58f6');
/**
 * Optimize cache efficiency by minimizing the values that CloudFront includes in the cache key.
 * Query strings and cookies are not included in the cache key, and only the normalized 'Accept-Encoding' header is included.
 * Disables cache compression.
 */
CachePolicy.CACHING_OPTIMIZED_FOR_UNCOMPRESSED_OBJECTS = CachePolicy.fromManagedCachePolicy('b2884449-e4de-46a7-ac36-70bc7f1ddd6d');
/** Disables caching. This policy is useful for dynamic content and for requests that are not cacheable. */
CachePolicy.CACHING_DISABLED = CachePolicy.fromManagedCachePolicy('4135ea2d-6df8-44a3-9df3-4b5a84be39ad');
/** Designed for use with an origin that is an AWS Elemental MediaPackage endpoint. */
CachePolicy.ELEMENTAL_MEDIA_PACKAGE = CachePolicy.fromManagedCachePolicy('08627262-05a9-4f76-9ded-b50ca2e3a84f');
/**
 * Determines whether any cookies in viewer requests are included in the cache key and
 * automatically included in requests that CloudFront sends to the origin.
 */
class CacheCookieBehavior {
    constructor(behavior, cookies) {
        this.behavior = behavior;
        this.cookies = cookies;
    }
    /**
     * Cookies in viewer requests are not included in the cache key and
     * are not automatically included in requests that CloudFront sends to the origin.
     */
    static none() { return new CacheCookieBehavior('none'); }
    /**
     * All cookies in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     */
    static all() { return new CacheCookieBehavior('all'); }
    /**
     * Only the provided `cookies` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     */
    static allowList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to allow must be provided');
        }
        return new CacheCookieBehavior('whitelist', cookies);
    }
    /**
     * All cookies except the provided `cookies` are included in the cache key and
     * automatically included in requests that CloudFront sends to the origin.
     */
    static denyList(...cookies) {
        if (cookies.length === 0) {
            throw new Error('At least one cookie to deny must be provided');
        }
        return new CacheCookieBehavior('allExcept', cookies);
    }
}
exports.CacheCookieBehavior = CacheCookieBehavior;
_b = JSII_RTTI_SYMBOL_1;
CacheCookieBehavior[_b] = { fqn: "monocdk.aws_cloudfront.CacheCookieBehavior", version: "1.149.0" };
/**
 * Determines whether any HTTP headers are included in the cache key and automatically included in requests that CloudFront sends to the origin.
 */
class CacheHeaderBehavior {
    constructor(behavior, headers) {
        this.behavior = behavior;
        this.headers = headers;
    }
    /** HTTP headers are not included in the cache key and are not automatically included in requests that CloudFront sends to the origin. */
    static none() { return new CacheHeaderBehavior('none'); }
    /** Listed headers are included in the cache key and are automatically included in requests that CloudFront sends to the origin. */
    static allowList(...headers) {
        if (headers.length === 0) {
            throw new Error('At least one header to allow must be provided');
        }
        return new CacheHeaderBehavior('whitelist', headers);
    }
}
exports.CacheHeaderBehavior = CacheHeaderBehavior;
_c = JSII_RTTI_SYMBOL_1;
CacheHeaderBehavior[_c] = { fqn: "monocdk.aws_cloudfront.CacheHeaderBehavior", version: "1.149.0" };
/**
 * Determines whether any URL query strings in viewer requests are included in the cache key
 * and automatically included in requests that CloudFront sends to the origin.
 */
class CacheQueryStringBehavior {
    constructor(behavior, queryStrings) {
        this.behavior = behavior;
        this.queryStrings = queryStrings;
    }
    /**
     * Query strings in viewer requests are not included in the cache key and
     * are not automatically included in requests that CloudFront sends to the origin.
     */
    static none() { return new CacheQueryStringBehavior('none'); }
    /**
     * All query strings in viewer requests are included in the cache key and are automatically included in requests that CloudFront sends to the origin.
     */
    static all() { return new CacheQueryStringBehavior('all'); }
    /**
     * Only the provided `queryStrings` are included in the cache key and automatically included in requests that CloudFront sends to the origin.
     */
    static allowList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to allow must be provided');
        }
        return new CacheQueryStringBehavior('whitelist', queryStrings);
    }
    /**
     * All query strings except the provided `queryStrings` are included in the cache key and
     * automatically included in requests that CloudFront sends to the origin.
     */
    static denyList(...queryStrings) {
        if (queryStrings.length === 0) {
            throw new Error('At least one query string to deny must be provided');
        }
        return new CacheQueryStringBehavior('allExcept', queryStrings);
    }
}
exports.CacheQueryStringBehavior = CacheQueryStringBehavior;
_d = JSII_RTTI_SYMBOL_1;
CacheQueryStringBehavior[_d] = { fqn: "monocdk.aws_cloudfront.CacheQueryStringBehavior", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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