"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupVault = exports.BackupVaultEvents = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
class BackupVaultBase extends core_1.Resource {
    /**
     * Grant the actions defined in actions to the given grantee
     * on this Backup Vault resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        for (const action of actions) {
            if (action.indexOf('*') >= 0) {
                throw new Error("AWS Backup access policies don't support a wildcard in the Action key.");
            }
        }
        return iam.Grant.addToPrincipal({
            grantee: grantee,
            actions: actions,
            resourceArns: [this.backupVaultArn],
        });
    }
}
/**
 * A backup vault
 */
class BackupVault extends BackupVaultBase {
    constructor(scope, id, props = {}) {
        var _b;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_backup_BackupVaultProps(props);
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_]{2,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-_]{2,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        this.accessPolicy = (_b = props.accessPolicy) !== null && _b !== void 0 ? _b : new iam.PolicyDocument();
        if (props.blockRecoveryPointDeletion) {
            this.blockRecoveryPointDeletion();
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: core_1.Lazy.any({ produce: () => this.accessPolicy.toJSON() }),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Import an existing backup vault by name
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        const backupVaultArn = core_1.Stack.of(scope).formatArn({
            service: 'backup',
            resource: 'backup-vault',
            resourceName: backupVaultName,
            arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
        });
        return BackupVault.fromBackupVaultArn(scope, id, backupVaultArn);
    }
    /**
     * Import an existing backup vault by arn
     */
    static fromBackupVaultArn(scope, id, backupVaultArn) {
        const parsedArn = core_1.Stack.of(scope).splitArn(backupVaultArn, core_1.ArnFormat.SLASH_RESOURCE_NAME);
        if (!parsedArn.resourceName) {
            throw new Error(`Backup Vault Arn ${backupVaultArn} does not have a resource name.`);
        }
        class Import extends BackupVaultBase {
            constructor() {
                super(...arguments);
                this.backupVaultName = parsedArn.resourceName;
                this.backupVaultArn = backupVaultArn;
            }
        }
        return new Import(scope, id, {
            account: parsedArn.account,
            region: parsedArn.region,
        });
    }
    /**
     * Adds a statement to the vault access policy
     */
    addToAccessPolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        this.accessPolicy.addStatements(statement);
    }
    /**
     * Adds a statement to the vault access policy that prevents anyone
     * from deleting a recovery point.
     */
    blockRecoveryPointDeletion() {
        this.addToAccessPolicy(new iam.PolicyStatement({
            effect: iam.Effect.DENY,
            actions: [
                'backup:DeleteRecoveryPoint',
                'backup:UpdateRecoveryPointLifecycle',
            ],
            principals: [new iam.AnyPrincipal()],
            resources: ['*'],
        }));
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = core_1.Names.uniqueId(this);
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
exports.BackupVault = BackupVault;
_a = JSII_RTTI_SYMBOL_1;
BackupVault[_a] = { fqn: "monocdk.aws_backup.BackupVault", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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