"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlanRule = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const events = require("../../aws-events");
const core_1 = require("../../core");
/**
 * A backup plan rule
 */
class BackupPlanRule {
    /** @param props Rule properties */
    constructor(props) {
        var _b, _c;
        jsiiDeprecationWarnings.monocdk_aws_backup_BackupPlanRuleProps(props);
        if (props.deleteAfter && props.moveToColdStorageAfter &&
            props.deleteAfter.toDays() < props.moveToColdStorageAfter.toDays()) {
            throw new Error('`deleteAfter` must be greater than `moveToColdStorageAfter`');
        }
        if (props.scheduleExpression && !/^cron/.test(props.scheduleExpression.expressionString)) {
            throw new Error('`scheduleExpression` must be of type `cron`');
        }
        const deleteAfter = (props.enableContinuousBackup && !props.deleteAfter) ? core_1.Duration.days(35) : props.deleteAfter;
        if (props.enableContinuousBackup && props.moveToColdStorageAfter) {
            throw new Error('`moveToColdStorageAfter` must not be specified if `enableContinuousBackup` is enabled');
        }
        if (props.enableContinuousBackup && props.deleteAfter &&
            (((_b = props.deleteAfter) === null || _b === void 0 ? void 0 : _b.toDays()) < 1 || ((_c = props.deleteAfter) === null || _c === void 0 ? void 0 : _c.toDays()) > 35)) {
            throw new Error(`'deleteAfter' must be between 1 and 35 days if 'enableContinuousBackup' is enabled, but got ${props.deleteAfter.toHumanString()}`);
        }
        this.props = {
            ...props,
            deleteAfter,
        };
    }
    /**
     * Daily with 35 days retention
     */
    static daily(backupVault) {
        jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Daily',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
            }),
            deleteAfter: core_1.Duration.days(35),
        });
    }
    /**
     * Weekly with 3 months retention
     */
    static weekly(backupVault) {
        jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Weekly',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
                weekDay: 'SAT',
            }),
            deleteAfter: core_1.Duration.days(30 * 3),
        });
    }
    /**
     * Monthly 1 year retention, move to cold storage after 1 month
     */
    static monthly1Year(backupVault) {
        jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly1Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30),
            deleteAfter: core_1.Duration.days(365),
        });
    }
    /**
     * Monthly 5 year retention, move to cold storage after 3 months
     */
    static monthly5Year(backupVault) {
        jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly5Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 5),
        });
    }
    /**
     * Monthly 7 year retention, move to cold storage after 3 months
     */
    static monthly7Year(backupVault) {
        jsiiDeprecationWarnings.monocdk_aws_backup_IBackupVault(backupVault);
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly7Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 7),
        });
    }
}
exports.BackupPlanRule = BackupPlanRule;
_a = JSII_RTTI_SYMBOL_1;
BackupPlanRule[_a] = { fqn: "monocdk.aws_backup.BackupPlanRule", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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