"use strict";
var _a, _b, _c, _d, _e, _f, _g;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.GroupMetric = exports.GroupMetrics = exports.UpdatePolicy = exports.Signals = exports.Monitoring = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("../../aws-ec2");
const elbv2 = require("../../aws-elasticloadbalancingv2");
const iam = require("../../aws-iam");
const core_1 = require("../../core");
const aspects_1 = require("./aspects");
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
/**
 * Configure whether the AutoScalingGroup waits for signals
 *
 * If you do configure waiting for signals, you should make sure the instances
 * invoke `cfn-signal` somewhere in their UserData to signal that they have
 * started up (either successfully or unsuccessfully).
 *
 * Signals are used both during intial creation and subsequent updates.
 */
class Signals {
    /**
     * Wait for the desiredCapacity of the AutoScalingGroup amount of signals to have been received
     *
     * If no desiredCapacity has been configured, wait for minCapacity signals intead.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForAll(options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                var _h;
                return this.doRender(options, (_h = renderOptions.desiredCapacity) !== null && _h !== void 0 ? _h : renderOptions.minCapacity);
            }
        }();
    }
    /**
     * Wait for the minCapacity of the AutoScalingGroup amount of signals to have been received
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForMinCapacity(options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy(renderOptions) {
                return this.doRender(options, renderOptions.minCapacity);
            }
        }();
    }
    /**
     * Wait for a specific amount of signals to have been received
     *
     * You should send one signal per instance, so this represents the number of
     * instances to wait for.
     *
     * This number is used during initial creation and during replacing updates.
     * During rolling updates, all updated instances must send a signal.
     */
    static waitForCount(count, options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        validatePercentage(options.minSuccessPercentage);
        return new class extends Signals {
            renderCreationPolicy() {
                return this.doRender(options, count);
            }
        }();
    }
    /**
     * Helper to render the actual creation policy, as the logic between them is quite similar
     */
    doRender(options, count) {
        var _h;
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_SignalsOptions(options);
        const minSuccessfulInstancesPercent = validatePercentage(options.minSuccessPercentage);
        return {
            ...options.minSuccessPercentage !== undefined ? { autoScalingCreationPolicy: { minSuccessfulInstancesPercent } } : {},
            resourceSignal: {
                count,
                timeout: (_h = options.timeout) === null || _h === void 0 ? void 0 : _h.toIsoString(),
            },
        };
    }
}
exports.Signals = Signals;
_a = JSII_RTTI_SYMBOL_1;
Signals[_a] = { fqn: "monocdk.aws_autoscaling.Signals", version: "1.149.0" };
/**
 * How existing instances should be updated
 */
class UpdatePolicy {
    /**
     * Create a new AutoScalingGroup and switch over to it
     */
    static replacingUpdate() {
        return new class extends UpdatePolicy {
            _renderUpdatePolicy() {
                return {
                    autoScalingReplacingUpdate: { willReplace: true },
                };
            }
        }();
    }
    /**
     * Replace the instances in the AutoScalingGroup one by one, or in batches
     */
    static rollingUpdate(options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_RollingUpdateOptions(options);
        const minSuccessPercentage = validatePercentage(options.minSuccessPercentage);
        return new class extends UpdatePolicy {
            _renderUpdatePolicy(renderOptions) {
                var _h, _j, _k, _l, _m, _o, _p, _q, _r;
                return {
                    autoScalingRollingUpdate: {
                        maxBatchSize: options.maxBatchSize,
                        minInstancesInService: options.minInstancesInService,
                        suspendProcesses: (_h = options.suspendProcesses) !== null && _h !== void 0 ? _h : DEFAULT_SUSPEND_PROCESSES,
                        minSuccessfulInstancesPercent: minSuccessPercentage !== null && minSuccessPercentage !== void 0 ? minSuccessPercentage : (_k = (_j = renderOptions.creationPolicy) === null || _j === void 0 ? void 0 : _j.autoScalingCreationPolicy) === null || _k === void 0 ? void 0 : _k.minSuccessfulInstancesPercent,
                        waitOnResourceSignals: ((_l = options.waitOnResourceSignals) !== null && _l !== void 0 ? _l : ((_m = renderOptions.creationPolicy) === null || _m === void 0 ? void 0 : _m.resourceSignal) !== undefined) ? true : undefined,
                        pauseTime: (_p = (_o = options.pauseTime) === null || _o === void 0 ? void 0 : _o.toIsoString()) !== null && _p !== void 0 ? _p : (_r = (_q = renderOptions.creationPolicy) === null || _q === void 0 ? void 0 : _q.resourceSignal) === null || _r === void 0 ? void 0 : _r.timeout,
                    },
                };
            }
        }();
    }
}
exports.UpdatePolicy = UpdatePolicy;
_b = JSII_RTTI_SYMBOL_1;
UpdatePolicy[_b] = { fqn: "monocdk.aws_autoscaling.UpdatePolicy", version: "1.149.0" };
/**
 * A set of group metrics
 */
class GroupMetrics {
    constructor(...metrics) {
        /**
         * @internal
         */
        this._metrics = new Set();
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_GroupMetric(metrics);
        metrics === null || metrics === void 0 ? void 0 : metrics.forEach(metric => this._metrics.add(metric));
    }
    /**
     * Report all group metrics.
     */
    static all() {
        return new GroupMetrics();
    }
}
exports.GroupMetrics = GroupMetrics;
_c = JSII_RTTI_SYMBOL_1;
GroupMetrics[_c] = { fqn: "monocdk.aws_autoscaling.GroupMetrics", version: "1.149.0" };
/**
 * Group metrics that an Auto Scaling group sends to Amazon CloudWatch.
 */
class GroupMetric {
    constructor(name) {
        this.name = name;
    }
}
exports.GroupMetric = GroupMetric;
_d = JSII_RTTI_SYMBOL_1;
GroupMetric[_d] = { fqn: "monocdk.aws_autoscaling.GroupMetric", version: "1.149.0" };
/**
 * The minimum size of the Auto Scaling group
 */
GroupMetric.MIN_SIZE = new GroupMetric('GroupMinSize');
/**
 * The maximum size of the Auto Scaling group
 */
GroupMetric.MAX_SIZE = new GroupMetric('GroupMaxSize');
/**
 * The number of instances that the Auto Scaling group attempts to maintain
 */
GroupMetric.DESIRED_CAPACITY = new GroupMetric('GroupDesiredCapacity');
/**
 * The number of instances that are running as part of the Auto Scaling group
 * This metric does not include instances that are pending or terminating
 */
GroupMetric.IN_SERVICE_INSTANCES = new GroupMetric('GroupInServiceInstances');
/**
 * The number of instances that are pending
 * A pending instance is not yet in service, this metric does not include instances that are in service or terminating
 */
GroupMetric.PENDING_INSTANCES = new GroupMetric('GroupPendingInstances');
/**
 * The number of instances that are in a Standby state
 * Instances in this state are still running but are not actively in service
 */
GroupMetric.STANDBY_INSTANCES = new GroupMetric('GroupStandbyInstances');
/**
 * The number of instances that are in the process of terminating
 * This metric does not include instances that are in service or pending
 */
GroupMetric.TERMINATING_INSTANCES = new GroupMetric('GroupTerminatingInstances');
/**
 * The total number of instances in the Auto Scaling group
 * This metric identifies the number of instances that are in service, pending, and terminating
 */
GroupMetric.TOTAL_INSTANCES = new GroupMetric('GroupTotalInstances');
class AutoScalingGroupBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
    }
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        if ((props.targetRequestsPerMinute === undefined) === (props.targetRequestsPerSecond === undefined)) {
            throw new Error('Specify exactly one of \'targetRequestsPerMinute\' or \'targetRequestsPerSecond\'');
        }
        let rpm;
        if (props.targetRequestsPerSecond !== undefined) {
            if (core_1.Token.isUnresolved(props.targetRequestsPerSecond)) {
                throw new Error('\'targetRequestsPerSecond\' cannot be an unresolved value; use \'targetRequestsPerMinute\' instead.');
            }
            rpm = props.targetRequestsPerSecond * 60;
        }
        else {
            rpm = props.targetRequestsPerMinute;
        }
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: rpm,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
    addUserData(..._commands) {
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _h, _j, _k, _l, _m;
        super(scope, id, {
            physicalName: props.autoScalingGroupName,
        });
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.groupMetrics = [];
        this.notifications = [];
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_AutoScalingGroupProps(props);
        this.newInstancesProtectedFromScaleIn = props.newInstancesProtectedFromScaleIn;
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        if (props.groupMetrics) {
            this.groupMetrics.push(...props.groupMetrics);
        }
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_h = props.userData) !== null && _h !== void 0 ? _h : imageConfig.userData;
        const userDataToken = core_1.Lazy.string({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.list({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = (_j = props.minCapacity) !== null && _j !== void 0 ? _j : 1;
        const maxCapacity = (_l = (_k = props.maxCapacity) !== null && _k !== void 0 ? _k : desiredCapacity) !== null && _l !== void 0 ? _l : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            core_1.Annotations.of(this).addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _h;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_h = nc.scalingEvents) !== null && _h !== void 0 ? _h : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            autoScalingGroupName: this.physicalName,
            cooldown: (_m = props.cooldown) === null || _m === void 0 ? void 0 : _m.toSeconds().toString(),
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.list({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.list({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            metricsCollection: core_1.Lazy.any({ produce: () => this.renderMetricsCollection() }),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
            newInstancesProtectedFromScaleIn: core_1.Lazy.any({ produce: () => this.newInstancesProtectedFromScaleIn }),
            terminationPolicies: props.terminationPolicies,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.getResourceNameAttribute(this.autoScalingGroup.ref),
            this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                service: 'autoscaling',
                resource: 'autoScalingGroup:*:autoScalingGroupName',
                resourceName: this.autoScalingGroupName,
            });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props, { desiredCapacity, minCapacity });
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.spotPrice = props.spotPrice;
        if (props.requireImdsv2) {
            core_1.Aspects.of(this).add(new aspects_1.AutoScalingGroupRequireImdsv2Aspect());
        }
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
                this.osType = ec2.OperatingSystemType.UNKNOWN;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        jsiiDeprecationWarnings.monocdk_aws_ec2_ISecurityGroup(securityGroup);
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancing_LoadBalancer(loadBalancer);
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_IApplicationTargetGroup(targetGroup);
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        jsiiDeprecationWarnings.monocdk_aws_elasticloadbalancingv2_INetworkTargetGroup(targetGroup);
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        jsiiDeprecationWarnings.monocdk_aws_iam_PolicyStatement(statement);
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the AutoScalingGroup resource.
     * - Add commands to the UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for `cfn-init` to finish
     *   before reporting success.
     */
    applyCloudFormationInit(init, options = {}) {
        var _h;
        jsiiDeprecationWarnings.monocdk_aws_ec2_CloudFormationInit(init);
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_ApplyCloudFormationInitOptions(options);
        if (!((_h = this.autoScalingGroup.cfnOptions.creationPolicy) === null || _h === void 0 ? void 0 : _h.resourceSignal)) {
            throw new Error('When applying CloudFormationInit, you must also configure signals by supplying \'signals\' at instantiation time.');
        }
        init.attach(this.autoScalingGroup, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
            includeRole: options.includeRole,
            includeUrl: options.includeUrl,
        });
    }
    /**
     * Ensures newly-launched instances are protected from scale-in.
     */
    protectNewInstancesFromScaleIn() {
        this.newInstancesProtectedFromScaleIn = true;
    }
    /**
     * Returns `true` if newly-launched instances are protected from scale-in.
     */
    areNewInstancesProtectedFromScaleIn() {
        return this.newInstancesProtectedFromScaleIn === true;
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props, signalOptions) {
        var _h;
        // Make sure people are not using the old and new properties together
        const oldProps = [
            'updateType',
            'rollingUpdateConfiguration',
            'resourceSignalCount',
            'resourceSignalTimeout',
            'replacingUpdateMinSuccessfulInstancesPercent',
        ];
        for (const prop of oldProps) {
            if ((props.signals || props.updatePolicy) && props[prop] !== undefined) {
                throw new Error(`Cannot set 'signals'/'updatePolicy' and '${prop}' together. Prefer 'signals'/'updatePolicy'`);
            }
        }
        // Reify updatePolicy to `rollingUpdate` default in case it is combined with `init`
        props = {
            ...props,
            updatePolicy: (_h = props.updatePolicy) !== null && _h !== void 0 ? _h : (props.init ? UpdatePolicy.rollingUpdate() : undefined),
        };
        if (props.signals || props.updatePolicy) {
            this.applyNewSignalUpdatePolicies(props, signalOptions);
        }
        else {
            this.applyLegacySignalUpdatePolicies(props);
        }
        // The following is technically part of the "update policy" but it's also a completely
        // separate aspect of rolling/replacing update, so it's just its own top-level property.
        // Default is 'true' because that's what you're most likely to want
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.signals && !props.init) {
            // To be able to send a signal using `cfn-init`, the execution role needs
            // `cloudformation:SignalResource`. Normally the binding of CfnInit would
            // grant that permissions and another one, but if the user wants to use
            // `signals` without `init`, add the permissions here.
            //
            // If they call `applyCloudFormationInit()` after construction, nothing bad
            // happens either, we'll just have a duplicate statement which doesn't hurt.
            this.addToRolePolicy(new iam.PolicyStatement({
                actions: ['cloudformation:SignalResource'],
                resources: [core_1.Aws.STACK_ID],
            }));
        }
    }
    /**
     * Use 'signals' and 'updatePolicy' to determine the creation and update policies
     */
    applyNewSignalUpdatePolicies(props, signalOptions) {
        var _h, _j;
        this.autoScalingGroup.cfnOptions.creationPolicy = (_h = props.signals) === null || _h === void 0 ? void 0 : _h.renderCreationPolicy(signalOptions);
        this.autoScalingGroup.cfnOptions.updatePolicy = (_j = props.updatePolicy) === null || _j === void 0 ? void 0 : _j._renderUpdatePolicy({
            creationPolicy: this.autoScalingGroup.cfnOptions.creationPolicy,
        });
    }
    applyLegacySignalUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toIsoString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
    renderMetricsCollection() {
        if (this.groupMetrics.length === 0) {
            return undefined;
        }
        return this.groupMetrics.map(group => {
            var _h;
            return ({
                granularity: '1Minute',
                metrics: ((_h = group._metrics) === null || _h === void 0 ? void 0 : _h.size) !== 0 ? [...group._metrics].map(m => m.name) : undefined,
            });
        });
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
_e = JSII_RTTI_SYMBOL_1;
AutoScalingGroup[_e] = { fqn: "monocdk.aws_autoscaling.AutoScalingGroup", version: "1.149.0" };
/**
 * The type of update to perform on instances in this AutoScalingGroup
 *
 * @deprecated Use UpdatePolicy instead
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launched
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custom group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_ScalingEvent(types);
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
_f = JSII_RTTI_SYMBOL_1;
ScalingEvents[_f] = { fqn: "monocdk.aws_autoscaling.ScalingEvents", version: "1.149.0" };
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
/**
 * Fleet termination launch events
 */
ScalingEvents.TERMINATION_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
// Recommended list of processes to suspend from here:
// https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
const DEFAULT_SUSPEND_PROCESSES = [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
    ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS];
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_Ec2HealthCheckOptions(options);
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        jsiiDeprecationWarnings.monocdk_aws_autoscaling_ElbHealthCheckOptions(options);
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
_g = JSII_RTTI_SYMBOL_1;
HealthCheck[_g] = { fqn: "monocdk.aws_autoscaling.HealthCheck", version: "1.149.0" };
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    var _h;
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toIsoString(),
        suspendProcesses: (_h = config.suspendProcesses) !== null && _h !== void 0 ? _h : DEFAULT_SUSPEND_PROCESSES,
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                core_1.Annotations.of(construct).addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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