"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findAlarmThresholds = exports.normalizeIntervals = void 0;
/**
 * Normalize the given interval set to cover the complete number line and make sure it has at most one gap
 */
function normalizeIntervals(intervals, changesAreAbsolute) {
    // Make intervals a complete numberline
    const full = orderAndCompleteIntervals(intervals);
    // Add 'undefined's in uncovered areas of the number line
    makeGapsUndefined(full);
    // In case of relative changes, treat 0-change also as 'undefined' (= no change action)
    if (!changesAreAbsolute) {
        makeZerosUndefined(full);
    }
    // Combine adjacent undefines and make sure there's at most one of them
    combineUndefineds(full);
    validateAtMostOneUndefined(full);
    return full;
}
exports.normalizeIntervals = normalizeIntervals;
/**
 * Completely order scaling intervals, making their lower and upper bounds concrete.
 */
function orderAndCompleteIntervals(intervals) {
    if (intervals.length < 2) {
        throw new Error('Require at least 2 intervals');
    }
    for (const interval of intervals) {
        if (interval.lower === undefined && interval.upper === undefined) {
            throw new Error(`Must supply at least one of 'upper' or 'lower', got: ${JSON.stringify(interval)}`);
        }
    }
    // Make a copy
    intervals = intervals.map(x => ({ ...x }));
    // Sort by whatever number we have for each interval
    intervals.sort(comparatorFromKey((x) => { var _a; return (_a = x.lower) !== null && _a !== void 0 ? _a : x.upper; }));
    // Propagate boundaries until no more change
    while (propagateBounds(intervals)) { /* Repeat */ }
    const lastIndex = intervals.length - 1;
    // Validate that no intervals have undefined bounds now, which must mean they're complete.
    if (intervals[0].lower === undefined) {
        intervals[0] = { ...intervals[0], lower: 0 };
    }
    if (intervals[lastIndex].upper === undefined) {
        intervals[lastIndex] = { ...intervals[lastIndex], upper: Infinity };
    }
    for (const interval of intervals) {
        if (interval.lower === undefined || interval.upper === undefined) {
            throw new Error(`Could not determine the lower and upper bounds for ${JSON.stringify(interval)}`);
        }
    }
    const completeIntervals = intervals;
    // Validate that we have nonoverlapping intervals now.
    for (let i = 0; i < completeIntervals.length - 1; i++) {
        if (overlap(completeIntervals[i], completeIntervals[i + 1])) {
            throw new Error(`Two intervals overlap: ${JSON.stringify(completeIntervals[i])} and ${JSON.stringify(completeIntervals[i + 1])}`);
        }
    }
    // Fill up the gaps
    return completeIntervals;
}
/**
 * Make the intervals cover the complete number line
 *
 * This entails adding intervals with an 'undefined' change to fill up the gaps.
 *
 * Since metrics have a halfopen interval, the first one will get a lower bound
 * of 0, the last one will get an upper bound of +Infinity.
 *
 * In case of absolute adjustments, the lower number of the adjacent bound will
 * be used, which means conservative change. In case of relative adjustments,
 * we'll use relative adjusment 0 (which means no change).
 */
function makeGapsUndefined(intervals) {
    // Add edge intervals if necessary, but only for relative adjustments. Since we're
    // going to make scaling intervals extend all the way out to infinity on either side,
    // the result is the same for absolute adjustments anyway.
    if (intervals[0].lower !== 0) {
        intervals.splice(0, 0, {
            lower: 0,
            upper: intervals[0].lower,
            change: undefined,
        });
    }
    if (last(intervals).upper !== Infinity) {
        intervals.push({
            lower: last(intervals).upper,
            upper: Infinity,
            change: undefined,
        });
    }
    let i = 1;
    while (i < intervals.length) {
        if (intervals[i - 1].upper < intervals[i].lower) {
            intervals.splice(i, 0, {
                lower: intervals[i - 1].upper,
                upper: intervals[i].lower,
                change: undefined,
            });
        }
        else {
            i++;
        }
    }
}
/**
 * Turn zero changes into undefined, in-place
 */
function makeZerosUndefined(intervals) {
    for (let i = 0; i < intervals.length; ++i) {
        const interval = intervals[i];
        if (interval.change === 0) {
            intervals[i] = { ...interval, change: undefined };
        }
    }
}
/**
 * If there are adjacent "undefined" intervals, combine them
 */
function combineUndefineds(intervals) {
    let i = 0;
    while (i < intervals.length - 1) {
        if (intervals[i].change === undefined && intervals[i + 1].change === undefined) {
            intervals[i] = { ...intervals[i], upper: intervals[i + 1].upper };
            intervals.splice(i + 1, 1);
        }
        else {
            i++;
        }
    }
}
function validateAtMostOneUndefined(intervals) {
    const undef = intervals.filter(x => x.change === undefined);
    if (undef.length > 1) {
        throw new Error(`Can have at most one no-change interval, got ${JSON.stringify(undef)}`);
    }
}
function comparatorFromKey(keyFn) {
    return (a, b) => {
        const keyA = keyFn(a);
        const keyB = keyFn(b);
        if (keyA < keyB) {
            return -1;
        }
        if (keyA === keyB) {
            return 0;
        }
        return 1;
    };
}
function propagateBounds(intervals) {
    let ret = false;
    // Propagate upper bounds upwards
    for (let i = 0; i < intervals.length - 1; i++) {
        if (intervals[i].upper !== undefined && intervals[i + 1].lower === undefined) {
            intervals[i + 1] = { ...intervals[i + 1], lower: intervals[i].upper };
            ret = true;
        }
    }
    // Propagate lower bounds downwards
    for (let i = intervals.length - 1; i >= 1; i--) {
        if (intervals[i].lower !== undefined && intervals[i - 1].upper === undefined) {
            intervals[i - 1] = { ...intervals[i - 1], upper: intervals[i].lower };
            ret = true;
        }
    }
    return ret;
}
/**
 * Whether two intervals overlap
 */
function overlap(a, b) {
    return a.lower < b.upper && a.upper > b.lower;
}
function last(xs) {
    return xs[xs.length - 1];
}
/**
 * Locate the intervals that should have the alarm thresholds, by index.
 *
 * Pick the intervals on either side of the singleton "undefined" interval, or
 * pick the middle interval if there's no such interval.
 */
function findAlarmThresholds(intervals) {
    const gapIndex = intervals.findIndex(x => x.change === undefined);
    if (gapIndex !== -1) {
        return {
            lowerAlarmIntervalIndex: gapIndex > 0 ? gapIndex - 1 : undefined,
            upperAlarmIntervalIndex: gapIndex < intervals.length - 1 ? gapIndex + 1 : undefined,
        };
    }
    if (intervals.length === 1) {
        return { upperAlarmIntervalIndex: 0 };
    }
    const middleIndex = Math.floor(intervals.length / 2);
    return {
        lowerAlarmIntervalIndex: middleIndex - 1,
        upperAlarmIntervalIndex: middleIndex,
    };
}
exports.findAlarmThresholds = findAlarmThresholds;
//# sourceMappingURL=data:application/json;base64,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