"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const core_1 = require("../../core");
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * The Schema for a GraphQL Api
 *
 * If no options are configured, schema will be generated
 * code-first.
 */
class Schema {
    constructor(options) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_SchemaOptions(options);
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
        this.types = [];
    }
    /**
     * Generate a Schema from file
     *
     * @returns `SchemaAsset` with immutable schema defintion
     * @param filePath the file path of the schema file
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * Called when the GraphQL Api is initialized to allow this object to bind
     * to the stack.
     *
     * @param api The binding GraphQL Api
     */
    bind(api) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_GraphqlApi(api);
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: this.mode === private_1.SchemaMode.CODE ?
                    core_1.Lazy.string({
                        produce: () => this.types.reduce((acc, type) => `${acc}${type._bindToGraphqlApi(api).toString()}\n`, `${this.declareSchema()}${this.definition}`),
                    })
                    : this.definition,
            });
        }
        return this.schema;
    }
    /**
     * Escape hatch to add to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_ResolvableField(field);
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_ResolvableField(field);
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        var _b, _c;
        jsiiDeprecationWarnings.monocdk_aws_appsync_Field(field);
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add subscription. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.subscription) {
            this.subscription = new schema_intermediate_1.ObjectType('Subscription', { definition: {} });
            this.addType(this.subscription);
        }
        const directives = (_c = (_b = field.fieldOptions) === null || _b === void 0 ? void 0 : _b.directives) === null || _c === void 0 ? void 0 : _c.filter((directive) => directive.mutationFields);
        if (directives && directives.length > 1) {
            throw new Error(`Subscription fields must not have more than one @aws_subscribe directives. Received: ${directives.length}`);
        }
        this.subscription.addField({ fieldName, field });
        return this.subscription;
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_IIntermediateType(type);
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.types.push(type);
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => { var _b; return this[key] ? `${key}: ${(_b = this[key]) === null || _b === void 0 ? void 0 : _b.name}` : ''; })
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
_a = JSII_RTTI_SYMBOL_1;
Schema[_a] = { fqn: "monocdk.aws_appsync.Schema", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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