"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Type = exports.Directive = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const graphqlapi_1 = require("./graphqlapi");
/**
 * Directives for types
 *
 * i.e. @aws_iam or @aws_subscribe
 *
 */
class Directive {
    constructor(statement, options) {
        this.statement = statement;
        this.mode = options === null || options === void 0 ? void 0 : options.mode;
        this.mutationFields = options === null || options === void 0 ? void 0 : options.mutationFields;
    }
    /**
     * Add the @aws_iam directive
     */
    static iam() {
        return new Directive('@aws_iam', { mode: graphqlapi_1.AuthorizationType.IAM });
    }
    /**
     * Add the @aws_oidc directive
     */
    static oidc() {
        return new Directive('@aws_oidc', { mode: graphqlapi_1.AuthorizationType.OIDC });
    }
    /**
     * Add the @aws_api_key directive
     */
    static apiKey() {
        return new Directive('@aws_api_key', { mode: graphqlapi_1.AuthorizationType.API_KEY });
    }
    /**
     * Add the @aws_auth or @aws_cognito_user_pools directive
     *
     * @param groups the groups to allow access to
     */
    static cognito(...groups) {
        if (groups.length === 0) {
            throw new Error(`Cognito authorization requires at least one Cognito group to be supplied. Received: ${groups.length}`);
        }
        // this function creates the cognito groups as a string (i.e. ["group1", "group2", "group3"])
        const stringify = (array) => {
            return array.reduce((acc, element) => `${acc}"${element}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_auth(cognito_groups: [${stringify(groups)}])`, { mode: graphqlapi_1.AuthorizationType.USER_POOL });
    }
    /**
     * Add the @aws_subscribe directive. Only use for top level Subscription type.
     *
     * @param mutations the mutation fields to link to
     */
    static subscribe(...mutations) {
        if (mutations.length === 0) {
            throw new Error(`Subscribe directive requires at least one mutation field to be supplied. Received: ${mutations.length}`);
        }
        // this function creates the subscribe directive as a string (i.e. ["mutation_field_1", "mutation_field_2"])
        const stringify = (array) => {
            return array.reduce((acc, mutation) => `${acc}"${mutation}", `, '').slice(0, -2);
        };
        return new Directive(`@aws_subscribe(mutations: [${stringify(mutations)}])`, { mutationFields: mutations });
    }
    /**
     * Add a custom directive
     *
     * @param statement - the directive statement to append
     */
    static custom(statement) {
        return new Directive(statement);
    }
    /**
     * Method called when the stringifying Directive for schema generation
     *
     * @param modes the authorization modes
     *
     * @internal
     */
    _bindToAuthModes(modes) {
        this.modes = modes;
        return this;
    }
    /**
     * Generate the directive statement
     */
    toString() {
        if (this.modes && this.mode && !this.modes.some((mode) => mode === this.mode)) {
            throw new Error(`No Authorization Type ${this.mode} declared in GraphQL Api.`);
        }
        if (this.mode === graphqlapi_1.AuthorizationType.USER_POOL && this.modes && this.modes.length > 1) {
            this.statement = this.statement.replace('@aws_auth', '@aws_cognito_user_pools');
        }
        return this.statement;
    }
}
exports.Directive = Directive;
_a = JSII_RTTI_SYMBOL_1;
Directive[_a] = { fqn: "monocdk.aws_appsync.Directive", version: "1.149.0" };
/**
 * Enum containing the Types that can be used to define ObjectTypes
 */
var Type;
(function (Type) {
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     */
    Type["ID"] = "ID";
    /**
     * `String` scalar type is a free-form human-readable text.
     */
    Type["STRING"] = "String";
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     */
    Type["INT"] = "Int";
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     */
    Type["FLOAT"] = "Float";
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     */
    Type["BOOLEAN"] = "Boolean";
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Calendar_dates
     */
    Type["AWS_DATE"] = "AWSDate";
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Times
     */
    Type["AWS_TIME"] = "AWSTime";
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @see https://en.wikipedia.org/wiki/ISO_8601#Combined_date_and_time_representations
     */
    Type["AWS_DATE_TIME"] = "AWSDateTime";
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     */
    Type["AWS_TIMESTAMP"] = "AWSTimestamp";
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     */
    Type["AWS_EMAIL"] = "AWSEmail";
    /**
     * `AWSJson` scalar type represents a JSON string.
     */
    Type["AWS_JSON"] = "AWSJSON";
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     */
    Type["AWS_URL"] = "AWSURL";
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     */
    Type["AWS_PHONE"] = "AWSPhone";
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     */
    Type["AWS_IP_ADDRESS"] = "AWSIPAddress";
    /**
     * Type used for Intermediate Types
     * (i.e. an interface or an object type)
     */
    Type["INTERMEDIATE"] = "INTERMEDIATE";
})(Type = exports.Type || (exports.Type = {}));
//# sourceMappingURL=data:application/json;base64,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