"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("../../aws-iam");
const core_1 = require("../../core");
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
    /**
     * Lambda authorization type
     */
    AuthorizationType["LAMBDA"] = "AWS_LAMBDA";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_GraphqlApi(api);
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
_a = JSII_RTTI_SYMBOL_1;
IamResource[_a] = { fqn: "monocdk.aws_appsync.IamResource", version: "1.149.0" };
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_appsync_GraphqlApiProps(props);
        const defaultMode = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.defaultAuthorization) !== null && _d !== void 0 ? _d : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_f = (_e = props.authorizationConfig) === null || _e === void 0 ? void 0 : _e.additionalAuthorizationModes) !== null && _f !== void 0 ? _f : [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(defaultMode.lambdaAuthorizerConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_g = props.schema) !== null && _g !== void 0 ? _g : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_h = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _h === void 0 ? void 0 : _h.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _c;
        jsiiDeprecationWarnings.monocdk_aws_appsync_GraphqlApiAttributes(attrs);
        const arn = (_c = attrs.graphqlApiArn) !== null && _c !== void 0 ? _c : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        jsiiDeprecationWarnings.monocdk_aws_appsync_IamResource(resources);
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.LAMBDA).length > 1) {
            throw new Error('You can only have a single AWS Lambda function configured to authorize your API.');
        }
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing User Pool Configuration');
            }
            if (mode.authorizationType === AuthorizationType.LAMBDA && !mode.lambdaAuthorizerConfig) {
                throw new Error('Missing Lambda Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        jsiiDeprecationWarnings.monocdk_CfnResource(construct);
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        var _c, _d;
        if (!config)
            return undefined;
        const logsRoleArn = (_d = (_c = config.role) === null || _c === void 0 ? void 0 : _c.roleArn) !== null && _d !== void 0 ? _d : new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        }).roleArn;
        return {
            cloudWatchLogsRoleArn: logsRoleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupLambdaAuthorizerConfig(config) {
        var _c;
        if (!config)
            return undefined;
        return {
            authorizerResultTtlInSeconds: (_c = config.resultsCacheTtl) === null || _c === void 0 ? void 0 : _c.toSeconds(),
            authorizerUri: config.handler.functionArn,
            identityValidationExpression: config.validationRegex,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
                lambdaAuthorizerConfig: this.setupLambdaAuthorizerConfig(mode.lambdaAuthorizerConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        var _c, _d;
        if (((_c = config === null || config === void 0 ? void 0 : config.expires) === null || _c === void 0 ? void 0 : _c.isBefore(core_1.Duration.days(1))) || ((_d = config === null || config === void 0 ? void 0 : config.expires) === null || _d === void 0 ? void 0 : _d.isAfter(core_1.Duration.days(365)))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = (config === null || config === void 0 ? void 0 : config.expires) ? config === null || config === void 0 ? void 0 : config.expires.toEpoch() : undefined;
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     */
    addType(type) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_IIntermediateType(type);
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_ResolvableField(field);
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_ResolvableField(field);
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_ResolvableField(field);
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
_b = JSII_RTTI_SYMBOL_1;
GraphqlApi[_b] = { fqn: "monocdk.aws_appsync.GraphqlApi", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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