"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApiBase = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const data_source_1 = require("./data-source");
const resolver_1 = require("./resolver");
/**
 * Base Class for GraphQL API
 */
class GraphqlApiBase extends core_1.Resource {
    /**
     * add a new dummy data source to this API. Useful for pipeline resolvers
     * and for backend changes that don't require a data source.
     *
     * @param id The data source's id
     * @param options The optional configuration for this data source
     */
    addNoneDataSource(id, options) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_DataSourceOptions(options);
        return new data_source_1.NoneDataSource(this, id, {
            api: this,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     *
     * @param id The data source's id
     * @param table The DynamoDB table backing this data source
     * @param options The optional configuration for this data source
     */
    addDynamoDbDataSource(id, table, options) {
        jsiiDeprecationWarnings.monocdk_aws_dynamodb_ITable(table);
        jsiiDeprecationWarnings.monocdk_aws_appsync_DataSourceOptions(options);
        return new data_source_1.DynamoDbDataSource(this, id, {
            api: this,
            table,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
        });
    }
    /**
     * add a new http data source to this API
     *
     * @param id The data source's id
     * @param endpoint The http endpoint
     * @param options The optional configuration for this data source
     */
    addHttpDataSource(id, endpoint, options) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_HttpDataSourceOptions(options);
        return new data_source_1.HttpDataSource(this, id, {
            api: this,
            endpoint,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
            authorizationConfig: options === null || options === void 0 ? void 0 : options.authorizationConfig,
        });
    }
    /**
     * add a new Lambda data source to this API
     *
     * @param id The data source's id
     * @param lambdaFunction The Lambda function to call to interact with this data source
     * @param options The optional configuration for this data source
     */
    addLambdaDataSource(id, lambdaFunction, options) {
        jsiiDeprecationWarnings.monocdk_aws_lambda_IFunction(lambdaFunction);
        jsiiDeprecationWarnings.monocdk_aws_appsync_DataSourceOptions(options);
        return new data_source_1.LambdaDataSource(this, id, {
            api: this,
            lambdaFunction,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
        });
    }
    /**
     * add a new Rds data source to this API
     * @param id The data source's id
     * @param serverlessCluster The serverless cluster to interact with this data source
     * @param secretStore The secret store that contains the username and password for the serverless cluster
     * @param databaseName The optional name of the database to use within the cluster
     * @param options The optional configuration for this data source
     */
    addRdsDataSource(id, serverlessCluster, secretStore, databaseName, options) {
        jsiiDeprecationWarnings.monocdk_aws_rds_IServerlessCluster(serverlessCluster);
        jsiiDeprecationWarnings.monocdk_aws_secretsmanager_ISecret(secretStore);
        jsiiDeprecationWarnings.monocdk_aws_appsync_DataSourceOptions(options);
        return new data_source_1.RdsDataSource(this, id, {
            api: this,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
            serverlessCluster,
            secretStore,
            databaseName,
        });
    }
    /**
     * add a new elasticsearch data source to this API
     *
     * @deprecated - use `addOpenSearchDataSource`
     * @param id The data source's id
     * @param domain The elasticsearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addElasticsearchDataSource(id, domain, options) {
        jsiiDeprecationWarnings.print("monocdk.aws_appsync.GraphqlApiBase#addElasticsearchDataSource", "- use `addOpenSearchDataSource`");
        jsiiDeprecationWarnings.monocdk_aws_elasticsearch_IDomain(domain);
        jsiiDeprecationWarnings.monocdk_aws_appsync_DataSourceOptions(options);
        return new data_source_1.ElasticsearchDataSource(this, id, {
            api: this,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
            domain,
        });
    }
    /**
     * add a new OpenSearch data source to this API
     *
     * @param id The data source's id
     * @param domain The OpenSearch domain for this data source
     * @param options The optional configuration for this data source
     */
    addOpenSearchDataSource(id, domain, options) {
        jsiiDeprecationWarnings.monocdk_aws_opensearchservice_IDomain(domain);
        jsiiDeprecationWarnings.monocdk_aws_appsync_DataSourceOptions(options);
        return new data_source_1.OpenSearchDataSource(this, id, {
            api: this,
            name: options === null || options === void 0 ? void 0 : options.name,
            description: options === null || options === void 0 ? void 0 : options.description,
            domain,
        });
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        jsiiDeprecationWarnings.monocdk_aws_appsync_ExtendedResolverProps(props);
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this,
            ...props,
        });
    }
    /**
     * Add schema dependency if not imported
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        jsiiDeprecationWarnings.monocdk_CfnResource(construct);
        construct;
        return false;
    }
}
exports.GraphqlApiBase = GraphqlApiBase;
_a = JSII_RTTI_SYMBOL_1;
GraphqlApiBase[_a] = { fqn: "monocdk.aws_appsync.GraphqlApiBase", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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