"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../aws-iam");
const cdk = require("../../core");
const appmesh_generated_1 = require("./appmesh.generated");
const mesh_1 = require("./mesh");
const utils_1 = require("./private/utils");
class VirtualNodeBase extends cdk.Resource {
    grantStreamAggregatedResources(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['appmesh:StreamAggregatedResources'],
            resourceArns: [this.virtualNodeArn],
        });
    }
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.string({ produce: () => cdk.Names.uniqueId(this) }),
        });
        this.backends = new Array();
        this.listeners = new Array();
        jsiiDeprecationWarnings.monocdk_aws_appmesh_VirtualNodeProps(props);
        this.mesh = props.mesh;
        this.serviceDiscoveryConfig = (_b = props.serviceDiscovery) === null || _b === void 0 ? void 0 : _b.bind(this);
        (_c = props.backends) === null || _c === void 0 ? void 0 : _c.forEach(backend => this.addBackend(backend));
        (_d = props.listeners) === null || _d === void 0 ? void 0 : _d.forEach(listener => this.addListener(listener));
        const accessLogging = (_e = props.accessLog) === null || _e === void 0 ? void 0 : _e.bind(this);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            meshOwner: utils_1.renderMeshOwner(this.env.account, this.mesh.env.account),
            spec: {
                backends: cdk.Lazy.any({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.any({ produce: () => this.listeners.map(listener => listener.listener) }, { omitEmptyArray: true }),
                backendDefaults: props.backendDefaults !== undefined
                    ? {
                        clientPolicy: {
                            tls: utils_1.renderTlsClientPolicy(this, (_f = props.backendDefaults) === null || _f === void 0 ? void 0 : _f.tlsClientPolicy),
                        },
                    }
                    : undefined,
                serviceDiscovery: renderServiceDiscovery(this.serviceDiscoveryConfig),
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualNodeAccessLog,
                } : undefined,
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new class extends VirtualNodeBase {
            constructor() {
                super(...arguments);
                this.virtualNodeArn = virtualNodeArn;
                this.parsedArn = cdk.Fn.split('/', cdk.Stack.of(scope).splitArn(virtualNodeArn, cdk.ArnFormat.SLASH_RESOURCE_NAME).resourceName);
                this.mesh = mesh_1.Mesh.fromMeshName(this, 'Mesh', cdk.Fn.select(0, this.parsedArn));
                this.virtualNodeName = cdk.Fn.select(2, this.parsedArn);
            }
        }(scope, id);
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeAttributes(scope, id, attrs) {
        jsiiDeprecationWarnings.monocdk_aws_appmesh_VirtualNodeAttributes(attrs);
        return new class extends VirtualNodeBase {
            constructor() {
                super(...arguments);
                this.mesh = attrs.mesh;
                this.virtualNodeName = attrs.virtualNodeName;
                this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                    service: 'appmesh',
                    resource: `mesh/${attrs.mesh.meshName}/virtualNode`,
                    resourceName: this.virtualNodeName,
                });
            }
        }(scope, id);
    }
    /**
     * Utility method to add an inbound listener for this VirtualNode
     *
     * Note: At this time, Virtual Nodes support at most one listener. Adding
     * more than one will result in a failure to deploy the CloudFormation stack.
     * However, the App Mesh team has plans to add support for multiple listeners
     * on Virtual Nodes and Virtual Routers.
     *
     * @see https://github.com/aws/aws-app-mesh-roadmap/issues/120
     */
    addListener(listener) {
        jsiiDeprecationWarnings.monocdk_aws_appmesh_VirtualNodeListener(listener);
        if (!this.serviceDiscoveryConfig) {
            throw new Error('Service discovery information is required for a VirtualNode with a listener.');
        }
        this.listeners.push(listener.bind(this));
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackend(backend) {
        jsiiDeprecationWarnings.monocdk_aws_appmesh_Backend(backend);
        this.backends.push(backend.bind(this).virtualServiceBackend);
    }
}
exports.VirtualNode = VirtualNode;
_a = JSII_RTTI_SYMBOL_1;
VirtualNode[_a] = { fqn: "monocdk.aws_appmesh.VirtualNode", version: "1.149.0" };
function renderServiceDiscovery(config) {
    return config
        ? {
            dns: config === null || config === void 0 ? void 0 : config.dns,
            awsCloudMap: config === null || config === void 0 ? void 0 : config.cloudmap,
        }
        : undefined;
}
//# sourceMappingURL=data:application/json;base64,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