import * as cloudmap from '../../aws-servicediscovery';
import { CfnVirtualNode } from './appmesh.generated';
import { Construct } from '../../core';
/**
 * Properties for VirtualNode Service Discovery
 */
export interface ServiceDiscoveryConfig {
    /**
     * DNS based Service Discovery
     *
     * @default - no DNS based service discovery
     */
    readonly dns?: CfnVirtualNode.DnsServiceDiscoveryProperty;
    /**
     * Cloud Map based Service Discovery
     *
     * @default - no Cloud Map based service discovery
     */
    readonly cloudmap?: CfnVirtualNode.AwsCloudMapServiceDiscoveryProperty;
}
/**
 * Enum of DNS service discovery response type
 */
export declare enum DnsResponseType {
    /**
     * DNS resolver returns a loadbalanced set of endpoints and the traffic would be sent to the given endpoints.
     * It would not drain existing connections to other endpoints that are not part of this list.
     */
    LOAD_BALANCER = "LOADBALANCER",
    /**
     * DNS resolver is returning all the endpoints.
     * This also means that if an endpoint is missing, it would drain the current connections to the missing endpoint.
     */
    ENDPOINTS = "ENDPOINTS"
}
/**
 * Provides the Service Discovery method a VirtualNode uses
 */
export declare abstract class ServiceDiscovery {
    /**
     * Returns DNS based service discovery
     *
     * @param hostname
     * @param responseType Specifies the DNS response type for the virtual node.
     *  The default is `DnsResponseType.LOAD_BALANCER`.
     */
    static dns(hostname: string, responseType?: DnsResponseType): ServiceDiscovery;
    /**
     * Returns Cloud Map based service discovery
     *
     * @param service The AWS Cloud Map Service to use for service discovery
     * @param instanceAttributes A string map that contains attributes with values that you can use to
     *  filter instances by any custom attribute that you specified when you
     *  registered the instance. Only instances that match all of the specified
     *  key/value pairs will be returned.
     */
    static cloudMap(service: cloudmap.IService, instanceAttributes?: {
        [key: string]: string;
    }): ServiceDiscovery;
    /**
     * Binds the current object when adding Service Discovery to a VirtualNode
     */
    abstract bind(scope: Construct): ServiceDiscoveryConfig;
}
