"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpRoute = exports.HttpRouteKey = exports.HttpMethod = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("../../../aws-iam");
const core_1 = require("../../../core");
const constructs_1 = require("constructs");
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
/**
 * Supported HTTP methods
 */
var HttpMethod;
(function (HttpMethod) {
    /** HTTP ANY */
    HttpMethod["ANY"] = "ANY";
    /** HTTP DELETE */
    HttpMethod["DELETE"] = "DELETE";
    /** HTTP GET */
    HttpMethod["GET"] = "GET";
    /** HTTP HEAD */
    HttpMethod["HEAD"] = "HEAD";
    /** HTTP OPTIONS */
    HttpMethod["OPTIONS"] = "OPTIONS";
    /** HTTP PATCH */
    HttpMethod["PATCH"] = "PATCH";
    /** HTTP POST */
    HttpMethod["POST"] = "POST";
    /** HTTP PUT */
    HttpMethod["PUT"] = "PUT";
})(HttpMethod = exports.HttpMethod || (exports.HttpMethod = {}));
/**
 * HTTP route in APIGateway is a combination of the HTTP method and the path component.
 * This class models that combination.
 */
class HttpRouteKey {
    constructor(method, path) {
        this.method = method !== null && method !== void 0 ? method : HttpMethod.ANY;
        this.path = path;
        this.key = path ? `${method} ${path}` : '$default';
    }
    /**
     * Create a route key with the combination of the path and the method.
     * @param method default is 'ANY'
     */
    static with(path, method) {
        jsiiDeprecationWarnings.monocdk_aws_apigatewayv2_HttpMethod(method);
        if (path !== '/' && (!path.startsWith('/') || path.endsWith('/'))) {
            throw new Error('A route path must always start with a "/" and not end with a "/"');
        }
        return new HttpRouteKey(method, path);
    }
}
exports.HttpRouteKey = HttpRouteKey;
_a = JSII_RTTI_SYMBOL_1;
HttpRouteKey[_a] = { fqn: "monocdk.aws_apigatewayv2.HttpRouteKey", version: "1.149.0" };
/**
 * The catch-all route of the API, i.e., when no other routes match
 */
HttpRouteKey.DEFAULT = new HttpRouteKey();
/**
 * Supported Route Authorizer types
 */
var HttpRouteAuthorizationType;
(function (HttpRouteAuthorizationType) {
    /** AWS IAM */
    HttpRouteAuthorizationType["AWS_IAM"] = "AWS_IAM";
    /** JSON Web Tokens */
    HttpRouteAuthorizationType["JWT"] = "JWT";
    /** Lambda Authorizer */
    HttpRouteAuthorizationType["CUSTOM"] = "CUSTOM";
    /** No authorizer */
    HttpRouteAuthorizationType["NONE"] = "NONE";
})(HttpRouteAuthorizationType || (HttpRouteAuthorizationType = {}));
/**
 * Route class that creates the Route for API Gateway HTTP API
 * @resource AWS::ApiGatewayV2::Route
 */
class HttpRoute extends core_1.Resource {
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g;
        super(scope, id);
        jsiiDeprecationWarnings.monocdk_aws_apigatewayv2_HttpRouteProps(props);
        this.httpApi = props.httpApi;
        this.path = props.routeKey.path;
        this.method = props.routeKey.method;
        this.routeArn = this.produceRouteArn(props.routeKey.method);
        const config = props.integration._bindToRoute({
            route: this,
            scope: this,
        });
        this.authBindResult = (_c = props.authorizer) === null || _c === void 0 ? void 0 : _c.bind({
            route: this,
            scope: this.httpApi instanceof constructs_1.Construct ? this.httpApi : this,
        });
        if (this.authBindResult && !(this.authBindResult.authorizationType in HttpRouteAuthorizationType)) {
            throw new Error(`authorizationType should either be AWS_IAM, JWT, CUSTOM, or NONE but was '${this.authBindResult.authorizationType}'`);
        }
        let authorizationScopes = (_d = this.authBindResult) === null || _d === void 0 ? void 0 : _d.authorizationScopes;
        if (this.authBindResult && props.authorizationScopes) {
            authorizationScopes = Array.from(new Set([
                ...authorizationScopes !== null && authorizationScopes !== void 0 ? authorizationScopes : [],
                ...props.authorizationScopes,
            ]));
        }
        if ((authorizationScopes === null || authorizationScopes === void 0 ? void 0 : authorizationScopes.length) === 0) {
            authorizationScopes = undefined;
        }
        const routeProps = {
            apiId: props.httpApi.apiId,
            routeKey: props.routeKey.key,
            target: `integrations/${config.integrationId}`,
            authorizerId: (_e = this.authBindResult) === null || _e === void 0 ? void 0 : _e.authorizerId,
            authorizationType: (_g = (_f = this.authBindResult) === null || _f === void 0 ? void 0 : _f.authorizationType) !== null && _g !== void 0 ? _g : 'NONE',
            authorizationScopes,
        };
        const route = new apigatewayv2_generated_1.CfnRoute(this, 'Resource', routeProps);
        this.routeId = route.ref;
    }
    produceRouteArn(httpMethod) {
        var _c;
        const stage = '*';
        const iamHttpMethod = httpMethod === HttpMethod.ANY ? '*' : httpMethod;
        const path = (_c = this.path) !== null && _c !== void 0 ? _c : '/';
        // When the user has provided a path with path variables, we replace the
        // path variable and all that follows with a wildcard.
        const iamPath = path.replace(/\{.*?\}.*/, '*');
        return `arn:aws:execute-api:${this.stack.region}:${this.stack.account}:${this.httpApi.apiId}/${stage}/${iamHttpMethod}${iamPath}`;
    }
    grantInvoke(grantee, options = {}) {
        var _c;
        jsiiDeprecationWarnings.monocdk_aws_iam_IGrantable(grantee);
        jsiiDeprecationWarnings.monocdk_aws_apigatewayv2_GrantInvokeOptions(options);
        if (!this.authBindResult || this.authBindResult.authorizationType !== HttpRouteAuthorizationType.AWS_IAM) {
            throw new Error('To use grantInvoke, you must use IAM authorization');
        }
        const httpMethods = Array.from(new Set((_c = options.httpMethods) !== null && _c !== void 0 ? _c : [this.method]));
        if (this.method !== HttpMethod.ANY && httpMethods.some(method => method !== this.method)) {
            throw new Error('This route does not support granting invoke for all requested http methods');
        }
        const resourceArns = httpMethods.map(httpMethod => {
            return this.produceRouteArn(httpMethod);
        });
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['execute-api:Invoke'],
            resourceArns: resourceArns,
        });
    }
}
exports.HttpRoute = HttpRoute;
_b = JSII_RTTI_SYMBOL_1;
HttpRoute[_b] = { fqn: "monocdk.aws_apigatewayv2.HttpRoute", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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