"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.UsagePlan = exports.Period = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const cx_api_1 = require("../../cx-api");
const apigateway_generated_1 = require("./apigateway.generated");
const util_1 = require("./util");
/**
 * Time period for which quota settings apply.
 */
var Period;
(function (Period) {
    Period["DAY"] = "DAY";
    Period["WEEK"] = "WEEK";
    Period["MONTH"] = "MONTH";
})(Period = exports.Period || (exports.Period = {}));
class UsagePlanBase extends core_1.Resource {
    /**
     * Adds an ApiKey.
     *
     * @param apiKey the api key to associate with this usage plan
     * @param options options that control the behaviour of this method
     */
    addApiKey(apiKey, options) {
        let id;
        const prefix = 'UsagePlanKeyResource';
        if (core_1.FeatureFlags.of(this).isEnabled(cx_api_1.APIGATEWAY_USAGEPLANKEY_ORDERINSENSITIVE_ID)) {
            id = `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}`;
        }
        else {
            // Postfixing apikey id only from the 2nd child, to keep physicalIds of UsagePlanKey for existing CDK apps unmodified.
            id = this.node.tryFindChild(prefix) ? `${prefix}:${core_1.Names.nodeUniqueId(apiKey.node)}` : prefix;
        }
        const resource = new apigateway_generated_1.CfnUsagePlanKey(this, id, {
            keyId: apiKey.keyId,
            keyType: "API_KEY" /* API_KEY */,
            usagePlanId: this.usagePlanId,
        });
        if (options === null || options === void 0 ? void 0 : options.overrideLogicalId) {
            resource.overrideLogicalId(options === null || options === void 0 ? void 0 : options.overrideLogicalId);
        }
    }
}
class UsagePlan extends UsagePlanBase {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.apiStages = new Array();
        jsiiDeprecationWarnings.monocdk_aws_apigateway_UsagePlanProps(props);
        let resource;
        resource = new apigateway_generated_1.CfnUsagePlan(this, 'Resource', {
            apiStages: core_1.Lazy.any({ produce: () => this.renderApiStages(this.apiStages) }),
            description: props.description,
            quota: this.renderQuota(props),
            throttle: this.renderThrottle(props.throttle),
            usagePlanName: props.name,
        });
        this.apiStages.push(...(props.apiStages || []));
        this.usagePlanId = resource.ref;
        // Add ApiKey when
        if (props.apiKey) {
            this.addApiKey(props.apiKey);
        }
    }
    /**
     * Import an externally defined usage plan using its ARN.
     *
     * @param scope  the construct that will "own" the imported usage plan.
     * @param id     the id of the imported usage plan in the construct tree.
     * @param usagePlanId the id of an existing usage plan.
     */
    static fromUsagePlanId(scope, id, usagePlanId) {
        class Import extends UsagePlanBase {
            constructor() {
                super(scope, id);
                this.usagePlanId = usagePlanId;
            }
        }
        return new Import();
    }
    /**
     * Adds an apiStage.
     * @param apiStage
     */
    addApiStage(apiStage) {
        jsiiDeprecationWarnings.monocdk_aws_apigateway_UsagePlanPerApiStage(apiStage);
        this.apiStages.push(apiStage);
    }
    /**
     *
     * @param props
     */
    renderApiStages(apiStages) {
        if (apiStages && apiStages.length > 0) {
            const stages = [];
            apiStages.forEach((apiStage) => {
                stages.push(this.createStage(apiStage));
            });
            return stages;
        }
        return undefined;
    }
    createStage(apiStage) {
        const stage = apiStage.stage ? apiStage.stage.stageName.toString() : undefined;
        const apiId = apiStage.stage ? apiStage.stage.restApi.restApiId : undefined;
        const throttle = this.renderThrottlePerMethod(apiStage.throttle);
        return {
            apiId,
            stage,
            throttle,
        };
    }
    renderQuota(props) {
        if (props.quota === undefined) {
            return undefined;
        }
        else {
            const limit = props.quota ? props.quota.limit : undefined;
            util_1.validateInteger(limit, 'Throttle quota limit');
            const ret = {
                limit: limit ? limit : undefined,
                offset: props.quota ? props.quota.offset : undefined,
                period: props.quota ? props.quota.period : undefined,
            };
            return ret;
        }
    }
    renderThrottle(props) {
        let ret;
        if (props !== undefined) {
            const burstLimit = props.burstLimit;
            util_1.validateInteger(burstLimit, 'Throttle burst limit');
            const rateLimit = props.rateLimit;
            util_1.validateDouble(rateLimit, 'Throttle rate limit');
            ret = {
                burstLimit: burstLimit,
                rateLimit: rateLimit,
            };
        }
        return ret;
    }
    renderThrottlePerMethod(throttlePerMethod) {
        const ret = {};
        if (throttlePerMethod && throttlePerMethod.length > 0) {
            throttlePerMethod.forEach((value) => {
                const method = value.method;
                // this methodId is resource path and method for example /GET or /pets/GET
                const methodId = `${method.resource.path}/${method.httpMethod}`;
                ret[methodId] = this.renderThrottle(value.throttle);
            });
        }
        return ret;
    }
}
exports.UsagePlan = UsagePlan;
_a = JSII_RTTI_SYMBOL_1;
UsagePlan[_a] = { fqn: "monocdk.aws_apigateway.UsagePlan", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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