"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionsIntegration = void 0;
const jsiiDeprecationWarnings = require("../../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("../../../aws-iam");
const sfn = require("../../../aws-stepfunctions");
const core_1 = require("../../../core");
const integration_1 = require("../integration");
const aws_1 = require("./aws");
/**
 * Options to integrate with various StepFunction API
 */
class StepFunctionsIntegration {
    /**
     * Integrates a Synchronous Express State Machine from AWS Step Functions to an API Gateway method.
     *
     * @example
     *
     *    const stateMachine = new stepfunctions.StateMachine(this, 'MyStateMachine', {
     *       definition: stepfunctions.Chain.start(new stepfunctions.Pass(this, 'Pass')),
     *    });
     *
     *    const api = new apigateway.RestApi(this, 'Api', {
     *       restApiName: 'MyApi',
     *    });
     *    api.root.addMethod('GET', apigateway.StepFunctionsIntegration.startExecution(stateMachine));
     */
    static startExecution(stateMachine, options) {
        jsiiDeprecationWarnings.monocdk_aws_stepfunctions_IStateMachine(stateMachine);
        jsiiDeprecationWarnings.monocdk_aws_apigateway_StepFunctionsExecutionIntegrationOptions(options);
        return new StepFunctionsExecutionIntegration(stateMachine, options);
    }
}
exports.StepFunctionsIntegration = StepFunctionsIntegration;
_a = JSII_RTTI_SYMBOL_1;
StepFunctionsIntegration[_a] = { fqn: "monocdk.aws_apigateway.StepFunctionsIntegration", version: "1.149.0" };
class StepFunctionsExecutionIntegration extends aws_1.AwsIntegration {
    constructor(stateMachine, options = {}) {
        super({
            service: 'states',
            action: 'StartSyncExecution',
            options: {
                credentialsRole: options.credentialsRole,
                integrationResponses: integrationResponse(),
                passthroughBehavior: integration_1.PassthroughBehavior.NEVER,
                requestTemplates: requestTemplates(stateMachine, options),
                ...options,
            },
        });
        this.stateMachine = stateMachine;
    }
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        this.stateMachine.grantExecution(principal, 'states:StartSyncExecution');
        let stateMachineName;
        if (this.stateMachine instanceof sfn.StateMachine) {
            const stateMachineType = this.stateMachine.stateMachineType;
            if (stateMachineType !== sfn.StateMachineType.EXPRESS) {
                throw new Error('State Machine must be of type "EXPRESS". Please use StateMachineType.EXPRESS as the stateMachineType');
            }
            //if not imported, extract the name from the CFN layer to reach the
            //literal value if it is given (rather than a token)
            stateMachineName = this.stateMachine.node.defaultChild.stateMachineName;
        }
        else {
            //imported state machine
            stateMachineName = `StateMachine-${this.stateMachine.stack.node.addr}`;
        }
        let deploymentToken;
        if (stateMachineName !== undefined && !core_1.Token.isUnresolved(stateMachineName)) {
            deploymentToken = JSON.stringify({ stateMachineName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
/**
 * Defines the integration response that passes the result on success,
 * or the error on failure, from the synchronous execution to the caller.
 *
 * @returns integrationResponse mapping
 */
function integrationResponse() {
    const errorResponse = [
        {
            /**
             * Specifies the regular expression (regex) pattern used to choose
             * an integration response based on the response from the back end.
             * In this case it will match all '4XX' HTTP Errors
             */
            selectionPattern: '4\\d{2}',
            statusCode: '400',
            responseTemplates: {
                'application/json': `{
            "error": "Bad request!"
          }`,
            },
        },
        {
            /**
             * Match all '5XX' HTTP Errors
             */
            selectionPattern: '5\\d{2}',
            statusCode: '500',
            responseTemplates: {
                'application/json': '"error": $input.path(\'$.error\')',
            },
        },
    ];
    const integResponse = [
        {
            statusCode: '200',
            responseTemplates: {
                /* eslint-disable */
                'application/json': [
                    '#set($inputRoot = $input.path(\'$\'))',
                    '#if($input.path(\'$.status\').toString().equals("FAILED"))',
                    '#set($context.responseOverride.status = 500)',
                    '{',
                    '"error": "$input.path(\'$.error\')",',
                    '"cause": "$input.path(\'$.cause\')"',
                    '}',
                    '#else',
                    '$input.path(\'$.output\')',
                    '#end',
                ].join('\n'),
            },
        },
        ...errorResponse,
    ];
    return integResponse;
}
/**
 * Defines the request template that will be used for the integration
 * @param stateMachine
 * @param options
 * @returns requestTemplate
 */
function requestTemplates(stateMachine, options) {
    const templateStr = templateString(stateMachine, options);
    const requestTemplate = {
        'application/json': templateStr,
    };
    return requestTemplate;
}
/**
 * Reads the VTL template and returns the template string to be used
 * for the request template.
 *
 * @param stateMachine
 * @param includeRequestContext
 * @param options
 * @reutrns templateString
 */
function templateString(stateMachine, options) {
    var _b, _c, _d, _e;
    let templateStr;
    let requestContextStr = '';
    const includeHeader = (_b = options.headers) !== null && _b !== void 0 ? _b : false;
    const includeQueryString = (_c = options.querystring) !== null && _c !== void 0 ? _c : true;
    const includePath = (_d = options.path) !== null && _d !== void 0 ? _d : true;
    const includeAuthorizer = (_e = options.authorizer) !== null && _e !== void 0 ? _e : false;
    if (options.requestContext && Object.keys(options.requestContext).length > 0) {
        requestContextStr = requestContext(options.requestContext);
    }
    templateStr = fs.readFileSync(path.join(__dirname, 'stepfunctions.vtl'), { encoding: 'utf-8' });
    templateStr = templateStr.replace('%STATEMACHINE%', stateMachine.stateMachineArn);
    templateStr = templateStr.replace('%INCLUDE_HEADERS%', String(includeHeader));
    templateStr = templateStr.replace('%INCLUDE_QUERYSTRING%', String(includeQueryString));
    templateStr = templateStr.replace('%INCLUDE_PATH%', String(includePath));
    templateStr = templateStr.replace('%INCLUDE_AUTHORIZER%', String(includeAuthorizer));
    templateStr = templateStr.replace('%REQUESTCONTEXT%', requestContextStr);
    return templateStr;
}
function requestContext(requestContextObj) {
    const context = {
        accountId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.accountId) ? '$context.identity.accountId' : undefined,
        apiId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.apiId) ? '$context.apiId' : undefined,
        apiKey: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.apiKey) ? '$context.identity.apiKey' : undefined,
        authorizerPrincipalId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.authorizerPrincipalId) ? '$context.authorizer.principalId' : undefined,
        caller: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.caller) ? '$context.identity.caller' : undefined,
        cognitoAuthenticationProvider: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoAuthenticationProvider) ? '$context.identity.cognitoAuthenticationProvider' : undefined,
        cognitoAuthenticationType: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoAuthenticationType) ? '$context.identity.cognitoAuthenticationType' : undefined,
        cognitoIdentityId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoIdentityId) ? '$context.identity.cognitoIdentityId' : undefined,
        cognitoIdentityPoolId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoIdentityPoolId) ? '$context.identity.cognitoIdentityPoolId' : undefined,
        httpMethod: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.httpMethod) ? '$context.httpMethod' : undefined,
        stage: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.stage) ? '$context.stage' : undefined,
        sourceIp: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.sourceIp) ? '$context.identity.sourceIp' : undefined,
        user: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.user) ? '$context.identity.user' : undefined,
        userAgent: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.userAgent) ? '$context.identity.userAgent' : undefined,
        userArn: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.userArn) ? '$context.identity.userArn' : undefined,
        requestId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.requestId) ? '$context.requestId' : undefined,
        resourceId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.resourceId) ? '$context.resourceId' : undefined,
        resourcePath: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.resourcePath) ? '$context.resourcePath' : undefined,
    };
    const contextAsString = JSON.stringify(context);
    // The VTL Template conflicts with double-quotes (") for strings.
    // Before sending to the template, we replace double-quotes (") with @@ and replace it back inside the .vtl file
    const doublequotes = '"';
    const replaceWith = '@@';
    return contextAsString.split(doublequotes).join(replaceWith);
}
//# sourceMappingURL=data:application/json;base64,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