"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetApiDefinition = exports.InlineApiDefinition = exports.S3ApiDefinition = exports.ApiDefinition = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const s3_assets = require("../../aws-s3-assets");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const cxapi = require("../../cx-api");
const constructs_1 = require("constructs");
/**
 * Represents an OpenAPI definition asset.
 */
class ApiDefinition {
    /**
     * Creates an API definition from a specification file in an S3 bucket
     */
    static fromBucket(bucket, key, objectVersion) {
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        return new S3ApiDefinition(bucket, key, objectVersion);
    }
    /**
     * Create an API definition from an inline object. The inline object must follow the
     * schema of OpenAPI 2.0 or OpenAPI 3.0
     *
     * @example
     *
     *   apigateway.ApiDefinition.fromInline({
     *     openapi: '3.0.2',
     *     paths: {
     *       '/pets': {
     *         get: {
     *           'responses': {
     *             200: {
     *               content: {
     *                 'application/json': {
     *                   schema: {
     *                     $ref: '#/components/schemas/Empty',
     *                   },
     *                 },
     *               },
     *             },
     *           },
     *           'x-amazon-apigateway-integration': {
     *             responses: {
     *               default: {
     *                 statusCode: '200',
     *               },
     *             },
     *             requestTemplates: {
     *               'application/json': '{"statusCode": 200}',
     *             },
     *             passthroughBehavior: 'when_no_match',
     *             type: 'mock',
     *           },
     *         },
     *       },
     *     },
     *     components: {
     *       schemas: {
     *         Empty: {
     *           title: 'Empty Schema',
     *           type: 'object',
     *         },
     *       },
     *     },
     *   });
     */
    static fromInline(definition) {
        return new InlineApiDefinition(definition);
    }
    /**
     * Loads the API specification from a local disk asset.
     */
    static fromAsset(file, options) {
        jsiiDeprecationWarnings.monocdk_aws_s3_assets_AssetOptions(options);
        return new AssetApiDefinition(file, options);
    }
    /**
     * Called after the CFN RestApi resource has been created to allow the Api
     * Definition to bind to it. Specifically it's required to allow assets to add
     * metadata for tooling like SAM CLI to be able to find their origins.
     */
    bindAfterCreate(_scope, _restApi) {
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        jsiiDeprecationWarnings.monocdk_aws_apigateway_IRestApi(_restApi);
        return;
    }
}
exports.ApiDefinition = ApiDefinition;
_a = JSII_RTTI_SYMBOL_1;
ApiDefinition[_a] = { fqn: "monocdk.aws_apigateway.ApiDefinition", version: "1.149.0" };
/**
 * OpenAPI specification from an S3 archive.
 */
class S3ApiDefinition extends ApiDefinition {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        jsiiDeprecationWarnings.monocdk_aws_s3_IBucket(bucket);
        if (!bucket.bucketName) {
            throw new Error('bucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        return {
            s3Location: {
                bucket: this.bucketName,
                key: this.key,
                version: this.objectVersion,
            },
        };
    }
}
exports.S3ApiDefinition = S3ApiDefinition;
_b = JSII_RTTI_SYMBOL_1;
S3ApiDefinition[_b] = { fqn: "monocdk.aws_apigateway.S3ApiDefinition", version: "1.149.0" };
/**
 * OpenAPI specification from an inline JSON object.
 */
class InlineApiDefinition extends ApiDefinition {
    constructor(definition) {
        super();
        this.definition = definition;
        if (typeof (definition) !== 'object') {
            throw new Error('definition should be of type object');
        }
        if (Object.keys(definition).length === 0) {
            throw new Error('JSON definition cannot be empty');
        }
    }
    bind(_scope) {
        jsiiDeprecationWarnings.monocdk_Construct(_scope);
        return {
            inlineDefinition: this.definition,
        };
    }
}
exports.InlineApiDefinition = InlineApiDefinition;
_c = JSII_RTTI_SYMBOL_1;
InlineApiDefinition[_c] = { fqn: "monocdk.aws_apigateway.InlineApiDefinition", version: "1.149.0" };
/**
 * OpenAPI specification from a local file.
 */
class AssetApiDefinition extends ApiDefinition {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
        jsiiDeprecationWarnings.monocdk_aws_s3_assets_AssetOptions(options);
    }
    bind(scope) {
        jsiiDeprecationWarnings.monocdk_Construct(scope);
        // If the same AssetAPIDefinition is used multiple times, retain only the first instantiation.
        if (this.asset === undefined) {
            this.asset = new s3_assets.Asset(scope, 'APIDefinition', {
                path: this.path,
                ...this.options,
            });
        }
        if (this.asset.isZipArchive) {
            throw new Error(`Asset cannot be a .zip file or a directory (${this.path})`);
        }
        return {
            s3Location: {
                bucket: this.asset.s3BucketName,
                key: this.asset.s3ObjectKey,
            },
        };
    }
    bindAfterCreate(scope, restApi) {
        jsiiDeprecationWarnings.monocdk_Construct(scope);
        jsiiDeprecationWarnings.monocdk_aws_apigateway_IRestApi(restApi);
        if (!scope.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        if (!this.asset) {
            throw new Error('bindToResource() must be called after bind()');
        }
        const child = constructs_1.Node.of(restApi).defaultChild;
        child.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, this.asset.assetPath);
        child.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'BodyS3Location');
    }
}
exports.AssetApiDefinition = AssetApiDefinition;
_d = JSII_RTTI_SYMBOL_1;
AssetApiDefinition[_d] = { fqn: "monocdk.aws_apigateway.AssetApiDefinition", version: "1.149.0" };
//# sourceMappingURL=data:application/json;base64,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