"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Template = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const core_1 = require("../../core");
const fs = require("fs-extra");
const match_1 = require("./match");
const matcher_1 = require("./matcher");
const conditions_1 = require("./private/conditions");
const cyclic_1 = require("./private/cyclic");
const mappings_1 = require("./private/mappings");
const outputs_1 = require("./private/outputs");
const parameters_1 = require("./private/parameters");
const resources_1 = require("./private/resources");
/**
 * Suite of assertions that can be run on a CDK stack.
 * Typically used, as part of unit tests, to validate that the rendered
 * CloudFormation template has expected resources and properties.
 */
class Template {
    constructor(template) {
        this.template = template;
        cyclic_1.checkTemplateForCyclicDependencies(this.template);
    }
    /**
     * Base your assertions on the CloudFormation template synthesized by a CDK `Stack`.
     * @param stack the CDK Stack to run assertions on
     */
    static fromStack(stack) {
        jsiiDeprecationWarnings.monocdk_Stack(stack);
        return new Template(toTemplate(stack));
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as an in-memory
     * JSON object.
     * @param template the CloudFormation template formatted as a nested set of records
     */
    static fromJSON(template) {
        return new Template(template);
    }
    /**
     * Base your assertions from an existing CloudFormation template formatted as a
     * JSON string.
     * @param template the CloudFormation template in
     */
    static fromString(template) {
        return new Template(JSON.parse(template));
    }
    /**
     * The CloudFormation template deserialized into an object.
     */
    toJSON() {
        return this.template;
    }
    /**
     * Assert that the given number of resources of the given type exist in the
     * template.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param count number of expected instances
     */
    resourceCountIs(type, count) {
        const counted = resources_1.countResources(this.template, type);
        if (counted !== count) {
            throw new Error(`Expected ${count} resources of type ${type} but found ${counted}`);
        }
    }
    /**
     * Assert that a resource of the given type and properties exists in the
     * CloudFormation template.
     * By default, performs partial matching on the `Properties` key of the resource, via the
     * `Match.objectLike()`. To configure different behavour, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the 'Properties' section of the resource as should be expected in the template.
     */
    hasResourceProperties(type, props) {
        const matchError = resources_1.hasResourceProperties(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that a resource of the given type and given definition exists in the
     * CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param type the resource type; ex: `AWS::S3::Bucket`
     * @param props the entire defintion of the resource as should be expected in the template.
     */
    hasResource(type, props) {
        const matchError = resources_1.hasResource(this.template, type, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching resources of a given type and properties in the CloudFormation template.
     * @param type the type to match in the CloudFormation template
     * @param props by default, matches all resources with the given type.
     * When a literal is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findResources(type, props = {}) {
        return resources_1.findResources(this.template, type, props);
    }
    /**
     * Assert that a Parameter with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the parameter, via the `Match.objectLike()`.
     * To configure different behavior, use other matchers in the `Match` class.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props the parameter as should be expected in the template.
     */
    hasParameter(logicalId, props) {
        const matchError = parameters_1.hasParameter(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Parameters that match the given properties in the CloudFormation template.
     * @param logicalId the name of the parameter. Provide `'*'` to match all parameters in the template.
     * @param props by default, matches all Parameters in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findParameters(logicalId, props = {}) {
        return parameters_1.findParameters(this.template, logicalId, props);
    }
    /**
     * Assert that an Output with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props the output as should be expected in the template.
     */
    hasOutput(logicalId, props) {
        const matchError = outputs_1.hasOutput(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Outputs that match the given properties in the CloudFormation template.
     * @param logicalId the name of the output. Provide `'*'` to match all outputs in the template.
     * @param props by default, matches all Outputs in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findOutputs(logicalId, props = {}) {
        return outputs_1.findOutputs(this.template, logicalId, props);
    }
    /**
     * Assert that a Mapping with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props the output as should be expected in the template.
     */
    hasMapping(logicalId, props) {
        const matchError = mappings_1.hasMapping(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Mappings that match the given properties in the CloudFormation template.
     * @param logicalId the name of the mapping. Provide `'*'` to match all mappings in the template.
     * @param props by default, matches all Mappings in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findMappings(logicalId, props = {}) {
        return mappings_1.findMappings(this.template, logicalId, props);
    }
    /**
     * Assert that a Condition with the given properties exists in the CloudFormation template.
     * By default, performs partial matching on the resource, via the `Match.objectLike()`.
     * To configure different behavour, use other matchers in the `Match` class.
     * @param logicalId the name of the mapping. Provide `'*'` to match all conditions in the template.
     * @param props the output as should be expected in the template.
     */
    hasCondition(logicalId, props) {
        const matchError = conditions_1.hasCondition(this.template, logicalId, props);
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching Conditions that match the given properties in the CloudFormation template.
     * @param logicalId the name of the condition. Provide `'*'` to match all conditions in the template.
     * @param props by default, matches all Conditions in the template.
     * When a literal object is provided, performs a partial match via `Match.objectLike()`.
     * Use the `Match` APIs to configure a different behaviour.
     */
    findConditions(logicalId, props = {}) {
        return conditions_1.findConditions(this.template, logicalId, props);
    }
    /**
     * Assert that the CloudFormation template matches the given value
     * @param expected the expected CloudFormation template as key-value pairs.
     */
    templateMatches(expected) {
        const matcher = matcher_1.Matcher.isMatcher(expected) ? expected : match_1.Match.objectLike(expected);
        const result = matcher.test(this.template);
        if (result.hasFailed()) {
            throw new Error([
                'Template did not match as expected. The following mismatches were found:',
                ...result.toHumanStrings().map(s => `\t${s}`),
            ].join('\n'));
        }
    }
}
exports.Template = Template;
_a = JSII_RTTI_SYMBOL_1;
Template[_a] = { fqn: "monocdk.assertions.Template", version: "1.149.0" };
function toTemplate(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    const assembly = root.synth();
    if (stack.nestedStackParent) {
        // if this is a nested stack (it has a parent), then just read the template as a string
        return JSON.parse(fs.readFileSync(path.join(assembly.directory, stack.templateFile)).toString('utf-8'));
    }
    return assembly.getStackArtifact(stack.artifactId).template;
}
//# sourceMappingURL=data:application/json;base64,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