"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const jsiiDeprecationWarnings = require("../../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const core_1 = require("../../core");
const messages_1 = require("./private/messages");
/**
 * Suite of assertions that can be run on a CDK Stack.
 * Focused on asserting annotations.
 */
class Annotations {
    constructor(messages) {
        this._messages = convertArrayToMessagesType(messages);
    }
    /**
     * Base your assertions on the messages returned by a synthesized CDK `Stack`.
     * @param stack the CDK Stack to run assertions on
     */
    static fromStack(stack) {
        jsiiDeprecationWarnings.monocdk_Stack(stack);
        return new Annotations(toMessages(stack));
    }
    /**
     * Assert that an error with the given message exists in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the error. Provide `'*'` to match all errors in the template.
     * @param message the error message as should be expected. This should be a string or Matcher object.
     */
    hasError(constructPath, message) {
        const matchError = messages_1.hasMessage(this._messages, constructPath, constructMessage('error', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that an error with the given message does not exist in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the error. Provide `'*'` to match all errors in the template.
     * @param message the error message as should be expected. This should be a string or Matcher object.
     */
    hasNoError(constructPath, message) {
        const matchError = messages_1.hasNoMessage(this._messages, constructPath, constructMessage('error', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching errors of a given construct path and message.
     *
     * @param constructPath the construct path to the error. Provide `'*'` to match all errors in the template.
     * @param message the error message as should be expected. This should be a string or Matcher object.
     */
    findError(constructPath, message) {
        return convertMessagesTypeToArray(messages_1.findMessage(this._messages, constructPath, constructMessage('error', message)));
    }
    /**
     * Assert that an warning with the given message exists in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the warning. Provide `'*'` to match all warnings in the template.
     * @param message the warning message as should be expected. This should be a string or Matcher object.
     */
    hasWarning(constructPath, message) {
        const matchError = messages_1.hasMessage(this._messages, constructPath, constructMessage('warning', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that an warning with the given message does not exist in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the warning. Provide `'*'` to match all warnings in the template.
     * @param message the warning message as should be expected. This should be a string or Matcher object.
     */
    hasNoWarning(constructPath, message) {
        const matchError = messages_1.hasNoMessage(this._messages, constructPath, constructMessage('warning', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching warning of a given construct path and message.
     *
     * @param constructPath the construct path to the warning. Provide `'*'` to match all warnings in the template.
     * @param message the warning message as should be expected. This should be a string or Matcher object.
     */
    findWarning(constructPath, message) {
        return convertMessagesTypeToArray(messages_1.findMessage(this._messages, constructPath, constructMessage('warning', message)));
    }
    /**
     * Assert that an info with the given message exists in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the info. Provide `'*'` to match all info in the template.
     * @param message the info message as should be expected. This should be a string or Matcher object.
     */
    hasInfo(constructPath, message) {
        const matchError = messages_1.hasMessage(this._messages, constructPath, constructMessage('info', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Assert that an info with the given message does not exist in the synthesized CDK `Stack`.
     *
     * @param constructPath the construct path to the info. Provide `'*'` to match all info in the template.
     * @param message the info message as should be expected. This should be a string or Matcher object.
     */
    hasNoInfo(constructPath, message) {
        const matchError = messages_1.hasNoMessage(this._messages, constructPath, constructMessage('info', message));
        if (matchError) {
            throw new Error(matchError);
        }
    }
    /**
     * Get the set of matching infos of a given construct path and message.
     *
     * @param constructPath the construct path to the info. Provide `'*'` to match all infos in the template.
     * @param message the info message as should be expected. This should be a string or Matcher object.
     */
    findInfo(constructPath, message) {
        return convertMessagesTypeToArray(messages_1.findMessage(this._messages, constructPath, constructMessage('info', message)));
    }
}
exports.Annotations = Annotations;
_a = JSII_RTTI_SYMBOL_1;
Annotations[_a] = { fqn: "monocdk.assertions.Annotations", version: "1.149.0" };
function constructMessage(type, message) {
    return {
        level: type,
        entry: {
            data: message,
        },
    };
}
function convertArrayToMessagesType(messages) {
    return messages.reduce((obj, item, index) => {
        return {
            ...obj,
            [index]: item,
        };
    }, {});
}
function convertMessagesTypeToArray(messages) {
    return Object.values(messages);
}
function toMessages(stack) {
    const root = stack.node.root;
    if (!core_1.Stage.isStage(root)) {
        throw new Error('unexpected: all stacks must be part of a Stage or an App');
    }
    // to support incremental assertions (i.e. "expect(stack).toNotContainSomething(); doSomething(); expect(stack).toContainSomthing()")
    const force = true;
    const assembly = root.synth({ force });
    return assembly.getStackArtifact(stack.artifactId).messages;
}
//# sourceMappingURL=data:application/json;base64,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