"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultGlobalRestApiProps = exports.DefaultRegionalLambdaRestApiProps = exports.DefaultGlobalLambdaRestApiProps = void 0;
const api = require("@aws-cdk/aws-apigateway");
/**
 * Private function to configure an api.RestApiProps
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultRestApiProps(_endpointType, _logGroup) {
    return {
        endpointConfiguration: {
            types: _endpointType
        },
        cloudWatchRole: false,
        // Configure API Gateway Access logging
        deployOptions: {
            accessLogDestination: new api.LogGroupLogDestination(_logGroup),
            accessLogFormat: api.AccessLogFormat.jsonWithStandardFields(),
            loggingLevel: api.MethodLoggingLevel.INFO,
            dataTraceEnabled: true,
            tracingEnabled: true
        },
        defaultMethodOptions: {
            authorizationType: api.AuthorizationType.IAM
        }
    };
}
/**
 * Provides the default set of properties for Edge/Global Lambda backed RestApi
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultGlobalLambdaRestApiProps(_existingLambdaObj, _logGroup) {
    const baseProps = DefaultRestApiProps([api.EndpointType.EDGE], _logGroup);
    const extraProps = {
        handler: _existingLambdaObj,
    };
    return Object.assign(baseProps, extraProps);
}
exports.DefaultGlobalLambdaRestApiProps = DefaultGlobalLambdaRestApiProps;
/**
 * Provides the default set of properties for Regional Lambda backed RestApi
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultRegionalLambdaRestApiProps(_existingLambdaObj, _logGroup) {
    const baseProps = DefaultRestApiProps([api.EndpointType.REGIONAL], _logGroup);
    const extraProps = {
        handler: _existingLambdaObj,
    };
    return Object.assign(baseProps, extraProps);
}
exports.DefaultRegionalLambdaRestApiProps = DefaultRegionalLambdaRestApiProps;
/**
 * Provides the default set of properties for Edge/Global RestApi
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _endpointType - endpoint type for Api Gateway e.g. Regional, Global, Private
 * @param _logGroup - CW Log group for Api Gateway access logging
 */
function DefaultGlobalRestApiProps(_logGroup) {
    return DefaultRestApiProps([api.EndpointType.EDGE], _logGroup);
}
exports.DefaultGlobalRestApiProps = DefaultGlobalRestApiProps;
//# sourceMappingURL=data:application/json;base64,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