"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const path_1 = require("path");
const lib_1 = require("../lib");
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'demoapi',
    schemaDefinitionFile: path_1.join(__dirname, 'schema.graphql'),
    authorizationConfig: {
        defaultAuthorization: {
            userPool,
            defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
        },
        additionalAuthorizationModes: [
            {
                apiKeyDesc: 'My API Key',
            },
        ],
    },
});
const noneDS = api.addNoneDataSource('None', 'Dummy data source');
noneDS.createResolver({
    typeName: 'Query',
    fieldName: 'getServiceVersion',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: '2017-02-28',
    })),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: 'v1',
    })),
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const customerDS = api.addDynamoDbDataSource('Customer', 'The customer data source', customerTable);
const orderDS = api.addDynamoDbDataSource('Order', 'The order data source', orderTable);
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: 'Eq', op: lib_1.KeyCondition.eq },
    { suffix: 'Lt', op: lib_1.KeyCondition.lt },
    { suffix: 'Le', op: lib_1.KeyCondition.le },
    { suffix: 'Gt', op: lib_1.KeyCondition.gt },
    { suffix: 'Ge', op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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