"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const fs_1 = require("fs");
const appsync_generated_1 = require("./appsync.generated");
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
function isUserPoolConfig(obj) {
    return obj.userPool !== undefined;
}
function isApiKeyConfig(obj) {
    return obj.apiKeyDesc !== undefined;
}
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * An AppSync GraphQL API
 */
class GraphQLApi extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        let apiLogsRole;
        if (props.logConfig) {
            apiLogsRole = new aws_iam_1.Role(this, 'ApiLogsRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
            apiLogsRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'));
        }
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: 'API_KEY',
            ...props.logConfig && {
                logConfig: {
                    cloudWatchLogsRoleArn: apiLogsRole ? apiLogsRole.roleArn : undefined,
                    excludeVerboseContent: props.logConfig.excludeVerboseContent,
                    fieldLogLevel: props.logConfig.fieldLogLevel ? props.logConfig.fieldLogLevel.toString() : undefined,
                },
            },
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphQlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        if (props.authorizationConfig) {
            this.setupAuth(props.authorizationConfig);
        }
        let definition;
        if (props.schemaDefinition) {
            definition = props.schemaDefinition;
        }
        else if (props.schemaDefinitionFile) {
            definition = fs_1.readFileSync(props.schemaDefinitionFile).toString('UTF-8');
        }
        else {
            throw new Error('Missing Schema definition. Provide schemaDefinition or schemaDefinitionFile');
        }
        this.schema = new appsync_generated_1.CfnGraphQLSchema(this, 'Schema', {
            apiId: this.apiId,
            definition,
        });
    }
    /**
     * the configured API key, if present
     */
    get apiKey() {
        return this._apiKey;
    }
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name, description) {
        return new NoneDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
        });
    }
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name, description, table) {
        return new DynamoDbDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            table,
        });
    }
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name, description, lambdaFunction) {
        return new LambdaDataSource(this, `${name}DS`, {
            api: this,
            description,
            name,
            lambdaFunction,
        });
    }
    setupAuth(auth) {
        if (isUserPoolConfig(auth.defaultAuthorization)) {
            const { authenticationType, userPoolConfig } = this.userPoolDescFrom(auth.defaultAuthorization);
            this.api.authenticationType = authenticationType;
            this.api.userPoolConfig = userPoolConfig;
        }
        else if (isApiKeyConfig(auth.defaultAuthorization)) {
            this.api.authenticationType = this.apiKeyDesc(auth.defaultAuthorization).authenticationType;
        }
        this.api.additionalAuthenticationProviders = [];
        for (const mode of (auth.additionalAuthorizationModes || [])) {
            if (isUserPoolConfig(mode)) {
                this.api.additionalAuthenticationProviders.push(this.userPoolDescFrom(mode));
            }
            else if (isApiKeyConfig(mode)) {
                this.api.additionalAuthenticationProviders.push(this.apiKeyDesc(mode));
            }
        }
    }
    userPoolDescFrom(upConfig) {
        return {
            authenticationType: 'AMAZON_COGNITO_USER_POOLS',
            userPoolConfig: {
                appIdClientRegex: upConfig.appIdClientRegex,
                userPoolId: upConfig.userPool.userPoolId,
                awsRegion: upConfig.userPool.stack.region,
                defaultAction: upConfig.defaultAction ? upConfig.defaultAction.toString() : 'ALLOW',
            },
        };
    }
    apiKeyDesc(akConfig) {
        let expires;
        if (akConfig.expires) {
            expires = new Date(akConfig.expires).valueOf();
            const now = Date.now();
            const days = (d) => now + core_1.Duration.days(d).toMilliseconds();
            if (expires < days(1) || expires > days(365)) {
                throw Error('API key expiration must be between 1 and 365 days.');
            }
            expires = Math.round(expires / 1000);
        }
        const key = new appsync_generated_1.CfnApiKey(this, `${akConfig.apiKeyDesc || ''}ApiKey`, {
            expires,
            description: akConfig.apiKeyDesc,
            apiId: this.apiId,
        });
        this._apiKey = key.attrApiKey;
        return { authenticationType: 'API_KEY' };
    }
}
exports.GraphQLApi = GraphQLApi;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: props.name,
            description: props.description,
            serviceRoleArn: (_a = this.serviceRole) === null || _a === void 0 ? void 0 : _a.roleArn,
            ...extended,
        });
        this.name = props.name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
function concatAndDedup(left, right) {
    return left.concat(right).filter((elem, index, self) => {
        return index === self.indexOf(elem);
    });
}
/**
 * Utility class to represent DynamoDB key conditions.
 */
class BaseKeyCondition {
    and(cond) {
        return new (class extends BaseKeyCondition {
            constructor(left, right) {
                super();
                this.left = left;
                this.right = right;
            }
            renderCondition() {
                return `${this.left.renderCondition()} AND ${this.right.renderCondition()}`;
            }
            keyNames() {
                return concatAndDedup(this.left.keyNames(), this.right.keyNames());
            }
            args() {
                return concatAndDedup(this.left.args(), this.right.args());
            }
        })(this, cond);
    }
    renderExpressionNames() {
        return this.keyNames()
            .map((keyName) => {
            return `"#${keyName}" : "${keyName}"`;
        })
            .join(', ');
    }
    renderExpressionValues() {
        return this.args()
            .map((arg) => {
            return `":${arg}" : $util.dynamodb.toDynamoDBJson($ctx.args.${arg})`;
        })
            .join(', ');
    }
}
/**
 * Utility class to represent DynamoDB "begins_with" key conditions.
 */
class BeginsWith extends BaseKeyCondition {
    constructor(keyName, arg) {
        super();
        this.keyName = keyName;
        this.arg = arg;
    }
    renderCondition() {
        return `begins_with(#${this.keyName}, :${this.arg})`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB binary key conditions.
 */
class BinaryCondition extends BaseKeyCondition {
    constructor(keyName, op, arg) {
        super();
        this.keyName = keyName;
        this.op = op;
        this.arg = arg;
    }
    renderCondition() {
        return `#${this.keyName} ${this.op} :${this.arg}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg];
    }
}
/**
 * Utility class to represent DynamoDB "between" key conditions.
 */
class Between extends BaseKeyCondition {
    constructor(keyName, arg1, arg2) {
        super();
        this.keyName = keyName;
        this.arg1 = arg1;
        this.arg2 = arg2;
    }
    renderCondition() {
        return `#${this.keyName} BETWEEN :${this.arg1} AND :${this.arg2}`;
    }
    keyNames() {
        return [this.keyName];
    }
    args() {
        return [this.arg1, this.arg2];
    }
}
/**
 * Factory class for DynamoDB key conditions.
 */
class KeyCondition {
    constructor(cond) {
        this.cond = cond;
    }
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '=', arg));
    }
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<', arg));
    }
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '<=', arg));
    }
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>', arg));
    }
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName, arg) {
        return new KeyCondition(new BinaryCondition(keyName, '>=', arg));
    }
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName, arg) {
        return new KeyCondition(new BeginsWith(keyName, arg));
    }
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName, arg1, arg2) {
        return new KeyCondition(new Between(keyName, arg1, arg2));
    }
    /**
     * Conjunction between two conditions.
     */
    and(keyCond) {
        return new KeyCondition(this.cond.and(keyCond.cond));
    }
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate() {
        return `"query" : {
            "expression" : "${this.cond.renderCondition()}",
            "expressionNames" : {
                ${this.cond.renderExpressionNames()}
            },
            "expressionValues" : {
                ${this.cond.renderExpressionValues()}
            }
        }`;
    }
}
exports.KeyCondition = KeyCondition;
/**
 * Utility class representing the assigment of a value to an attribute.
 */
class Assign {
    constructor(attr, arg) {
        this.attr = attr;
        this.arg = arg;
    }
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment() {
        return `"${this.attr}" : $util.dynamodb.toDynamoDBJson(${this.arg})`;
    }
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map) {
        return `$util.qr($${map}.put("${this.attr}", "${this.arg}"))`;
    }
}
exports.Assign = Assign;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
class PartitionKeyStep {
    constructor(key) {
        this.key = key;
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val) {
        return new PartitionKey(new Assign(this.key, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto() {
        return new PartitionKey(new Assign(this.key, '$util.autoId()'));
    }
}
exports.PartitionKeyStep = PartitionKeyStep;
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
class SortKeyStep {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val) {
        return new PrimaryKey(this.pkey, new Assign(this.skey, `$ctx.args.${val}`));
    }
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto() {
        return new PrimaryKey(this.pkey, new Assign(this.skey, '$util.autoId()'));
    }
}
exports.SortKeyStep = SortKeyStep;
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
class PrimaryKey {
    constructor(pkey, skey) {
        this.pkey = pkey;
        this.skey = skey;
    }
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key) {
        return new PartitionKeyStep(key);
    }
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate() {
        const assignments = [this.pkey.renderAsAssignment()];
        if (this.skey) {
            assignments.push(this.skey.renderAsAssignment());
        }
        return `"key" : {
            ${assignments.join(',')}
        }`;
    }
}
exports.PrimaryKey = PrimaryKey;
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
class PartitionKey extends PrimaryKey {
    constructor(pkey) {
        super(pkey);
    }
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key) {
        return new SortKeyStep(this.pkey, key);
    }
}
exports.PartitionKey = PartitionKey;
/**
 * Specifies the attribute value assignments.
 */
class AttributeValues {
    constructor(container, assignments = []) {
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr) {
        return new AttributeValuesStep(attr, this.container, this.assignments);
    }
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate() {
        return `
            #set($input = ${this.container})
            ${this.assignments.map(a => a.putInMap('input')).join('\n')}
            "attributeValues": $util.dynamodb.toMapValuesJson($input)`;
    }
}
exports.AttributeValues = AttributeValues;
/**
 * Utility class to allow assigning a value to an attribute.
 */
class AttributeValuesStep {
    constructor(attr, container, assignments) {
        this.attr = attr;
        this.container = container;
        this.assignments = assignments;
    }
    /**
     * Assign the value to the current attribute.
     */
    is(val) {
        this.assignments.push(new Assign(this.attr, val));
        return new AttributeValues(this.container, this.assignments);
    }
}
exports.AttributeValuesStep = AttributeValuesStep;
/**
 * Factory class for attribute value assignments.
 */
class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg) {
        return new AttributeValues('$ctx.args' + (arg ? `.${arg}` : ''));
    }
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr) {
        return new AttributeValues('{}').attribute(attr);
    }
}
exports.Values = Values;
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('UTF-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`{
            "version" : "2017-02-28",
            "operation" : "PutItem",
            ${key.renderTemplate()},
            ${values.renderTemplate()}
        }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
/**
 * An AppSync resolver
 */
class Resolver extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.resolver = new appsync_generated_1.CfnResolver(this, 'Resource', {
            apiId: props.api.apiId,
            typeName: props.typeName,
            fieldName: props.fieldName,
            dataSourceName: props.dataSource ? props.dataSource.name : undefined,
            kind: props.pipelineConfig ? 'PIPELINE' : 'UNIT',
            requestMappingTemplate: props.requestMappingTemplate ? props.requestMappingTemplate.renderTemplate() : undefined,
            responseMappingTemplate: props.responseMappingTemplate ? props.responseMappingTemplate.renderTemplate() : undefined,
        });
        this.resolver.addDependsOn(props.api.schema);
        if (props.dataSource) {
            this.resolver.addDependsOn(props.dataSource.ds);
        }
        this.arn = this.resolver.attrResolverArn;
    }
}
exports.Resolver = Resolver;
//# sourceMappingURL=data:application/json;base64,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