import { IUserPool } from '@aws-cdk/aws-cognito';
import { Table } from '@aws-cdk/aws-dynamodb';
import { IGrantable, IPrincipal, IRole } from '@aws-cdk/aws-iam';
import { IFunction } from '@aws-cdk/aws-lambda';
import { Construct, IResolvable } from '@aws-cdk/core';
import { CfnDataSource, CfnGraphQLSchema, CfnResolver } from './appsync.generated';
/**
 * Marker interface for the different authorization modes.
 */
export interface AuthMode {
}
/**
 * enum with all possible values for Cognito user-pool default actions
 */
export declare enum UserPoolDefaultAction {
    /**
     * ALLOW access to API
     */
    ALLOW = "ALLOW",
    /**
     * DENY access to API
     */
    DENY = "DENY"
}
/**
 * Configuration for Cognito user-pools in AppSync
 */
export interface UserPoolConfig extends AuthMode {
    /**
     * The Cognito user pool to use as identity source
     */
    readonly userPool: IUserPool;
    /**
     * the optional app id regex
     *
     * @default -  None
     */
    readonly appIdClientRegex?: string;
    /**
     * Default auth action
     *
     * @default ALLOW
     */
    readonly defaultAction?: UserPoolDefaultAction;
}
/**
 * Configuration for API Key authorization in AppSync
 */
export interface ApiKeyConfig extends AuthMode {
    /**
     * Unique description of the API key
     */
    readonly apiKeyDesc: string;
    /**
     * The time from creation time after which the API key expires, using RFC3339 representation.
     * It must be a minimum of 1 day and a maximum of 365 days from date of creation.
     * Rounded down to the nearest hour.
     * @default - 7 days from creation time
     */
    readonly expires?: string;
}
/**
 * Configuration of the API authorization modes.
 */
export interface AuthorizationConfig {
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly defaultAuthorization?: AuthMode;
    /**
     * Additional authorization modes
     *
     * @default - No other modes
     */
    readonly additionalAuthorizationModes?: [AuthMode];
}
/**
 * log-level for fields in AppSync
 */
export declare enum FieldLogLevel {
    /**
     * No logging
     */
    NONE = "NONE",
    /**
     * Error logging
     */
    ERROR = "ERROR",
    /**
     * All logging
     */
    ALL = "ALL"
}
/**
 * Logging configuration for AppSync
 */
export interface LogConfig {
    /**
     * exclude verbose content
     *
     * @default false
     */
    readonly excludeVerboseContent?: boolean | IResolvable;
    /**
     * log level for fields
     *
     * @default - Use AppSync default
     */
    readonly fieldLogLevel?: FieldLogLevel;
}
/**
 * Properties for an AppSync GraphQL API
 */
export interface GraphQLApiProps {
    /**
     * the name of the GraphQL API
     */
    readonly name: string;
    /**
     * Optional authorization configuration
     *
     * @default - API Key authorization
     */
    readonly authorizationConfig?: AuthorizationConfig;
    /**
     * Logging configuration for this api
     *
     * @default - None
     */
    readonly logConfig?: LogConfig;
    /**
     * GraphQL schema definition. You have to specify a definition or a file containing one.
     *
     * @default - Use schemaDefinitionFile
     */
    readonly schemaDefinition?: string;
    /**
     * File containing the GraphQL schema definition. You have to specify a definition or a file containing one.
     *
     * @default - Use schemaDefinition
     */
    readonly schemaDefinitionFile?: string;
}
/**
 * An AppSync GraphQL API
 */
export declare class GraphQLApi extends Construct {
    /**
     * the id of the GraphQL API
     */
    readonly apiId: string;
    /**
     * the ARN of the API
     */
    readonly arn: string;
    /**
     * the URL of the endpoint created by AppSync
     */
    readonly graphQlUrl: string;
    /**
     * the name of the API
     */
    name: string;
    /**
     * underlying CFN schema resource
     */
    readonly schema: CfnGraphQLSchema;
    /**
     * the configured API key, if present
     */
    get apiKey(): string | undefined;
    private api;
    private _apiKey?;
    constructor(scope: Construct, id: string, props: GraphQLApiProps);
    /**
     * add a new dummy data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     */
    addNoneDataSource(name: string, description: string): NoneDataSource;
    /**
     * add a new DynamoDB data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param table The DynamoDB table backing this data source [disable-awslint:ref-via-interface]
     */
    addDynamoDbDataSource(name: string, description: string, table: Table): DynamoDbDataSource;
    /**
     * add a new Lambda data source to this API
     * @param name The name of the data source
     * @param description The description of the data source
     * @param lambdaFunction The Lambda function to call to interact with this data source
     */
    addLambdaDataSource(name: string, description: string, lambdaFunction: IFunction): LambdaDataSource;
    private setupAuth;
    private userPoolDescFrom;
    private apiKeyDesc;
}
/**
 * Base properties for an AppSync datasource
 */
export interface BaseDataSourceProps {
    /**
     * The API to attach this data source to
     */
    readonly api: GraphQLApi;
    /**
     * The name of the data source
     */
    readonly name: string;
    /**
     * the description of the data source
     *
     * @default - None
     */
    readonly description?: string;
}
/**
 * properties for an AppSync datasource backed by a resource
 */
export interface BackedDataSourceProps extends BaseDataSourceProps {
    /**
     * The IAM service role to be assumed by AppSync to interact with the data source
     *
     * @default -  Create a new role
     */
    readonly serviceRole?: IRole;
}
/**
 * props used by implementations of BaseDataSource to provide configuration. Should not be used directly.
 */
export interface ExtendedDataSourceProps {
    /**
     * the type of the AppSync datasource
     */
    readonly type: string;
    /**
     * configuration for DynamoDB Datasource
     *
     * @default - No config
     */
    readonly dynamoDbConfig?: CfnDataSource.DynamoDBConfigProperty | IResolvable;
    /**
     * configuration for Elasticsearch Datasource
     *
     * @default - No config
     */
    readonly elasticsearchConfig?: CfnDataSource.ElasticsearchConfigProperty | IResolvable;
    /**
     * configuration for HTTP Datasource
     *
     * @default - No config
     */
    readonly httpConfig?: CfnDataSource.HttpConfigProperty | IResolvable;
    /**
     * configuration for Lambda Datasource
     *
     * @default - No config
     */
    readonly lambdaConfig?: CfnDataSource.LambdaConfigProperty | IResolvable;
    /**
     * configuration for RDS Datasource
     *
     * @default - No config
     */
    readonly relationalDatabaseConfig?: CfnDataSource.RelationalDatabaseConfigProperty | IResolvable;
}
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
export declare abstract class BaseDataSource extends Construct {
    /**
     * the name of the data source
     */
    readonly name: string;
    /**
     * the underlying CFN data source resource
     */
    readonly ds: CfnDataSource;
    protected api: GraphQLApi;
    protected serviceRole?: IRole;
    constructor(scope: Construct, id: string, props: BackedDataSourceProps, extended: ExtendedDataSourceProps);
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props: BaseResolverProps): Resolver;
}
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
export declare abstract class BackedDataSource extends BaseDataSource implements IGrantable {
    /**
     * the principal of the data source to be IGrantable
     */
    readonly grantPrincipal: IPrincipal;
    constructor(scope: Construct, id: string, props: BackedDataSourceProps, extended: ExtendedDataSourceProps);
}
/**
 * Properties for an AppSync dummy datasource
 */
export interface NoneDataSourceProps extends BaseDataSourceProps {
}
/**
 * An AppSync dummy datasource
 */
export declare class NoneDataSource extends BaseDataSource {
    constructor(scope: Construct, id: string, props: NoneDataSourceProps);
}
/**
 * Properties for an AppSync DynamoDB datasource
 */
export interface DynamoDbDataSourceProps extends BackedDataSourceProps {
    /**
     * The DynamoDB table backing this data source
     * [disable-awslint:ref-via-interface]
     */
    readonly table: Table;
    /**
     * Specify whether this DS is read only or has read and write permissions to the DynamoDB table
     *
     * @default false
     */
    readonly readOnlyAccess?: boolean;
    /**
     * use credentials of caller to access DynamoDB
     *
     * @default false
     */
    readonly useCallerCredentials?: boolean;
}
/**
 * An AppSync datasource backed by a DynamoDB table
 */
export declare class DynamoDbDataSource extends BackedDataSource {
    constructor(scope: Construct, id: string, props: DynamoDbDataSourceProps);
}
/**
 * Properties for an AppSync Lambda datasource
 */
export interface LambdaDataSourceProps extends BackedDataSourceProps {
    /**
     * The Lambda function to call to interact with this data source
     */
    readonly lambdaFunction: IFunction;
}
/**
 * An AppSync datasource backed by a Lambda function
 */
export declare class LambdaDataSource extends BackedDataSource {
    constructor(scope: Construct, id: string, props: LambdaDataSourceProps);
}
/**
 * Factory class for DynamoDB key conditions.
 */
export declare class KeyCondition {
    private readonly cond;
    /**
     * Condition k = arg, true if the key attribute k is equal to the Query argument
     */
    static eq(keyName: string, arg: string): KeyCondition;
    /**
     * Condition k < arg, true if the key attribute k is less than the Query argument
     */
    static lt(keyName: string, arg: string): KeyCondition;
    /**
     * Condition k <= arg, true if the key attribute k is less than or equal to the Query argument
     */
    static le(keyName: string, arg: string): KeyCondition;
    /**
     * Condition k > arg, true if the key attribute k is greater than the the Query argument
     */
    static gt(keyName: string, arg: string): KeyCondition;
    /**
     * Condition k >= arg, true if the key attribute k is greater or equal to the Query argument
     */
    static ge(keyName: string, arg: string): KeyCondition;
    /**
     * Condition (k, arg). True if the key attribute k begins with the Query argument.
     */
    static beginsWith(keyName: string, arg: string): KeyCondition;
    /**
     * Condition k BETWEEN arg1 AND arg2, true if k >= arg1 and k <= arg2.
     */
    static between(keyName: string, arg1: string, arg2: string): KeyCondition;
    private constructor();
    /**
     * Conjunction between two conditions.
     */
    and(keyCond: KeyCondition): KeyCondition;
    /**
     * Renders the key condition to a VTL string.
     */
    renderTemplate(): string;
}
/**
 * Utility class representing the assigment of a value to an attribute.
 */
export declare class Assign {
    private readonly attr;
    private readonly arg;
    constructor(attr: string, arg: string);
    /**
     * Renders the assignment as a VTL string.
     */
    renderAsAssignment(): string;
    /**
     * Renders the assignment as a map element.
     */
    putInMap(map: string): string;
}
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a partition key.
 */
export declare class PartitionKeyStep {
    private readonly key;
    constructor(key: string);
    /**
     * Assign an auto-generated value to the partition key.
     */
    is(val: string): PartitionKey;
    /**
     * Assign an auto-generated value to the partition key.
     */
    auto(): PartitionKey;
}
/**
 * Utility class to allow assigning a value or an auto-generated id
 * to a sort key.
 */
export declare class SortKeyStep {
    private readonly pkey;
    private readonly skey;
    constructor(pkey: Assign, skey: string);
    /**
     * Assign an auto-generated value to the sort key.
     */
    is(val: string): PrimaryKey;
    /**
     * Assign an auto-generated value to the sort key.
     */
    auto(): PrimaryKey;
}
/**
 * Specifies the assignment to the primary key. It either
 * contains the full primary key or only the partition key.
 */
export declare class PrimaryKey {
    protected readonly pkey: Assign;
    private readonly skey?;
    /**
     * Allows assigning a value to the partition key.
     */
    static partition(key: string): PartitionKeyStep;
    constructor(pkey: Assign, skey?: Assign | undefined);
    /**
     * Renders the key assignment to a VTL string.
     */
    renderTemplate(): string;
}
/**
 * Specifies the assignment to the partition key. It can be
 * enhanced with the assignment of the sort key.
 */
export declare class PartitionKey extends PrimaryKey {
    constructor(pkey: Assign);
    /**
     * Allows assigning a value to the sort key.
     */
    sort(key: string): SortKeyStep;
}
/**
 * Specifies the attribute value assignments.
 */
export declare class AttributeValues {
    private readonly container;
    private readonly assignments;
    constructor(container: string, assignments?: Assign[]);
    /**
     * Allows assigning a value to the specified attribute.
     */
    attribute(attr: string): AttributeValuesStep;
    /**
     * Renders the attribute value assingments to a VTL string.
     */
    renderTemplate(): string;
}
/**
 * Utility class to allow assigning a value to an attribute.
 */
export declare class AttributeValuesStep {
    private readonly attr;
    private readonly container;
    private readonly assignments;
    constructor(attr: string, container: string, assignments: Assign[]);
    /**
     * Assign the value to the current attribute.
     */
    is(val: string): AttributeValues;
}
/**
 * Factory class for attribute value assignments.
 */
export declare class Values {
    /**
     * Treats the specified object as a map of assignments, where the property
     * names represent attribute names. It’s opinionated about how it represents
     * some of the nested objects: e.g., it will use lists (“L”) rather than sets
     * (“SS”, “NS”, “BS”). By default it projects the argument container ("$ctx.args").
     */
    static projecting(arg?: string): AttributeValues;
    /**
     * Allows assigning a value to the specified attribute.
     */
    static attribute(attr: string): AttributeValuesStep;
}
/**
 * MappingTemplates for AppSync resolvers
 */
export declare abstract class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template: string): MappingTemplate;
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName: string): MappingTemplate;
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList(): MappingTemplate;
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem(): MappingTemplate;
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable(): MappingTemplate;
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond: KeyCondition): MappingTemplate;
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName: string, idArg: string): MappingTemplate;
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName: string, idArg: string): MappingTemplate;
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key: PrimaryKey, values: AttributeValues): MappingTemplate;
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload?: string): MappingTemplate;
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult(): MappingTemplate;
    /**
     * this is called to render the mapping template to a VTL string
     */
    abstract renderTemplate(): string;
}
/**
 * Basic properties for an AppSync resolver
 */
export interface BaseResolverProps {
    /**
     * name of the GraphQL type this resolver is attached to
     */
    readonly typeName: string;
    /**
     * name of the GraphQL fiel din the given type this resolver is attached to
     */
    readonly fieldName: string;
    /**
     * configuration of the pipeline resolver
     *
     * @default - create a UNIT resolver
     */
    readonly pipelineConfig?: CfnResolver.PipelineConfigProperty | IResolvable;
    /**
     * The request mapping template for this resolver
     *
     * @default - No mapping template
     */
    readonly requestMappingTemplate?: MappingTemplate;
    /**
     * The response mapping template for this resolver
     *
     * @default - No mapping template
     */
    readonly responseMappingTemplate?: MappingTemplate;
}
/**
 * Additional properties for an AppSync resolver like GraphQL API reference and datasource
 */
export interface ResolverProps extends BaseResolverProps {
    /**
     * The API this resolver is attached to
     */
    readonly api: GraphQLApi;
    /**
     * The data source this resolver is using
     *
     * @default - No datasource
     */
    readonly dataSource?: BaseDataSource;
}
/**
 * An AppSync resolver
 */
export declare class Resolver extends Construct {
    /**
     * the ARN of the resolver
     */
    readonly arn: string;
    private resolver;
    constructor(scope: Construct, id: string, props: ResolverProps);
}
