"""Core public API."""

import argparse
import dataclasses
from typing import Callable, Optional, Sequence, Type, TypeVar, Union, cast, overload

from typing_extensions import Literal, assert_never

from . import _argparse_formatter, _calling, _fields, _parsers, _strings

T = TypeVar("T")


def generate_parser(
    f: Callable[..., T],
    *,
    prog: Optional[str] = None,
    description: Optional[str] = None,
    args: Optional[Sequence[str]] = None,
    default_instance: Optional[T] = None,
    avoid_subparsers: bool = False,
) -> argparse.ArgumentParser:
    """Returns the argparse parser that would be used under-the-hood if `dcargs.cli()`
    was called with the same arguments.

    This can be useful for libraries like argcomplete, pyzshcomplete, or shtab, which
    enable autocompletion for argparse parsers."""
    # Potentially we could do some caching in the future, to reduce the redundant work
    # done when both `generate_parser()` and `cli()` are called.
    out = _cli_impl(
        "parser",
        f,
        prog=prog,
        description=description,
        args=args,
        default_instance=default_instance,
        avoid_subparsers=avoid_subparsers,
    )
    assert isinstance(out, argparse.ArgumentParser)
    return out


def cli(
    f: Callable[..., T],
    *,
    prog: Optional[str] = None,
    description: Optional[str] = None,
    args: Optional[Sequence[str]] = None,
    default_instance: Optional[T] = None,
    avoid_subparsers: bool = False,
) -> T:
    """Call `f(...)`, with arguments populated from an automatically generated CLI
    interface.

    `f` should have type-annotated inputs, and can be a function or class. Note that if
    `f` is a class, `dcargs.cli()` returns an instance.

    The parser is generated by populating helptext from docstrings and types from
    annotations; a broad range of core type annotations are supported...
    - Types natively accepted by `argparse`: str, int, float, pathlib.Path, etc.
    - Default values for optional parameters.
    - Booleans, which are automatically converted to flags when provided a default
      value.
    - Enums (via `enum.Enum`).
    - Various annotations from the standard typing library. Some examples:
      - `typing.ClassVar[T]`.
      - `typing.Optional[T]`.
      - `typing.Literal[T]`.
      - `typing.Sequence[T]`.
      - `typing.List[T]`.
      - `typing.Dict[K, V]`.
      - `typing.Tuple`, such as `typing.Tuple[T1, T2, T3]` or
        `typing.Tuple[T, ...]`.
      - `typing.Set[T]`.
      - `typing.Final[T]` and `typing.Annotated[T]`.
      - `typing.Union[T1, T2]`.
      - Various nested combinations of the above: `Optional[Literal[T]]`,
        `Final[Optional[Sequence[T]]]`, etc.
    - Hierarchical structures via nested dataclasses, TypedDict, NamedTuple,
      classes.
      - Simple nesting.
      - Unions over nested structures (subparsers).
      - Optional unions over nested structures (optional subparsers).
    - Generics (including nested generics).

    Args:
        f: Callable.
        prog: The name of the program printed in helptext. Mirrors argument from
            `argparse.ArgumentParser()`.
        description: Description text for the parser, displayed when the --help flag is
            passed in. If not specified, `f`'s docstring is used. Mirrors argument from
            `argparse.ArgumentParser()`.
        args: If set, parse arguments from a sequence of strings instead of the
            commandline. Mirrors argument from `argparse.ArgumentParser.parse_args()`.
        default_instance: An instance of `T` to use for default values; only supported
            if `T` is a dataclass, TypedDict, or NamedTuple. Helpful for merging CLI
            arguments with values loaded from elsewhere. (for example, a config object
            loaded from a yaml file)
        avoid_subparsers: Avoid creating a subparser when defaults are provided for
            unions over nested types. Generates cleaner but less expressive CLIs.

    Returns:
        The output of `f(...)`.
    """
    out = _cli_impl(
        "f_out",
        f,
        prog=prog,
        description=description,
        args=args,
        default_instance=default_instance,
        avoid_subparsers=avoid_subparsers,
    )
    return out


@overload
def _cli_impl(
    _return_stage: Literal["parser"],
    f: Callable[..., T],
    *,
    prog: Optional[str] = None,
    description: Optional[str] = None,
    args: Optional[Sequence[str]] = None,
    default_instance: Optional[T] = None,
    avoid_subparsers: bool = False,
) -> argparse.ArgumentParser:
    ...


@overload
def _cli_impl(
    _return_stage: Literal["f_out"],
    f: Callable[..., T],
    *,
    prog: Optional[str] = None,
    description: Optional[str] = None,
    args: Optional[Sequence[str]] = None,
    default_instance: Optional[T] = None,
    avoid_subparsers: bool = False,
) -> T:
    ...


def _cli_impl(
    _return_stage: Literal["parser", "f_out"],
    f: Callable[..., T],
    *,
    prog: Optional[str] = None,
    description: Optional[str] = None,
    args: Optional[Sequence[str]] = None,
    default_instance: Optional[T] = None,
    avoid_subparsers: bool = False,
) -> Union[T, argparse.ArgumentParser]:
    default_instance_internal: Union[_fields.NonpropagatingMissingType, T] = (
        _fields.MISSING_NONPROP if default_instance is None else default_instance
    )

    if not _fields.is_nested_type(cast(Type, f), default_instance_internal):
        dummy_field = dataclasses.field(
            default=default_instance
            if default_instance is not None
            else dataclasses.MISSING
        )
        f = dataclasses.make_dataclass(
            cls_name="",
            fields=[(_strings.dummy_field_name, cast(Type, f), dummy_field)],
        )
        dummy_wrapped = True
    else:
        dummy_wrapped = False

    # Map a callable to the relevant CLI arguments + subparsers.
    parser_definition = _parsers.ParserSpecification.from_callable(
        f,
        description=description,
        parent_classes=set(),  # Used for recursive calls.
        parent_type_from_typevar=None,  # Used for recursive calls.
        default_instance=default_instance_internal,  # Overrides for default values.
        prefix="",  # Used for recursive calls.
        avoid_subparsers=avoid_subparsers,
    )

    # Generate parser!
    with _argparse_formatter.ansi_context():
        parser = argparse.ArgumentParser(
            prog=prog, formatter_class=_argparse_formatter.ArgparseHelpFormatter
        )
        parser_definition.apply(parser)
        if _return_stage == "parser":
            return parser
        value_from_prefixed_field_name = vars(parser.parse_args(args=args))

    if dummy_wrapped:
        value_from_prefixed_field_name = {
            k.replace(_strings.dummy_field_name, ""): v
            for k, v in value_from_prefixed_field_name.items()
        }

    try:
        # Attempt to call `f` using whatever was passed in.
        out, consumed_keywords = _calling.call_from_args(
            f,
            parser_definition,
            default_instance_internal,
            value_from_prefixed_field_name,
            field_name_prefix="",
            avoid_subparsers=avoid_subparsers,
        )
    except _calling.InstantiationError as e:
        # Emulate argparse's error behavior when invalid arguments are passed in.
        parser.print_usage()
        print()
        print(e.args[0])
        raise SystemExit()

    assert len(value_from_prefixed_field_name.keys() - consumed_keywords) == 0, (
        f"Parsed {value_from_prefixed_field_name.keys()}, but only consumed"
        f" {consumed_keywords}"
    )

    if dummy_wrapped:
        out = getattr(out, _strings.dummy_field_name)
    if _return_stage == "f_out":
        return out

    assert_never(_return_stage)
