#! /usr/bin/python3
"""### Store lists of various imformation on blocks, biomes and more."""

# __all__ = []  # everything is available for import
__version__ = "v5.0"

import os
import sys

# to translate a string of regular names
# into the appropriate list of minecraft block IDs
# >>> def f(string):
# >>>     return ["minecraft:" + i.strip().lower().replace(" ", "_")
# >>>         for i in string.split(", ")]

# to translate a 255 RGB to hex RGB value
# >>> def f(r, g, b): return "0x"+(hex(r)+hex(g)+hex(b)).replace("0x", "")

# See https://minecraft.fandom.com/wiki/Data_version#List_of_data_versions
SUPPORTS = 2566  # Supported Minecraft version code

# all major Minecraft version codes
VERSIONS = {
    2860: '1.18', 2865: '1.18.1',

    2724: '1.17', 2730: '1.17.1',

    2566: '1.16', 2567: '1.16.1', 2578: '1.16.2', 2580: '1.16.3',
    2584: '1.16.4', 2586: '1.16.5',

    2225: '1.15', 2227: '1.15.1', 2230: '1.15.2',

    1952: '1.14', 1957: '1.14.1', 1963: '1.14.2', 1968: '1.14.3',
    1976: '1.14.4',

    1519: '1.13', 1628: '1.13.1', 1631: '1.13.2',

    1139: '1.12', 1241: '1.12.1', 1343: '1.12.2',

    819: '1.11', 921: '1.11.1', 922: '1.11.2',

    510: '1.10', 511: '1.10.1', 512: '1.10.2',

    169: '1.9', 175: '1.9.1', 176: '1.9.2', 183: '1.9.3', 184: '1.9.4',

    0: 'Pre-1.9'
}

# ========================================================= custom values


AXES = ('x', 'y', 'z')
DIRECTIONS = ('top', 'bottom', 'north', 'east', 'south', 'west')
INVERTDIRECTION = {'top': 'bottom', 'bottom': 'top',
                   'north': 'south', 'east': 'west',
                   'south': 'north', 'west': 'east'}
DIRECTION2VECTOR = {'top': (0, 1, 0), 'bottom': (0, -1, 0),
                    'north': (0, 0, -1), 'east': (1, 0, 0),
                    'south': (0, 1), 'west': (-1, 0)}
AXIS2VECTOR = {'x': (1, 0, 0), 'y': (0, 1, 0), 'z': (0, 0, 1)}
# ========================================================= materials

COLORS = ('white', 'orange', 'magenta', 'light_blue', 'yellow', 'lime', 'pink',
          'gray', 'light_gray', 'cyan', 'purple', 'blue', 'brown', 'green',
          'red', 'black')

WOODS = ('oak', 'birch', 'spruce', 'jungle', 'acacia', 'crimson', 'warped')

# ========================================================= blocks

# every single block in Minecraft (sorted alphabetically)
BLOCKS = ('minecraft:acacia_button', 'minecraft:acacia_door',
          'minecraft:acacia_fence', 'minecraft:acacia_fence_gate',
          'minecraft:acacia_leaves', 'minecraft:acacia_log',
          'minecraft:acacia_planks', 'minecraft:acacia_pressure_plate',
          'minecraft:acacia_sapling', 'minecraft:acacia_sign',
          'minecraft:acacia_slab', 'minecraft:acacia_stairs',
          'minecraft:acacia_trapdoor', 'minecraft:acacia_wall_sign',
          'minecraft:acacia_wood', 'minecraft:activator_rail', 'minecraft:air',
          'minecraft:allium', 'minecraft:ancient_debris', 'minecraft:andesite',
          'minecraft:andesite_slab', 'minecraft:andesite_stairs',
          'minecraft:andesite_wall', 'minecraft:anvil',
          'minecraft:azure_bluet', 'minecraft:bamboo',
          'minecraft:bamboo_sapling', 'minecraft:barrel', 'minecraft:barrier',
          'minecraft:basalt', 'minecraft:beacon', 'minecraft:bedrock',
          'minecraft:bee_nest', 'minecraft:beehive', 'minecraft:beetroots',
          'minecraft:bell', 'minecraft:birch_button', 'minecraft:birch_door',
          'minecraft:birch_fence', 'minecraft:birch_fence_gate',
          'minecraft:birch_leaves', 'minecraft:birch_log',
          'minecraft:birch_planks', 'minecraft:birch_pressure_plate',
          'minecraft:birch_sapling', 'minecraft:birch_sign',
          'minecraft:birch_slab', 'minecraft:birch_stairs',
          'minecraft:birch_trapdoor', 'minecraft:birch_wall_sign',
          'minecraft:birch_wood', 'minecraft:black_banner',
          'minecraft:black_bed', 'minecraft:black_carpet',
          'minecraft:black_concrete', 'minecraft:black_concrete_powder',
          'minecraft:black_glazed_terracotta', 'minecraft:black_shulker_box',
          'minecraft:black_stained_glass',
          'minecraft:black_stained_glass_pane', 'minecraft:black_terracotta',
          'minecraft:black_wool', 'minecraft:blackstone',
          'minecraft:blackstone_slab', 'minecraft:blackstone_stairs',
          'minecraft:blackstone_wall', 'minecraft:blast_furnace',
          'minecraft:blue_banner', 'minecraft:blue_bed',
          'minecraft:blue_carpet', 'minecraft:blue_concrete',
          'minecraft:blue_concrete_powder', 'minecraft:blue_glazed_terracotta',
          'minecraft:blue_ice', 'minecraft:blue_orchid',
          'minecraft:blue_shulker_box', 'minecraft:blue_stained_glass',
          'minecraft:blue_stained_glass_pane', 'minecraft:blue_terracotta',
          'minecraft:blue_wool', 'minecraft:bone_block', 'minecraft:bookshelf',
          'minecraft:brain_coral', 'minecraft:brain_coral_block',
          'minecraft:brain_coral_fan', 'minecraft:brewing_stand',
          'minecraft:brick_slab', 'minecraft:brick_stairs',
          'minecraft:brick_wall', 'minecraft:bricks', 'minecraft:brown_banner',
          'minecraft:brown_bed', 'minecraft:brown_carpet',
          'minecraft:brown_concrete', 'minecraft:brown_concrete_powder',
          'minecraft:brown_glazed_terracotta', 'minecraft:brown_mushroom',
          'minecraft:brown_mushroom_block', 'minecraft:brown_shulker_box',
          'minecraft:brown_stained_glass',
          'minecraft:brown_stained_glass_pane', 'minecraft:brown_terracotta',
          'minecraft:brown_wool', 'minecraft:bubble_column',
          'minecraft:bubble_coral', 'minecraft:bubble_coral_block',
          'minecraft:bubble_coral_fan', 'minecraft:cactus', 'minecraft:cake',
          'minecraft:campfire', 'minecraft:carrots',
          'minecraft:cartography_table', 'minecraft:carved_pumpkin',
          'minecraft:cauldron', 'minecraft:cave_air', 'minecraft:chain',
          'minecraft:chain_command_block', 'minecraft:chest',
          'minecraft:chipped_anvil', 'minecraft:chiseled_nether_bricks',
          'minecraft:chiseled_quartz_block',
          'minecraft:chiseled_red_sandstone', 'minecraft:chiseled_sandstone',
          'minecraft:chiseled_stone_bricks', 'minecraft:chorus_flower',
          'minecraft:chorus_plant', 'minecraft:clay', 'minecraft:coal_block',
          'minecraft:coal_ore', 'minecraft:coarse_dirt',
          'minecraft:cobblestone', 'minecraft:cobblestone_slab',
          'minecraft:cobblestone_stairs', 'minecraft:cobblestone_wall',
          'minecraft:cobweb', 'minecraft:cocoa', 'minecraft:command_block',
          'minecraft:comparator', 'minecraft:composter', 'minecraft:conduit',
          'minecraft:cornflower', 'minecraft:cracked_nether_bricks',
          'minecraft:cracked_stone_bricks', 'minecraft:crafting_table',
          'minecraft:creeper_head', 'minecraft:creeper_wall_head',
          'minecraft:crimson_button', 'minecraft:crimson_door',
          'minecraft:crimson_fence', 'minecraft:crimson_fence_gate',
          'minecraft:crimson_fungus', 'minecraft:crimson_hyphae',
          'minecraft:crimson_nylium', 'minecraft:crimson_planks',
          'minecraft:crimson_pressure_plate', 'minecraft:crimson_roots',
          'minecraft:crimson_sign', 'minecraft:crimson_slab',
          'minecraft:crimson_stairs', 'minecraft:crimson_stem',
          'minecraft:crimson_trapdoor', 'minecraft:crimson_wall_sign',
          'minecraft:crying_obsidian', 'minecraft:cut_red_sandstone',
          'minecraft:cut_red_sandstone_slab', 'minecraft:cut_sandstone',
          'minecraft:cut_sandstone_slab', 'minecraft:cyan_banner',
          'minecraft:cyan_bed', 'minecraft:cyan_carpet',
          'minecraft:cyan_concrete', 'minecraft:cyan_concrete_powder',
          'minecraft:cyan_glazed_terracotta', 'minecraft:cyan_shulker_box',
          'minecraft:cyan_stained_glass', 'minecraft:cyan_stained_glass_pane',
          'minecraft:cyan_terracotta', 'minecraft:cyan_wool',
          'minecraft:damaged_anvil', 'minecraft:dandelion',
          'minecraft:dark_oak_button', 'minecraft:dark_oak_door',
          'minecraft:dark_oak_fence', 'minecraft:dark_oak_fence_gate',
          'minecraft:dark_oak_leaves', 'minecraft:dark_oak_log',
          'minecraft:dark_oak_planks', 'minecraft:dark_oak_pressure_plate',
          'minecraft:dark_oak_sapling', 'minecraft:dark_oak_sign',
          'minecraft:dark_oak_slab', 'minecraft:dark_oak_stairs',
          'minecraft:dark_oak_trapdoor', 'minecraft:dark_oak_wall_sign',
          'minecraft:dark_oak_wood', 'minecraft:dark_prismarine',
          'minecraft:dark_prismarine_slab', 'minecraft:dark_prismarine_stairs',
          'minecraft:daylight_detector', 'minecraft:dead_brain_coral',
          'minecraft:dead_brain_coral_block', 'minecraft:dead_brain_coral_fan',
          'minecraft:dead_bubble_coral', 'minecraft:dead_bubble_coral_block',
          'minecraft:dead_bubble_coral_fan', 'minecraft:dead_bush',
          'minecraft:dead_fire_coral', 'minecraft:dead_fire_coral_block',
          'minecraft:dead_fire_coral_fan', 'minecraft:dead_horn_coral',
          'minecraft:dead_horn_coral_block', 'minecraft:dead_horn_coral_fan',
          'minecraft:dead_tube_coral', 'minecraft:dead_tube_coral_block',
          'minecraft:dead_tube_coral_fan', 'minecraft:detector_rail',
          'minecraft:diamond_block', 'minecraft:diamond_ore',
          'minecraft:diorite', 'minecraft:diorite_slab',
          'minecraft:diorite_stairs', 'minecraft:diorite_wall',
          'minecraft:dirt', 'minecraft:dispenser', 'minecraft:dragon_egg',
          'minecraft:dragon_head', 'minecraft:dragon_wall_head',
          'minecraft:dried_kelp_block', 'minecraft:dropper',
          'minecraft:emerald_block', 'minecraft:emerald_ore',
          'minecraft:enchanting_table', 'minecraft:end_gateway',
          'minecraft:end_portal', 'minecraft:end_portal_frame',
          'minecraft:end_rod', 'minecraft:end_stone',
          'minecraft:end_stone_brick_slab', 'minecraft:end_stone_brick_stairs',
          'minecraft:end_stone_brick_wall', 'minecraft:end_stone_bricks',
          'minecraft:ender_chest', 'minecraft:farmland', 'minecraft:fern',
          'minecraft:fire', 'minecraft:fire_coral',
          'minecraft:fire_coral_block', 'minecraft:fire_coral_fan',
          'minecraft:fletching_table', 'minecraft:flower_pot',
          'minecraft:frosted_ice', 'minecraft:furnace',
          'minecraft:gilded_blackstone', 'minecraft:glass',
          'minecraft:glass_pane', 'minecraft:glowstone',
          'minecraft:gold_block', 'minecraft:gold_ore', 'minecraft:granite',
          'minecraft:granite_slab', 'minecraft:granite_stairs',
          'minecraft:granite_wall', 'minecraft:grass', 'minecraft:grass_block',
          'minecraft:grass_path', 'minecraft:gravel', 'minecraft:gray_banner',
          'minecraft:gray_bed', 'minecraft:gray_carpet',
          'minecraft:gray_concrete', 'minecraft:gray_concrete_powder',
          'minecraft:gray_glazed_terracotta', 'minecraft:gray_shulker_box',
          'minecraft:gray_stained_glass', 'minecraft:gray_stained_glass_pane',
          'minecraft:gray_terracotta', 'minecraft:gray_wool',
          'minecraft:green_banner', 'minecraft:green_bed',
          'minecraft:green_carpet', 'minecraft:green_concrete',
          'minecraft:green_concrete_powder',
          'minecraft:green_glazed_terracotta', 'minecraft:green_shulker_box',
          'minecraft:green_stained_glass',
          'minecraft:green_stained_glass_pane', 'minecraft:green_terracotta',
          'minecraft:green_wool', 'minecraft:grindstone',
          'minecraft:hay_block', 'minecraft:heavy_weighted_pressure_plate',
          'minecraft:honey_block', 'minecraft:honeycomb_block',
          'minecraft:hopper', 'minecraft:horn_coral',
          'minecraft:horn_coral_block', 'minecraft:horn_coral_fan',
          'minecraft:ice', 'minecraft:infested_chiseled_stone_bricks',
          'minecraft:infested_cobblestone',
          'minecraft:infested_cracked_stone_bricks',
          'minecraft:infested_mossy_stone_bricks', 'minecraft:infested_stone',
          'minecraft:infested_stone_bricks', 'minecraft:iron_bars',
          'minecraft:iron_block', 'minecraft:iron_door', 'minecraft:iron_ore',
          'minecraft:iron_trapdoor', 'minecraft:jack_o_lantern',
          'minecraft:jigsaw', 'minecraft:jukebox', 'minecraft:jungle_button',
          'minecraft:jungle_door', 'minecraft:jungle_fence',
          'minecraft:jungle_fence_gate', 'minecraft:jungle_leaves',
          'minecraft:jungle_log', 'minecraft:jungle_planks',
          'minecraft:jungle_pressure_plate', 'minecraft:jungle_sapling',
          'minecraft:jungle_sign', 'minecraft:jungle_slab',
          'minecraft:jungle_stairs', 'minecraft:jungle_trapdoor',
          'minecraft:jungle_wall_sign', 'minecraft:jungle_wood',
          'minecraft:kelp', 'minecraft:ladder', 'minecraft:lantern',
          'minecraft:lapis_block', 'minecraft:lapis_ore',
          'minecraft:large_fern', 'minecraft:lava', 'minecraft:lectern',
          'minecraft:lever', 'minecraft:light_blue_banner',
          'minecraft:light_blue_bed', 'minecraft:light_blue_carpet',
          'minecraft:light_blue_concrete',
          'minecraft:light_blue_concrete_powder',
          'minecraft:light_blue_glazed_terracotta',
          'minecraft:light_blue_shulker_box',
          'minecraft:light_blue_stained_glass',
          'minecraft:light_blue_stained_glass_pane',
          'minecraft:light_blue_terracotta', 'minecraft:light_blue_wool',
          'minecraft:light_gray_banner', 'minecraft:light_gray_bed',
          'minecraft:light_gray_carpet', 'minecraft:light_gray_concrete',
          'minecraft:light_gray_concrete_powder',
          'minecraft:light_gray_glazed_terracotta',
          'minecraft:light_gray_shulker_box',
          'minecraft:light_gray_stained_glass',
          'minecraft:light_gray_stained_glass_pane',
          'minecraft:light_gray_terracotta', 'minecraft:light_gray_wool',
          'minecraft:light_weighted_pressure_plate', 'minecraft:lilac',
          'minecraft:lily_of_the_valley', 'minecraft:lily_pad',
          'minecraft:lime_banner', 'minecraft:lime_bed',
          'minecraft:lime_carpet', 'minecraft:lime_concrete',
          'minecraft:lime_concrete_powder', 'minecraft:lime_glazed_terracotta',
          'minecraft:lime_shulker_box', 'minecraft:lime_stained_glass',
          'minecraft:lime_stained_glass_pane', 'minecraft:lime_terracotta',
          'minecraft:lime_wool', 'minecraft:lodestone', 'minecraft:loom',
          'minecraft:magenta_banner', 'minecraft:magenta_bed',
          'minecraft:magenta_carpet', 'minecraft:magenta_concrete',
          'minecraft:magenta_concrete_powder',
          'minecraft:magenta_glazed_terracotta',
          'minecraft:magenta_shulker_box', 'minecraft:magenta_stained_glass',
          'minecraft:magenta_stained_glass_pane',
          'minecraft:magenta_terracotta', 'minecraft:magenta_wool',
          'minecraft:magma_block', 'minecraft:melon', 'minecraft:melon_stem',
          'minecraft:mossy_cobblestone', 'minecraft:mossy_cobblestone_slab',
          'minecraft:mossy_cobblestone_stairs',
          'minecraft:mossy_cobblestone_wall',
          'minecraft:mossy_stone_brick_slab',
          'minecraft:mossy_stone_brick_stairs',
          'minecraft:mossy_stone_brick_wall', 'minecraft:mossy_stone_bricks',
          'minecraft:mushroom_stem', 'minecraft:mycelium',
          'minecraft:nether_brick_fence', 'minecraft:nether_brick_slab',
          'minecraft:nether_brick_stairs', 'minecraft:nether_brick_wall',
          'minecraft:nether_bricks', 'minecraft:nether_gold_ore',
          'minecraft:nether_portal', 'minecraft:nether_quartz_ore',
          'minecraft:nether_sprouts', 'minecraft:nether_wart',
          'minecraft:nether_wart_block', 'minecraft:netherite_block',
          'minecraft:netherrack', 'minecraft:note_block',
          'minecraft:oak_button', 'minecraft:oak_door', 'minecraft:oak_fence',
          'minecraft:oak_fence_gate', 'minecraft:oak_leaves',
          'minecraft:oak_log', 'minecraft:oak_planks',
          'minecraft:oak_pressure_plate', 'minecraft:oak_sapling',
          'minecraft:oak_sign', 'minecraft:oak_slab', 'minecraft:oak_stairs',
          'minecraft:oak_trapdoor', 'minecraft:oak_wall_sign',
          'minecraft:oak_wood', 'minecraft:observer', 'minecraft:obsidian',
          'minecraft:orange_banner', 'minecraft:orange_bed',
          'minecraft:orange_carpet', 'minecraft:orange_concrete',
          'minecraft:orange_concrete_powder',
          'minecraft:orange_glazed_terracotta', 'minecraft:orange_shulker_box',
          'minecraft:orange_stained_glass',
          'minecraft:orange_stained_glass_pane', 'minecraft:orange_terracotta',
          'minecraft:orange_tulip', 'minecraft:orange_wool',
          'minecraft:oxeye_daisy', 'minecraft:packed_ice', 'minecraft:peony',
          'minecraft:petrified_oak_slab', 'minecraft:pink_banner',
          'minecraft:pink_bed', 'minecraft:pink_carpet',
          'minecraft:pink_concrete', 'minecraft:pink_concrete_powder',
          'minecraft:pink_glazed_terracotta', 'minecraft:pink_shulker_box',
          'minecraft:pink_stained_glass', 'minecraft:pink_stained_glass_pane',
          'minecraft:pink_terracotta', 'minecraft:pink_tulip',
          'minecraft:pink_wool', 'minecraft:piston', 'minecraft:piston_head',
          'minecraft:player_head', 'minecraft:player_wall_head',
          'minecraft:podzol', 'minecraft:polished_andesite',
          'minecraft:polished_andesite_slab',
          'minecraft:polished_andesite_stairs', 'minecraft:polished_basalt',
          'minecraft:polished_blackstone',
          'minecraft:polished_blackstone_brick_slab',
          'minecraft:polished_blackstone_brick_stairs',
          'minecraft:polished_blackstone_brick_wall',
          'minecraft:polished_blackstone_bricks',
          'minecraft:polished_blackstone_button',
          'minecraft:polished_blackstone_slab',
          'minecraft:polished_blackstone_stairs',
          'minecraft:polished_blackstone_wall',
          'minecraft:polished_diorite', 'minecraft:polished_diorite_slab',
          'minecraft:polished_diorite_stairs', 'minecraft:polished_granite',
          'minecraft:polished_granite_slab',
          'minecraft:polished_granite_stairs', 'minecraft:poppy',
          'minecraft:potatoes', 'minecraft:powered_rail',
          'minecraft:prismarine_brick_slab',
          'minecraft:prismarine_brick_stairs', 'minecraft:prismarine_bricks',
          'minecraft:prismarine_slab', 'minecraft:prismarine_stairs',
          'minecraft:prismarine_wall', 'minecraft:pumpkin',
          'minecraft:pumpkin_stem', 'minecraft:purple_banner',
          'minecraft:purple_bed', 'minecraft:purple_carpet',
          'minecraft:purple_concrete', 'minecraft:purple_concrete_powder',
          'minecraft:purple_glazed_terracotta', 'minecraft:purple_shulker_box',
          'minecraft:purple_stained_glass',
          'minecraft:purple_stained_glass_pane', 'minecraft:purple_terracotta',
          'minecraft:purple_wool', 'minecraft:purpur_block',
          'minecraft:purpur_pillar', 'minecraft:purpur_slab',
          'minecraft:purpur_stairs', 'minecraft:quartz_block',
          'minecraft:quartz_bricks', 'minecraft:quartz_pillar',
          'minecraft:quartz_slab', 'minecraft:quartz_stairs', 'minecraft:rail',
          'minecraft:red_banner', 'minecraft:red_bed', 'minecraft:red_carpet',
          'minecraft:red_concrete', 'minecraft:red_concrete_powder',
          'minecraft:red_glazed_terracotta', 'minecraft:red_mushroom',
          'minecraft:red_mushroom_block', 'minecraft:red_nether_brick_slab',
          'minecraft:red_nether_brick_stairs',
          'minecraft:red_nether_brick_wall', 'minecraft:red_nether_bricks',
          'minecraft:red_sand', 'minecraft:red_sandstone',
          'minecraft:red_sandstone_slab', 'minecraft:red_sandstone_stairs',
          'minecraft:red_sandstone_wall', 'minecraft:red_shulker_box',
          'minecraft:red_stained_glass', 'minecraft:red_stained_glass_pane',
          'minecraft:red_terracotta', 'minecraft:red_tulip',
          'minecraft:red_wool', 'minecraft:redstone_block',
          'minecraft:redstone_lamp', 'minecraft:redstone_ore',
          'minecraft:redstone_torch', 'minecraft:redstone_wire',
          'minecraft:repeater', 'minecraft:repeating_command_block',
          'minecraft:respawn_anchor', 'minecraft:rose_bush', 'minecraft:sand',
          'minecraft:sandstone', 'minecraft:sandstone_slab',
          'minecraft:sandstone_stairs', 'minecraft:sandstone_wall',
          'minecraft:scaffolding', 'minecraft:sea_lantern',
          'minecraft:sea_pickle', 'minecraft:seagrass',
          'minecraft:shroomlight', 'minecraft:shulker_box',
          'minecraft:skeleton_skull', 'minecraft:skeleton_wall_skull',
          'minecraft:slime_block', 'minecraft:smithing_table',
          'minecraft:smoker', 'minecraft:smooth_quartz',
          'minecraft:smooth_quartz_slab', 'minecraft:smooth_quartz_stairs',
          'minecraft:smooth_red_sandstone',
          'minecraft:smooth_red_sandstone_slab',
          'minecraft:smooth_red_sandstone_stairs',
          'minecraft:smooth_sandstone', 'minecraft:smooth_sandstone_slab',
          'minecraft:smooth_sandstone_stairs', 'minecraft:smooth_stone',
          'minecraft:smooth_stone_slab', 'minecraft:snow',
          'minecraft:snow_block', 'minecraft:soul_campfire',
          'minecraft:soul_fire', 'minecraft:soul_lantern',
          'minecraft:soul_sand', 'minecraft:soul_soil', 'minecraft:soul_torch',
          'minecraft:spawner', 'minecraft:sponge', 'minecraft:spruce_button',
          'minecraft:spruce_door', 'minecraft:spruce_fence',
          'minecraft:spruce_fence_gate', 'minecraft:spruce_leaves',
          'minecraft:spruce_log', 'minecraft:spruce_planks',
          'minecraft:spruce_pressure_plate', 'minecraft:spruce_sapling',
          'minecraft:spruce_sign', 'minecraft:spruce_slab',
          'minecraft:spruce_stairs', 'minecraft:spruce_trapdoor',
          'minecraft:spruce_wall_sign', 'minecraft:spruce_wood',
          'minecraft:sticky_piston', 'minecraft:stone',
          'minecraft:stone_brick_slab', 'minecraft:stone_brick_stairs',
          'minecraft:stone_brick_wall', 'minecraft:stone_bricks',
          'minecraft:stone_button', 'minecraft:stone_pressure_plate',
          'minecraft:stone_slab', 'minecraft:stonecutter',
          'minecraft:stripped_acacia_log', 'minecraft:stripped_acacia_wood',
          'minecraft:stripped_birch_log', 'minecraft:stripped_birch_wood',
          'minecraft:stripped_crimson_hyphae',
          'minecraft:stripped_crimson_stem', 'minecraft:stripped_dark_oak_log',
          'minecraft:stripped_dark_oak_wood', 'minecraft:stripped_jungle_log',
          'minecraft:stripped_jungle_wood', 'minecraft:stripped_oak_log',
          'minecraft:stripped_oak_wood', 'minecraft:stripped_spruce_log',
          'minecraft:stripped_spruce_wood', 'minecraft:stripped_warped_hyphae',
          'minecraft:stripped_warped_stem', 'minecraft:structure_block',
          'minecraft:structure_void', 'minecraft:sugar_cane',
          'minecraft:sunflower', 'minecraft:sweet_berry_bush',
          'minecraft:tall_grass', 'minecraft:tall_seagrass',
          'minecraft:target', 'minecraft:terracotta', 'minecraft:tnt',
          'minecraft:torch', 'minecraft:trapped_chest', 'minecraft:tripwire',
          'minecraft:tripwire_hook', 'minecraft:tube_coral',
          'minecraft:tube_coral_block', 'minecraft:tube_coral_fan',
          'minecraft:turtle_egg', 'minecraft:twisting_vines', 'minecraft:vine',
          'minecraft:void_air', 'minecraft:warped_button',
          'minecraft:warped_door', 'minecraft:warped_fence',
          'minecraft:warped_fence_gate', 'minecraft:warped_fungus',
          'minecraft:warped_hyphae', 'minecraft:warped_nylium',
          'minecraft:warped_planks', 'minecraft:warped_pressure_plate',
          'minecraft:warped_roots', 'minecraft:warped_sign',
          'minecraft:warped_slab', 'minecraft:warped_stairs',
          'minecraft:warped_stem', 'minecraft:warped_trapdoor',
          'minecraft:warped_wall_sign', 'minecraft:warped_wart_block',
          'minecraft:water', 'minecraft:weeping_vines', 'minecraft:wet_sponge',
          'minecraft:wheat', 'minecraft:white_banner', 'minecraft:white_bed',
          'minecraft:white_carpet', 'minecraft:white_concrete',
          'minecraft:white_concrete_powder',
          'minecraft:white_glazed_terracotta', 'minecraft:white_shulker_box',
          'minecraft:white_stained_glass',
          'minecraft:white_stained_glass_pane', 'minecraft:white_terracotta',
          'minecraft:white_tulip', 'minecraft:white_wool',
          'minecraft:wither_rose', 'minecraft:wither_skeleton_skull',
          'minecraft:wither_skeleton_wall_skull', 'minecraft:yellow_banner',
          'minecraft:yellow_bed', 'minecraft:yellow_carpet',
          'minecraft:yellow_concrete', 'minecraft:yellow_concrete_powder',
          'minecraft:yellow_glazed_terracotta', 'minecraft:yellow_shulker_box',
          'minecraft:yellow_stained_glass',
          'minecraft:yellow_stained_glass_pane', 'minecraft:yellow_terracotta',
          'minecraft:yellow_wool', 'minecraft:zombie_head',
          'minecraft:zombie_wall_head')

# ========================================================= grouped by model

AIR = ('minecraft:air', 'minecraft:void_air', 'minecraft:cave_air')

# flora
LEAVES = ('minecraft:oak_leaves', 'minecraft:birch_leaves',
          'minecraft:spruce_leaves', 'minecraft:jungle_leaves',
          'minecraft:dark_oak_leaves', 'minecraft:acacia_leaves',)
SHORTFLOWERS = ('minecraft:dandelion', 'minecraft:poppy',
                'minecraft:blue_orchid', 'minecraft:allium',
                'minecraft:azure_bluet', 'minecraft:red_tulip',
                'minecraft:pink_tulip', 'minecraft:white_tulip',
                'minecraft:orange_tulip', 'minecraft:oxeye_daisy',
                'minecraft:cornflower', 'minecraft:lily_of_the_valley',
                'minecraft:wither_rose')
TALLFLOWERS = ('minecraft:sunflower', 'minecraft:lilac',
               'minecraft:rose_bush', 'minecraft:peony')
CROPS = ('minecraft:wheat', 'minecraft:beetroots',
         'minecraft:carrots', 'minecraft:potatoes',
         'minecraft:melon_stem', 'minecraft:pumpkin_stem')
FUNGI = ('minecraft:red_mushroom', 'minecraft:brown_mushroom',
         'minecraft:warped_fungus', 'minecraft:crimson_fungus')
WILDCROPS = ('minecraft:cocoa', 'minecraft:sweet_berry_bush')
TALLCROPS = ('minecraft:bamboo', 'minecraft:sugar_cane',
             'minecraft:cactus')
GRASSES = ('minecraft:grass', 'minecraft:tall_grass',
           'minecraft:fern', 'minecraft:large_fern')
VINES = ('minecraft:vine',
         'minecraft:weeping_vines', 'minecraft:weeping_vines')

FLOWERS = SHORTFLOWERS + TALLFLOWERS
TALLPLANTS = TALLCROPS + TALLFLOWERS
PLANTS = ('minecraft:dead_bush',) + FLOWERS + CROPS + FUNGI + WILDCROPS \
    + TALLCROPS + GRASSES + VINES

# landscape
# TODO: LIQUIDS
# TODO: SOILS
# TODO: STONES
# TODO: ORES


# construction
STAIRS = ('minecraft:oak_stairs', 'minecraft:spruce_stairs',
          'minecraft:birch_stairs', 'minecraft:jungle_stairs',
          'minecraft:acacia_stairs', 'minecraft:dark_oak_stairs',
          'minecraft:crimson_stairs', 'minecraft:warped_stairs',
          'minecraft:stone_stairs',
          'minecraft:granite_stairs', 'minecraft:polished_granite_stairs',
          'minecraft:diorite_stairs', 'minecraft:polished_diorite_stairs',
          'minecraft:andesite_stairs', 'minecraft:polished_andesite_stairs',
          'minecraft:cobblestone_stairs', 'minecraft:mossy_cobblestone_stairs',
          'minecraft:stone_brick_stairs', 'minecraft:mossy_stone_brick_stairs',
          'minecraft:brick_stairs', 'minecraft:end_stone_brick_stairs',
          'minecraft:nether_brick_stairs', 'minecraft:red_nether_brick_stairs',
          'minecraft:sandstone_stairs', 'minecraft:cut_sandstone_stairs',
          'minecraft:smooth_sandstone_stairs',
          'minecraft:red_sandstone_stairs',
          'minecraft:cut_red_sandstone_stairs',
          'minecraft:smooth_red_sandstone_stairs',
          'minecraft:quartz_stairs', 'minecraft:smooth_quartz_stairs',
          'minecraft:purpur_stairs',
          'minecraft:prismarine_stairs', 'minecraft:prismarine_brick_stairs',
          'minecraft:dark_prismarine_stairs',
          'minecraft:petrified_oak_stairs',
          'minecraft:blackstone_stairs',
          'minecraft:polished_blackstone_stairs',
          'minecraft:polished_blackstone_brick_stairs')
SLABS = ('minecraft:oak_slab', 'minecraft:spruce_slab',
         'minecraft:birch_slab', 'minecraft:jungle_slab',
         'minecraft:acacia_slab', 'minecraft:dark_oak_slab',
         'minecraft:crimson_slab', 'minecraft:warped_slab',
         'minecraft:stone_slab', 'minecraft:smooth_stone_slab',
         'minecraft:granite_slab', 'minecraft:polished_granite_slab',
         'minecraft:diorite_slab', 'minecraft:polished_diorite_slab',
         'minecraft:andesite_slab', 'minecraft:polished_andesite_slab',
         'minecraft:cobblestone_slab', 'minecraft:mossy_cobblestone_slab',
         'minecraft:stone_brick_slab', 'minecraft:mossy_stone_brick_slab',
         'minecraft:brick_slab', 'minecraft:end_stone_brick_slab',
         'minecraft:nether_brick_slab', 'minecraft:red_nether_brick_slab',
         'minecraft:sandstone_slab', 'minecraft:cut_sandstone_slab',
         'minecraft:smooth_sandstone_slab',
         'minecraft:red_sandstone_slab', 'minecraft:cut_red_sandstone_slab',
         'minecraft:smooth_red_sandstone_slab',
         'minecraft:quartz_slab', 'minecraft:smooth_quartz_slab',
         'minecraft:purpur_slab',
         'minecraft:prismarine_slab', 'minecraft:prismarine_brick_slab',
         'minecraft:dark_prismarine_slab',
         'minecraft:petrified_oak_slab',
         'minecraft:blackstone_slab', 'minecraft:polished_blackstone_slab',
         'minecraft:polished_blackstone_brick_slab')
FENCES = ('minecraft:oak_fence', 'minecraft:birch_fence',
          'minecraft:spruce_fence', 'minecraft:jungle_fence',
          'minecraft:dark_oak_fence', 'minecraft:acacia_fence',
          'minecraft:warped_fence', 'minecraft:crimson_fence')
WALLS = ('minecraft:cobblestone_wall', 'minecraft:mossy_cobblestone_wall',
         'minecraft:brick_wall', 'minecraft:prismarine_wall',
         'minecraft:red_sandstone_wall', 'minecraft:mossy_stone_brick_wall',
         'minecraft:granite_wall', 'minecraft:stone_brick_wall',
         'minecraft:nether_brick_wall', 'minecraft:andesite_wall',
         'minecraft:red_nether_brick_wall', 'minecraft:sandstone_wall',
         'minecraft:end_stone_brick_wall', 'minecraft:diorite_wall',
         'minecraft:blackstone_wall', 'minecraft:polished_blackstone_wall',
         'minecraft:polished_blackstone_brick_wall',
         'minecraft:cobbled_deepslate_wall', 'minecraft:deepslate_tile_wall',
         'minecraft:polished_deepslate_wall')
GLASS = ('minecraft:glass', 'minecraft:glass_pane',
         'minecraft:white_stained_glass',
         'minecraft:white_stained_glass_pane',
         'minecraft:orange_stained_glass',
         'minecraft:orange_stained_glass_pane',
         'minecraft:magenta_stained_glass',
         'minecraft:magenta_stained_glass_pane',
         'minecraft:light_blue_stained_glass',
         'minecraft:light_blue_stained_glass_pane',
         'minecraft:yellow_stained_glass',
         'minecraft:yellow_stained_glass_pane',
         'minecraft:lime_stained_glass',
         'minecraft:lime_stained_glass_pane',
         'minecraft:pink_stained_glass',
         'minecraft:pink_stained_glass_pane',
         'minecraft:gray_stained_glass',
         'minecraft:gray_stained_glass_pane',
         'minecraft:light_gray_stained_glass',
         'minecraft:light_gray_stained_glass_pane',
         'minecraft:cyan_stained_glass',
         'minecraft:cyan_stained_glass_pane',
         'minecraft:purple_stained_glass',
         'minecraft:purple_stained_glass_pane',
         'minecraft:blue_stained_glass',
         'minecraft:blue_stained_glass_pane',
         'minecraft:brown_stained_glass',
         'minecraft:brown_stained_glass_pane',
         'minecraft:green_stained_glass',
         'minecraft:green_stained_glass_pane',
         'minecraft:red_stained_glass',
         'minecraft:red_stained_glass_pane',
         'minecraft:black_stained_glass',
         'minecraft: black_stained_glass_pane')

# lighting
FIRES = ('minecraft:fire', 'minecraft:soul_fire')
CAMPFIRES = ('minecraft:campfire', 'minecraft:soul_campfire')
TORCHES = ('minecraft:torch', 'minecraft:soul_torch',
           'minecraft:redstone_torch')
LANTERNS = ('minecraft:lantern', 'minecraft:soul_lantern')
LIGHTBLOCKS = ('minecraft:beacon', 'minecraft:redstone_lamp',
               'minecraft:jack_o_lantern', 'minecraft:sea_lantern',
               'minecraft:glowstone')

LIGHTSOURCES = FIRES + CAMPFIRES + TORCHES + LANTERNS + LIGHTBLOCKS

# mechanical
WIRING = ('minecraft:redstone_wire',
          'minecraft:repeater', 'minecraft:comparator')
RAILS = ('minecraft:rail', 'minecraft:powered_rail',
         'minecraft:detector_rail', 'minecraft:activator_rail',)
SWITCHES = ('minecraft:lever',
            'minecraft:stone_button', 'minecraft:oak_button',
            'minecraft:spruce_button', 'minecraft:birch_button',
            'minecraft:jungle_button', 'minecraft:acacia_button',
            'minecraft:dark_oak_button',
            'minecraft:crimson_button', 'minecraft:warped_button',
            'minecraft:polished_blackstone_button',
            'minecraft:tripwire_hook')
PRESSUREPLATES = ('minecraft:crimson_pressure_plate',
                  'minecraft:warped_pressure_plate',
                  'minecraft:polished_blackstone_pressure_plate',
                  'minecraft:light_weighted_pressure_plate',
                  'minecraft:heavy_weighted_pressure_plate',
                  'minecraft:stone_pressure_plate',
                  'minecraft:oak_pressure_plate',
                  'minecraft:wooden_pressure_plate',
                  'minecraft:spruce_pressure_plate',
                  'minecraft:birch_pressure_plate',
                  'minecraft:jungle_pressure_plate',
                  'minecraft:acacia_pressure_plate',
                  'minecraft:dark_oak_pressure_plate')
DOORS = ('minecraft:iron_door', 'minecraft:oak_door',
         'minecraft:wooden_door', 'minecraft:spruce_door',
         'minecraft:birch_door', 'minecraft:jungle_door',
         'minecraft:acacia_door', 'minecraft:dark_oak_door',
         'minecraft:crimson_door', 'minecraft:warped_door')
TRAPDOORS = ('minecraft:oak_trapdoor', 'minecraft:trapdoor',
             'minecraft:spruce_trapdoor', 'minecraft:birch_trapdoor',
             'minecraft:jungle_trapdoor', 'minecraft:acacia_trapdoor',
             'minecraft:dark_oak_trapdoor', 'minecraft:iron_trapdoor')
GATES = ('minecraft:oak_fence_gate', 'minecraft:birch_fence_gate',
         'minecraft:spruce_fence_gate', 'minecraft:jungle_fence_gate',
         'minecraft:dark_oak_fence_gate', 'minecraft:acacia_fence_gate',
         'minecraft:warped_fence_gate', 'minecraft:crimson_fence_gate')

# decoration
FLOORHEADS = ('minecraft:skeleton_skull',
              'minecraft:wither_skeleton_skull',
              'minecraft:zombie_head', 'minecraft:player_head',
              'minecraft:creeper_head', 'minecraft:dragon_head')
WALLHEADS = ('minecraft:skeleton_wall_skull',
             'minecraft:wither_skeleton_wall_skull',
             'minecraft:zombie_wall_head',
             'minecraft:player_wall_head', 'minecraft:creeper_wall_head',
             'minecraft:dragon_wall_head',)
HEADS = FLOORHEADS + WALLHEADS

FLOORSIGNS = ('minecraft:oak_sign', 'minecraft:birch_sign',
              'minecraft:spruce_sign', 'minecraft:jungle_sign',
              'minecraft:dark_oak_sign', 'minecraft:acacia_sign',
              'minecraft:warped_sign', 'minecraft:crimson_sign')
WALLSIGNS = ('minecraft:oak_wall_sign', 'minecraft:birch_wall_sign',
             'minecraft:spruce_wall_sign', 'minecraft:jungle_wall_sign',
             'minecraft:dark_oak_wall_sign', 'minecraft:acacia_wall_sign',
             'minecraft:warped_wall_sign', 'minecraft:crimson_wall_sign')
SIGNS = FLOORSIGNS + WALLSIGNS

ANVILS = ('minecraft:anvil', 'minecraft:chipped_anvil',
          'minecraft:damaged_anvil')
BEDS = ('minecraft:white_bed', 'minecraft:orange_bed', 'minecraft:magenta_bed',
        'minecraft:light_blue_bed', 'minecraft:yellow_bed',
        'minecraft:lime_bed', 'minecraft:pink_bed', 'minecraft:gray_bed',
        'minecraft:light_gray_bed', 'minecraft:cyan_bed',
        'minecraft:purple_bed', 'minecraft:blue_bed', 'minecraft:brown_bed',
        'minecraft:green_bed', 'minecraft:red_bed', 'minecraft:black_bed')
FURNACES = ('minecraft:furnace', 'minecraft:smoker', 'minecraft:blast_furnace')
SHULKERBOXES = ('minecraft:shulker_box', 'minecraft:white_shulker_box',
                'minecraft:orange_shulker_box',
                'minecraft:magenta_shulker_box',
                'minecraft:light_blue_shulker_box',
                'minecraft:yellow_shulker_box',
                'minecraft:lime_shulker_box', 'minecraft:pink_shulker_box',
                'minecraft:gray_shulker_box',
                'minecraft:light_gray_shulker_box',
                'minecraft:cyan_shulker_box', 'minecraft:purple_shulker_box',
                'minecraft:blue_shulker_box', 'minecraft:brown_shulker_box',
                'minecraft:green_shulker_box', 'minecraft:red_shulker_box',
                'minecraft:black_shulker_box')

WOOLS = ('minecraft:white_wool', 'minecraft:wool',
         'minecraft:orange_wool', 'minecraft:magenta_wool',
         'minecraft:light_blue_wool', 'minecraft:yellow_wool',
         'minecraft:lime_wool', 'minecraft:pink_wool',
         'minecraft:gray_wool', 'minecraft:light_gray_wool',
         'minecraft:cyan_wool', 'minecraft:purple_wool',
         'minecraft:blue_wool', 'minecraft:brown_wool',
         'minecraft:green_wool', 'minecraft:red_wool',
         'minecraft:black_wool')
CARPETS = ('minecraft:white_carpet', 'minecraft:orange_carpet',
           'minecraft:magenta_carpet', 'minecraft:light_blue_carpet',
           'minecraft:yellow_carpet', 'minecraft:lime_carpet',
           'minecraft:pink_carpet', 'minecraft:gray_carpet',
           'minecraft:light_gray_carpet', 'minecraft:cyan_carpet',
           'minecraft:purple_carpet', 'minecraft:blue_carpet',
           'minecraft:brown_carpet', 'minecraft:green_carpet',
           'minecraft:red_carpet', 'minecraft:black_carpet')


ARTIFICIAL = ('minecraft:grass_path', 'minecraft:cobblestone') \
    + STAIRS + SLABS + FENCES + WALLS + GLASS + LIGHTSOURCES \
    + WIRING + RAILS + SWITCHES + PRESSUREPLATES + DOORS + TRAPDOORS + GATES \
    + HEADS + SIGNS + ANVILS + BEDS + FURNACES + SHULKERBOXES + WOOLS + CARPETS

# ================================================= grouped by obtrusiveness

INVISIBLE = AIR + ('minecraft:barrier', 'minecraft:structure_void')

# filter skylight
FILTERING = ('minecraft:water', 'minecraft:bubble_column',
             'minecraft:ice', 'minecraft:frosted_ice',
             'minecraft:cobweb',
             'minecraft:slime_block', 'minecraft:honey_block',
             'minecraft:spawner',
             'minecraft:lava',
             'minecraft:beacon',
             'minecraft:end_gateway',
             'minecraft:chorus_plant', 'minecraft:chorus_flower') \
    + LEAVES + SHULKERBOXES

# can be seen through easily
UNOBTRUSIVE = ('minecraft:ladder',
               'minecraft:tripwire',
               'minecraft:end_rod',
               'minecraft:nether_portal',
               'minecraft:iron_bars',
               'minecraft:chain',
               'minecraft:conduit',
               'minecraft:lily_pad',
               'minecraft:scaffolding',
               'minecraft:snow') \
    + GLASS + RAILS + WIRING + SWITCHES + TORCHES + SIGNS

# can be seen through moderately
OBTRUSIVE = ('minecraft:bell', 'minecraft:brewing_stand', 'minecraft:cake',
             'minecraft:campfire', 'minecraft:dragon_egg',
             'minecraft:flower_pot', 'minecraft:lectern', 'turtle_egg') \
    + ANVILS + HEADS + PLANTS + BEDS + FENCES + GATES + SLABS

TRANSPARENT = INVISIBLE + FILTERING + UNOBTRUSIVE + OBTRUSIVE

# all esle is considered opaque

# ========================================================= map colouring
# block visualization
# based on https://minecraft.gamepedia.com/Map_item_format#Base_colors
# liberty was taken to move stained glass panes and various flowers
# into the appropriate colour category

MAPTRANSPARENT = ('minecraft:redstone_lamp',
                  'minecraft:cake',
                  'minecraft:ladder',
                  'minecraft:tripwire',
                  'minecraft:flower_pot',
                  'minecraft:end_rod',
                  'minecraft:glass', 'minecraft:glass_pane',
                  'minecraft:nether_portal',
                  'minecraft:iron_bars',
                  'minecraft:chain') \
    + INVISIBLE + WIRING + RAILS + SWITCHES + HEADS + TORCHES

# base map colours
# WARNING: all non-transparent blocks are listed individually here again
PALETTE = {0x7fb238: ('minecraft:grass_block', 'minecraft:slime_block'),
           0xf7e9a3: ('minecraft:sand',
                      'minecraft:birch_planks', 'minecraft:stripped_birch_log',
                      'minecraft:birch_wood', 'minecraft:stripped_birch_wood',
                      'minecraft:birch_sign', 'minecraft:birch_wall_sign',
                      'minecraft:birch_pressure_plate',
                      'minecraft:birch_trapdoor', 'minecraft:birch_stairs',
                      'minecraft:birch_slab', 'minecraft:birch_fence_gate',
                      'minecraft:birch_fence', 'minecraft:birch_door',
                      'minecraft:sandstone', 'minecraft:sandstone_slab',
                      'minecraft:sandstone_stairs', 'minecraft:sandstone_wall',
                      'minecraft:cut_sandstone',
                      'minecraft:cut_sandstone_slab',
                      'minecraft:smooth_sandstone',
                      'minecraft:smooth_sandstone_slab',
                      'minecraft:smooth_sandstone_stairs',
                      'minecraft:chiseled_sandstone',
                      'minecraft:glowstone',
                      'minecraft:end_stone',
                      'minecraft:end_stone_bricks',
                      'minecraft:end_stone_brick_slab',
                      'minecraft:end_stone_brick_stairs',
                      'minecraft:end_stone_brick_wall',
                      'minecraft:bone_block',
                      'minecraft:turtle_egg',
                      'minecraft:scaffolding'),
           0xc7c7c7: ('minecraft:cobweb', 'minecraft:mushroom_stem'),
           0xff0000: ('minecraft:lava', 'minecraft:tnt',
                      'minecraft:fire', 'minecraft:redstone_block'),
           0xa0a0ff: ('minecraft:ice', 'minecraft:frosted_ice',
                      'minecraft:packed_ice', 'minecraft:blue_ice'),
           0xa7a7a7: ('minecraft:iron_block', 'minecraft:iron_door',
                      'minecraft:brewing_stand',
                      'minecraft:heavy_weighted_pressure_plate',
                      'minecraft:iron_trapdoor', 'minecraft:lantern',
                      'minecraft:anvil', 'minecraft:chipped_anvil',
                      'minecraft:damaged_anvil',
                      'minecraft:grindstone', 'minecraft:soul_lantern',
                      'minecraft:lodestone'),
           0x007c00: ('minecraft:oak_sapling', 'minecraft:spruce_sapling',
                      'minecraft:birch_sapling', 'minecraft:jungle_sapling',
                      'minecraft:acacia_sapling', 'minecraft:dark_oak_sapling',
                      'minecraft:wheat', 'minecraft:sugar_cane',
                      'minecraft:pumpkin_stem', 'minecraft:melon_stem',
                      'minecraft:lily_pad', 'minecraft:cocoa',
                      'minecraft:carrots', 'minecraft:potatoes',
                      'minecraft:beetroots', 'minecraft:sweet_berry_bush',
                      'minecraft:grass', 'minecraft:tall_grass',
                      'minecraft:fern', 'minecraft:large_fern',
                      'minecraft:vine',
                      'minecraft:oak_leaves', 'minecraft:spruce_leaves',
                      'minecraft:birch_leaves',
                      'minecraft:jungle_leaves', 'minecraft:acacia_leaves',
                      'minecraft:dark_oak_leaves',
                      'minecraft:cactus',
                      'minecraft:bamboo'),
           0xffffff: ('minecraft:snow', 'minecraft:snow_block',
                      'minecraft:white_bed', 'minecraft:white_wool',
                      'minecraft:white_stained_glass',
                      'minecraft:white_stained_glass_pane',
                      'minecraft:white_carpet', 'minecraft:white_banner',
                      'minecraft:white_shulker_box',
                      'minecraft:white_glazed_terracotta',
                      'minecraft:white_concrete',
                      'minecraft:white_concrete_powder',
                      'minecraft:lily_of_the_valley'),
           0xa4a8b8: ('minecraft:clay',
                      'minecraft:infested_stone',
                      'minecraft:infested_cobblestone',
                      'minecraft:infested_stone_bricks',
                      'minecraft:infested_cracked_stone_bricks',
                      'minecraft:infested_mossy_stone_bricks',
                      'minecraft:infested_chiseled_stone_bricks'),
           0x976d4d: ('minecraft:dirt', 'minecraft:coarse_dirt',
                      'minecraft:farmland', 'minecraft:grass_path',
                      'minecraft:granite', 'minecraft:granite_slab',
                      'minecraft:granite_stairs', 'minecraft:granite_wall',
                      'minecraft:polished_granite',
                      'minecraft:polished_granite_slab',
                      'minecraft:polished_granite_stairs',
                      'minecraft:jungle_planks', 'minecraft:jungle_log',
                      'minecraft:stripped_jungle_log',
                      'minecraft:jungle_wood',
                      'minecraft:stripped_jungle_wood',
                      'minecraft:jungle_sign',
                      'minecraft:jungle_wall_sign',
                      'minecraft:jungle_pressure_plate',
                      'minecraft:jungle_trapdoor',
                      'minecraft:jungle_stairs', 'minecraft:jungle_slab',
                      'minecraft:jungle_fence_gate',
                      'minecraft:jungle_fence', 'minecraft:jungle_door',
                      'minecraft:jukebox', 'minecraft:brown_mushroom_block'),
           0x707070: ('minecraft:stone', 'minecraft:stone_slab',
                      'minecraft:andesite', 'minecraft:andesite_slab',
                      'minecraft:andesite_stairs', 'minecraft:andesite_wall',
                      'minecraft:polished_andesite',
                      'minecraft:polished_andesite_slab',
                      'minecraft:polished_andesite_stairs',
                      'minecraft:cobblestone', 'minecraft:cobblestone_slab',
                      'minecraft:cobblestone_stairs',
                      'minecraft:cobblestone_wall',
                      'minecraft:bedrock', 'minecraft:gold_ore',
                      'minecraft:iron_ore', 'minecraft:coal_ore',
                      'minecraft:lapis_ore', 'minecraft:diamond_ore',
                      'minecraft:dispenser', 'minecraft:dropper',
                      'minecraft:mossy_cobblestone',
                      'minecraft:mossy_cobblestone_slab',
                      'minecraft:mossy_cobblestone_stairs',
                      'minecraft:mossy_cobblestone_wall',
                      'minecraft:spawner', 'minecraft:furnace',
                      'minecraft:stone_pressure_plate',
                      'minecraft:redstone_ore',
                      'minecraft:stone_bricks', 'minecraft:stone_brick_slab',
                      'minecraft:stone_brick_stairs',
                      'minecraft:stone_brick_wall',
                      'minecraft:mossy_stone_bricks',
                      'minecraft:mossy_stone_brick_slab',
                      'minecraft:mossy_stone_brick_stairs',
                      'minecraft:mossy_stone_brick_wall',
                      'minecraft:cracked_stone_bricks',
                      'minecraft:chiseled_stone_bricks',
                      'minecraft:emerald_ore', 'minecraft:ender_chest',
                      'minecraft:smooth_stone', 'minecraft:smooth_stone_slab',
                      'minecraft:observer', 'minecraft:smoker',
                      'minecraft:blast_furnace', 'minecraft:stonecutter',
                      'minecraft:sticky_piston', 'minecraft:piston',
                      'minecraft:piston_head', 'minecraft:gravel',
                      'minecraft:acacia_log', 'minecraft:cauldron',
                      'minecraft:hopper'),
           0x4040ff: ('minecraft:kelp', 'minecraft:seagrass',
                      'minecraft:tall_seagrass',
                      'minecraft:water', 'minecraft:bubble_column'),
           0x8f7748: ('minecraft:oak_planks', 'minecraft:oak_log',
                      'minecraft:stripped_oak_log', 'minecraft:oak_wood',
                      'minecraft:stripped_oak_wood', 'minecraft:oak_sign',
                      'minecraft:oak_wall_sign',
                      'minecraft:oak_pressure_plate', 'minecraft:oak_trapdoor',
                      'minecraft:oak_stairs',
                      'minecraft:oak_slab', 'minecraft:oak_fence_gate',
                      'minecraft:oak_fence', 'minecraft:oak_door',
                      'minecraft:note_block', 'minecraft:bookshelf',
                      'minecraft:chest', 'minecraft:trapped_chest',
                      'minecraft:crafting_table',
                      'minecraft:daylight_detector',
                      'minecraft:loom', 'minecraft:barrel',
                      'minecraft:cartography_table',
                      'minecraft:fletching_table',
                      'minecraft:lectern', 'minecraft:smithing_table',
                      'minecraft:composter', 'minecraft:bamboo_sapling',
                      'minecraft:dead_bush', 'minecraft:petrified_oak_slab',
                      'minecraft:beehive'),
           0xfffcf5: ('minecraft:diorite', 'minecraft:diorite_slab',
                      'minecraft:diorite_stairs', 'minecraft:diorite_wall',
                      'minecraft:polished_diorite',
                      'minecraft:polished_diorite_slab',
                      'minecraft:polished_diorite_stairs',
                      'minecraft:birch_log',
                      'minecraft:quartz_block', 'minecraft:quartz_slab',
                      'minecraft:quartz_stairs',
                      'minecraft:smooth_quartz',
                      'minecraft:smooth_quartz_slab',
                      'minecraft:smooth_quartz_stairs',
                      'minecraft:chiseled_quartz_block',
                      'minecraft:quartz_pillar', 'minecraft:quartz_bricks',
                      'minecraft:sea_lantern', 'minecraft:target'),
           0xd87f33: ('minecraft:acacia_planks',
                      'minecraft:stripped_acacia_log',
                      'minecraft:stripped_acacia_wood',
                      'minecraft:acacia_sign', 'minecraft:acacia_wall_sign',
                      'minecraft:acacia_pressure_plate',
                      'minecraft:acacia_trapdoor', 'minecraft:acacia_stairs',
                      'minecraft:acacia_slab',
                      'minecraft:acacia_fence_gate', 'minecraft:acacia_fence',
                      'minecraft:acacia_door',
                      'minecraft:red_sand',
                      'minecraft:orange_wool', 'minecraft:orange_carpet',
                      'minecraft:orange_shulker_box', 'minecraft:orange_bed',
                      'minecraft:orange_stained_glass',
                      'minecraft:orange_stained_glass_pane',
                      'minecraft:orange_banner',
                      'minecraft:orange_glazed_terracotta',
                      'minecraft:orange_concrete',
                      'minecraft:orange_concrete_powder',
                      'minecraft:pumpkin', 'minecraft:carved_pumpkin',
                      'minecraft:jack_o_lantern',
                      'minecraft:terracotta',
                      'minecraft:red_sandstone',
                      'minecraft:red_sandstone_slab',
                      'minecraft:red_sandstone_stairs',
                      'minecraft:red_sandstone_wall',
                      'minecraft:cut_red_sandstone',
                      'minecraft:cut_red_sandstone_slab',
                      'minecraft:smooth_red_sandstone',
                      'minecraft:smooth_red_sandstone_slab',
                      'minecraft:smooth_red_sandstone_stairs',
                      'minecraft:chiseled_red_sandstone',
                      'minecraft:honey_block', 'minecraft:honeycomb_block',
                      'minecraft:orange_tulip'),
           0xb24cd8: ('minecraft:magenta_wool', 'minecraft:magenta_carpet',
                      'minecraft:magenta_shulker_box', 'minecraft:magenta_bed',
                      'minecraft:magenta_stained_glass',
                      'minecraft:magenta_stained_glass_pane',
                      'minecraft:magenta_banner',
                      'minecraft:magenta_glazed_terracotta',
                      'minecraft:magenta_concrete',
                      'minecraft:magenta_concrete_powder',
                      'minecraft:purpur_block', 'minecraft:purpur_slab',
                      'minecraft:purpur_stairs', 'minecraft:purpur_pillar',
                      'minecraft:allium', 'minecraft:lilac'),
           0x6699d8: ('minecraft:light_blue_wool',
                      'minecraft:light_blue_carpet',
                      'minecraft:light_blue_shulker_box',
                      'minecraft:light_blue_bed',
                      'minecraft:light_blue_stained_glass',
                      'minecraft:light_blue_stained_glass_pane',
                      'minecraft:light_blue_banner',
                      'minecraft:light_blue_glazed_terracotta',
                      'minecraft:light_blue_concrete',
                      'minecraft:light_blue_concrete_powder',
                      'minecraft:soul_fire',
                      'minecraft:blue_orchid'),
           0xe5e533: ('minecraft:sponge', 'minecraft:wet_sponge',
                      'minecraft:yellow_wool', 'minecraft:yellow_carpet',
                      'minecraft:yellow_shulker_box', 'minecraft:yellow_bed',
                      'minecraft:yellow_stained_glass',
                      'minecraft:yellow_stained_glass_pane',
                      'minecraft:yellow_banner',
                      'minecraft:yellow_glazed_terracotta',
                      'minecraft:yellow_concrete',
                      'minecraft:yellow_concrete_powder',
                      'minecraft:hay_block',
                      'minecraft:horn_coral_block', 'minecraft:horn_coral',
                      'minecraft:horn_coral_fan',
                      'minecraft:bee_nest',
                      'minecraft:dandelion', 'minecraft:sunflower'),
           0x7fcc19: ('minecraft:lime_wool', 'minecraft:lime_carpet',
                      'minecraft:lime_shulker_box', 'minecraft:lime_bed',
                      'minecraft:lime_stained_glass',
                      'minecraft:lime_stained_glass_pane',
                      'minecraft:lime_banner',
                      'minecraft:lime_glazed_terracotta',
                      'minecraft:lime_concrete',
                      'minecraft:lime_concrete_powder',
                      'minecraft:melon'),
           0xf27fa5: ('minecraft:pink_wool', 'minecraft:pink_carpet',
                      'minecraft:pink_shulker_box', 'minecraft:pink_bed',
                      'minecraft:pink_stained_glass',
                      'minecraft:pink_stained_glass_pane',
                      'minecraft:pink_banner',
                      'minecraft:pink_glazed_terracotta',
                      'minecraft:pink_concrete',
                      'minecraft:pink_concrete_powder',
                      'minecraft:brain_coral_block', 'minecraft:brain_coral',
                      'minecraft:brain_coral_fan',
                      'minecraft:pink_tulip', 'minecraft:peony'),
           0x4c4c4c: ('minecraft:acacia_wood',
                      'minecraft:gray_wool', 'minecraft:gray_carpet',
                      'minecraft:gray_shulker_box', 'minecraft:gray_bed',
                      'minecraft:gray_stained_glass',
                      'minecraft:gray_stained_glass_pane',
                      'minecraft:gray_banner',
                      'minecraft:gray_glazed_terracotta',
                      'minecraft:gray_concrete',
                      'minecraft:gray_concrete_powder',
                      'minecraft:dead_tube_coral_block',
                      'minecraft:dead_tube_coral',
                      'minecraft:dead_tube_coral_fan',
                      'minecraft:dead_brain_coral_block',
                      'minecraft:dead_brain_coral',
                      'minecraft:dead_brain_coral_fan',
                      'minecraft:dead_bubble_coral_block',
                      'minecraft:dead_bubble_coral',
                      'minecraft:dead_bubble_coral_fan',
                      'minecraft:dead_fire_coral_block',
                      'minecraft:dead_fire_coral',
                      'minecraft:dead_fire_coral_fan',
                      'minecraft:dead_horn_coral_block',
                      'minecraft:dead_horn_coral',
                      'minecraft:dead_horn_coral_fan'),
           0x999999: ('minecraft:light_gray_wool',
                      'minecraft:light_gray_carpet',
                      'minecraft:light_gray_shulker_box',
                      'minecraft:light_gray_bed',
                      'minecraft:light_gray_stained_glass',
                      'minecraft:light_gray_stained_glass_pane',
                      'minecraft:light_gray_banner',
                      'minecraft:light_gray_glazed_terracotta',
                      'minecraft:light_gray_concrete',
                      'minecraft:light_gray_concrete_powder',
                      'minecraft:structure_block', 'minecraft:jigsaw',
                      'minecraft:azure_bluet', 'minecraft:oxeye_daisy',
                      'minecraft:white_tulip'),
           0x4c7f99: ('minecraft:cyan_wool', 'minecraft:cyan_carpet',
                      'minecraft:cyan_shulker_box', 'minecraft:cyan_bed',
                      'minecraft:cyan_stained_glass',
                      'minecraft:cyan_stained_glass_pane',
                      'minecraft:cyan_banner',
                      'minecraft:cyan_glazed_terracotta',
                      'minecraft:cyan_concrete',
                      'minecraft:cyan_concrete_powder',
                      'minecraft:prismarine_slab',
                      'minecraft:prismarine_stairs',
                      'minecraft:prismarine_wall',
                      'minecraft:warped_roots', 'minecraft:warped_door',
                      'minecraft:warped_fungus',
                      'minecraft:twisting_vines', 'minecraft:nether_sprouts'),
           0x7f3fb2: ('minecraft:shulker_box',
                      'minecraft:purple_wool', 'minecraft:purple_carpet',
                      'minecraft:purple_bed',
                      'minecraft:purple_stained_glass',
                      'minecraft:purple_stained_glass_pane',
                      'minecraft:purple_banner',
                      'minecraft:purple_glazed_terracotta',
                      'minecraft:purple_concrete',
                      'minecraft:purple_concrete_powder',
                      'minecraft:mycelium',
                      'minecraft:chorus_plant', 'minecraft:chorus_flower',
                      'minecraft:repeating_command_block',
                      'minecraft:bubble_coral_block', 'minecraft:bubble_coral',
                      'minecraft:bubble_coral_fan'),
           0x334cb2: ('minecraft:blue_wool', 'minecraft:blue_carpet',
                      'minecraft:blue_shulker_box', 'minecraft:blue_bed',
                      'minecraft:blue_stained_glass',
                      'minecraft:blue_stained_glass_pane',
                      'minecraft:blue_banner',
                      'minecraft:blue_glazed_terracotta',
                      'minecraft:blue_concrete',
                      'minecraft:blue_concrete_powder',
                      'minecraft:tube_coral_block', 'minecraft:tube_coral',
                      'minecraft:tube_coral_fan',
                      'minecraft:cornflower'),
           0x664c33: ('minecraft:dark_oak_planks', 'minecraft:dark_oak_log',
                      'minecraft:stripped_dark_oak_log',
                      'minecraft:dark_oak_wood',
                      'minecraft:stripped_dark_oak_wood',
                      'minecraft:dark_oak_sign',
                      'minecraft:dark_oak_wall_sign',
                      'minecraft:dark_oak_pressure_plate',
                      'minecraft:dark_oak_trapdoor',
                      'minecraft:dark_oak_stairs',
                      'minecraft:dark_oak_slab',
                      'minecraft:dark_oak_fence_gate',
                      'minecraft:dark_oak_fence',
                      'minecraft:dark_oak_door',
                      'minecraft:spruce_log',
                      'minecraft:brown_wool', 'minecraft:brown_carpet',
                      'minecraft:brown_shulker_box', 'minecraft:brown_bed',
                      'minecraft:brown_stained_glass',
                      'minecraft:brown_stained_glass_pane',
                      'minecraft:brown_banner',
                      'minecraft:brown_glazed_terracotta',
                      'minecraft:brown_concrete',
                      'minecraft:brown_concrete_powder',
                      'minecraft:soul_sand', 'minecraft:command_block',
                      'minecraft:brown_mushroom', 'minecraft:soul_soil'),
           0x667f33: ('minecraft:green_wool', 'minecraft:green_carpet',
                      'minecraft:green_shulker_box', 'minecraft:green_bed',
                      'minecraft:green_stained_glass',
                      'minecraft:green_stained_glass_pane',
                      'minecraft:green_banner',
                      'minecraft:green_glazed_terracotta',
                      'minecraft:green_concrete',
                      'minecraft:green_concrete_powder',
                      'minecraft:end_portal_frame',
                      'minecraft:chain_command_block',
                      'minecraft:dried_kelp_block', 'minecraft:sea_pickle'),
           0x993333: ('minecraft:red_wool', 'minecraft:red_carpet',
                      'minecraft:red_shulker_box', 'minecraft:red_bed',
                      'minecraft:red_stained_glass',
                      'minecraft:red_stained_glass_pane',
                      'minecraft:red_banner',
                      'minecraft:red_glazed_terracotta',
                      'minecraft:red_concrete',
                      'minecraft:red_concrete_powder',
                      'minecraft:bricks', 'minecraft:brick_slab',
                      'minecraft:brick_stairs', 'minecraft:brick_wall',
                      'minecraft:red_mushroom_block', 'minecraft:nether_wart',
                      'minecraft:enchanting_table',
                      'minecraft:nether_wart_block',
                      'minecraft:fire_coral_block', 'minecraft:fire_coral',
                      'minecraft:fire_coral_fan',
                      'minecraft:red_mushroom', 'minecraft:shroomlight',
                      'minecraft:poppy', 'minecraft:red_tulip',
                      'minecraft:rose_bush'),
           0x191919: ('minecraft:black_wool', 'minecraft:black_carpet',
                      'minecraft:black_shulker_box', 'minecraft:black_bed',
                      'minecraft:black_stained_glass',
                      'minecraft:black_stained_glass_pane',
                      'minecraft:black_banner',
                      'minecraft:black_glazed_terracotta',
                      'minecraft:black_concrete',
                      'minecraft:black_concrete_powder',
                      'minecraft:obsidian', 'minecraft:end_portal',
                      'minecraft:dragon_egg', 'minecraft:coal_block',
                      'minecraft:end_gateway', 'minecraft:basalt',
                      'minecraft:polished_basalt', 'minecraft:netherite_block',
                      'minecraft:ancient_debris', 'minecraft:crying_obsidian',
                      'minecraft:respawn_anchor',
                      'minecraft:blackstone', 'minecraft:blackstone_slab',
                      'minecraft:blackstone_stairs',
                      'minecraft:blackstone_wall',
                      'minecraft:polished_blackstone',
                      'minecraft:polished_blackstone_slab',
                      'minecraft:polished_blackstone_stairs',
                      'minecraft:polished_blackstone_wall',
                      'minecraft:polished_blackstone_bricks',
                      'minecraft:polished_blackstone_brick_slab',
                      'minecraft:polished_blackstone_brick_stairs',
                      'minecraft:polished_blackstone_brick_wall',
                      'minecraft:gilded_blackstone',
                      'minecraft:wither_rose'),
           0xfaee4d: ('minecraft:gold_block',
                      'minecraft:light_weighted_pressure_plate',
                      'minecraft:bell'),
           0x5cdbd5: ('minecraft:diamond_block', 'minecraft:beacon',
                      'minecraft:prismarine_bricks',
                      'minecraft:prismarine_brick_slab',
                      'minecraft:prismarine_brick_stairs',
                      'minecraft:dark_prismarine',
                      'minecraft:dark_prismarine_slab',
                      'minecraft:dark_prismarine_stairs',
                      'minecraft:conduit'),
           0x4a80ff: ('minecraft:lapis_block', ),
           0x00d93a: ('minecraft:emerald_block', ),
           0x815631: ('minecraft:podzol', 'minecraft:spruce_planks',
                      'minecraft:stripped_spruce_log', 'minecraft:spruce_wood',
                      'minecraft:stripped_spruce_wood',
                      'minecraft:spruce_sign', 'minecraft:spruce_wall_sign',
                      'minecraft:spruce_pressure_plate',
                      'minecraft:spruce_trapdoor', 'minecraft:spruce_stairs',
                      'minecraft:spruce_slab', 'minecraft:spruce_fence_gate',
                      'minecraft:spruce_fence',
                      'minecraft:spruce_door',
                      'minecraft:campfire', 'minecraft:soul_campfire'),
           0x700200: ('minecraft:netherrack',
                      'minecraft:nether_bricks',
                      'minecraft:nether_brick_fence',
                      'minecraft:nether_brick_slab',
                      'minecraft:nether_brick_stairs',
                      'minecraft:nether_brick_wall',
                      'minecraft:cracked_nether_bricks',
                      'minecraft:chiseled_nether_bricks',
                      'minecraft:nether_gold_ore',
                      'minecraft:nether_quartz_ore',
                      'minecraft:magma_block',
                      'minecraft:red_nether_bricks',
                      'minecraft:red_nether_brick_slab',
                      'minecraft:red_nether_brick_stairs',
                      'minecraft:red_nether_brick_wall',
                      'minecraft:crimson_roots', 'minecraft:crimson_door',
                      'minecraft:crimson_fungus',
                      'minecraft:weeping_vines'),
           0xd1b1a1: ('minecraft:white_terracotta', ),
           0x9f5224: ('minecraft:orange_terracotta', ),
           0x95576c: ('minecraft:magenta_terracotta', ),
           0x706c8a: ('minecraft:light_blue_terracotta', ),
           0xba8524: ('minecraft:yellow_terracotta', ),
           0x677535: ('minecraft:lime_terracotta', ),
           0xa04d4e: ('minecraft:pink_terracotta', ),
           0x392923: ('minecraft:gray_terracotta', ),
           0x876b62: ('minecraft:light_gray_terracotta', ),
           0x575c5c: ('minecraft:cyan_terracotta', ),
           0x7a4958: ('minecraft:purple_terracotta',
                      'minecraft:purple_shulker_box'),
           0x4c3e5c: ('minecraft:blue_terracotta', ),
           0x4c3223: ('minecraft:brown_terracotta', ),
           0x4c522a: ('minecraft:green_terracotta', ),
           0x8e3c2e: ('minecraft:red_terracotta', ),
           0x251610: ('minecraft:black_terracotta', ),
           0xbd3031: ('minecraft:crimson_nylium', ),
           0x943f61: ('minecraft:crimson_fence',
                      'minecraft:crimson_fence_gate',
                      'minecraft:crimson_planks',
                      'minecraft:crimson_pressure_plate',
                      'minecraft:crimson_sign', 'minecraft:crimson_wall_sign',
                      'minecraft:crimson_slab', 'minecraft:crimson_stairs',
                      'minecraft:crimson_stem',
                      'minecraft:stripped_crimson_stem',
                      'minecraft:crimson_trapdoor'),
           0x5c191d: ('minecraft:crimson_hyphae',
                      'minecraft:stripped_crimson_hyphae'),
           0x167e86: ('minecraft:warped_nylium', ),
           0x3a8e8c: ('minecraft:warped_fence', 'minecraft:warped_fence_gate',
                      'minecraft:warped_planks',
                      'minecraft:warped_pressure_plate',
                      'minecraft:warped_sign', 'minecraft:warped_wall_sign',
                      'minecraft:warped_slab', 'minecraft:warped_stairs',
                      'minecraft:warped_stem',
                      'minecraft:stripped_warped_stem',
                      'minecraft:warped_trapdoor'),
           0x562c3e: ('minecraft:warped_hyphae',
                      'minecraft:stripped_warped_hyphae'),
           0x14b485: ('minecraft:warped_wart_block', )}
PALETTELOOKUP = {}
for hex, blocks in PALETTE.items():
    for block in blocks:
        PALETTELOOKUP[block] = hex


# ========================================================= biome-related

BIOMES = {0: "ocean",
          1: "plains",
          2: "desert",
          3: "mountains",
          4: "forest",
          5: "taiga",
          6: "swamp",
          7: "river",
          8: "nether_wastes",
          9: "the_end",
          10: "frozen_ocean",
          11: "frozen_river",
          12: "snowy_tundra",
          13: "snowy_mountains",
          14: "mushroom_fields",
          15: "mushroom_field_shore",
          16: "beach",
          17: "desert_hills",
          18: "wooded_hills",
          19: "taiga_hills",
          20: "mountain_edge",
          21: "jungle",
          22: "jungle_hills",
          23: "jungle_edge",
          24: "deep_ocean",
          25: "stone_shore",
          26: "snowy_beach",
          27: "birch_forest",
          28: "birch_forest_hills",
          29: "dark_forest",
          30: "snowy_taiga",
          31: "snowy_taiga_hills",
          32: "giant_tree_taiga",
          33: "giant_tree_taiga_hills",
          34: "wooded_mountains",
          35: "savanna",
          36: "savanna_plateau",
          37: "badlands",
          38: "wooded_badlands_plateau",
          39: "badlands_plateau",
          40: "small_end_islands",
          41: "end_midlands",
          42: "end_highlands",
          43: "end_barrens",
          44: "warm_ocean",
          45: "lukewarm_ocean",
          46: "cold_ocean",
          47: "deep_warm_ocean",
          48: "deep_lukewarm_ocean",
          49: "deep_cold_ocean",
          50: "deep_frozen_ocean",
          127: "the_void",
          129: "sunflower_plains",
          130: "desert_lakes",
          131: "gravelly_mountains",
          132: "flower_forest",
          133: "taiga_mountains",
          134: "swamp_hills",
          140: "ice_spikes",
          149: "modified_jungle",
          151: "modified_jungle_edge",
          155: "tall_birch_forest",
          156: "tall_birch_hills",
          157: "dark_forest_hills",
          158: "snowy_taiga_mountains",
          160: "giant_spruce_taiga",
          161: "giant_spruce_taiga_hills",
          162: "modified_gravelly_mountains",
          163: "shattered_savanna",
          164: "shattered_savanna_plateau",
          165: "eroded_badlands",
          166: "modified_wooded_badlands_plateau",
          167: "modified_badlands_plateau",
          168: "bamboo_jungle",
          169: "bamboo_jungle_hills",
          170: "soul_sand_valley",
          171: "crimson_forest",
          172: "warped_forest",
          173: "basalt_deltas"
          }

# ========================================================= technical values

# the width of ASCII characters in pixels
# space between characters is 1
# the widest supported Unicode character is 9 wide
ASCIIPIXELS = {'A': 5, 'a': 5,
               'B': 5, 'b': 5,
               'C': 5, 'c': 5,
               'D': 5, 'd': 5,
               'E': 5, 'e': 5,
               'F': 5, 'f': 4,
               'G': 5, 'g': 5,
               'H': 5, 'h': 5,
               'I': 3, 'i': 1,
               'J': 5, 'j': 5,
               'K': 5, 'k': 4,
               'L': 5, 'l': 2,
               'M': 5, 'm': 5,
               'N': 5, 'n': 5,
               'O': 5, 'o': 5,
               'P': 5, 'p': 5,
               'Q': 5, 'q': 5,
               'R': 5, 'r': 5,
               'S': 5, 's': 5,
               'T': 5, 't': 3,
               'U': 5, 'u': 5,
               'V': 5, 'v': 5,
               'W': 5, 'w': 5,
               'X': 5, 'x': 5,
               'Y': 5, 'y': 5,
               'Z': 5, 'z': 5,
               '1': 5, '2': 5, '3': 5, '4': 5, '5': 5,
               '6': 5, '7': 5, '8': 5, '9': 5, '0': 5,
               ' ': 3, '!': 1, '@': 6, '#': 5, '$': 5, '£': 5, '%': 5, '^': 5,
               '&': 5, '*': 3, '(': 3, ')': 3, '_': 5, '-': 5, '+': 5, '=': 5,
               '~': 6, '[': 3, ']': 3, '{': 3, '}': 3, '|': 1, '\\': 5, ':': 1,
               ';': 1, '"': 3, "'": 1, ',': 1, '<': 4, '>': 4, '.': 1, '?': 5,
               '/': 5, '`': 2}

# terminal colour codes


def supports_color():
    """Return True if the running system's terminal supports color."""
    plat = sys.platform
    supported_platform = plat != 'Pocket PC' and (plat != 'win32'
                                                  or 'ANSICON' in os.environ)
    # isatty is not always implemented, #6223.
    is_a_tty = hasattr(sys.stdout, 'isatty') and sys.stdout.isatty()
    return supported_platform and is_a_tty


if supports_color():
    TCOLORS = {"black":    "\033[38;2;000;000;000m",
               "gray":     "\033[38;2;128;128;128m",
               "white":    "\033[38;2;255;255;255m",
               "pink":     "\033[38;2;255;192;203m",
               "red":      "\033[38;2;255;000;000m",
               "orange":   "\033[38;2;255;165;000m",
               "yellow":   "\033[38;2;255;255;000m",
               "darkgreen": "\033[38;2;000;128;000m",
               "green":    "\033[38;2;000;255;000m",
               "blue":     "\033[38;2;135;206;235m",
               "darkblue": "\033[38;2;000;000;255m",
               "magenta":  "\033[38;2;255;000;255m",
               "brown":    "\033[38;2;139;069;019m",
               "CLR":      "\033[0m"}  # 38 is replaced by 48 for background
else:
    TCOLORS = {"black": "", "gray": "", "white": "",
               "pink": "", "red": "", "orange": "",
               "yellow": "", "darkgreen": "", "green": "",
               "blue": "", "darkblue": "", "magenta": "",
               "brown": "", "CLR": ""}  # colour codes not printed

INVENTORYDIMENSIONS = {(9, 3): ('minecraft:barrel', 'minecraft:chest',
                                'minecraft:trapped_chest') + SHULKERBOXES,
                       (3, 3): ('minecraft:dispenser', 'minecraft:dropper'),
                       (5, 1): ('minecraft:hopper', 'minecraft:brewing_stand'),
                       (3, 1): FURNACES}
INVENTORYLOOKUP = {}
for dimensions, blocks in INVENTORYDIMENSIONS.items():
    for block in blocks:
        INVENTORYLOOKUP[block] = dimensions

# version checking


def closestVersion(version):
    """Retrieve next-best version code to given version code."""
    if version in VERSIONS:
        return version
    for v in sorted(VERSIONS.keys(), reverse=True):
        if version - v >= 0:
            return v
    return 0


def checkVersion():
    """Retrieve Minecraft version and check compatibility."""
    from .worldLoader import WorldSlice

    slice = WorldSlice(0, 0, 1, 1)    # single-chunk slice
    current = int(slice.nbtfile["Chunks"][0]["DataVersion"].value)
    closestname = "Unknown"
    # check compatibility
    if current not in VERSIONS or VERSIONS[SUPPORTS] not in VERSIONS[current]:
        closest = closestVersion(current)
        closestname = VERSIONS[closest]
        closestname += " snapshot" if current > closest else ""
        if closest > SUPPORTS:
            print(f"{TCOLORS['orange']}WARNING: You are using a newer "
                  "version of Minecraft then GDPC supports!\n"
                  f"\tSupports: {VERSIONS[SUPPORTS]} "
                  f"Detected: {closestname}{TCOLORS['CLR']}")
        elif closest < SUPPORTS:
            print(f"{TCOLORS['orange']}WARNING: You are using an older "
                  "version of Minecraft then GDPC supports!\n"
                  f"\tSupports: {VERSIONS[SUPPORTS]} "
                  f"Detected: {closestname}{TCOLORS['CLR']}")
        else:
            raise ValueError(f"{TCOLORS['red']}Invalid supported version: "
                             f"SUPPORTS = {current}!{TCOLORS['CLR']}")
    else:
        closestname = VERSIONS[current]

    return (current, closestname)


CURRENTV, CURRENTVNAME = checkVersion()
