"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        const lambda = new transliterator_1.Transliterator(this, 'Resource', {
            description: `Creates ${props.language} documentation from jsii-enabled npm packages`,
            environment: { ...environment, TARGET_LANGUAGE: props.language.toString() },
            logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
            vpc: props.vpc,
            vpcSubnets: (_c = props.vpcSubnets) !== null && _c !== void 0 ? _c : { subnetType: aws_ec2_1.SubnetType.ISOLATED },
        });
        this.function = lambda;
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.function);
        // The handler reads & writes to this bucket.
        bucket.grantRead(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}`);
        bucket.grantWrite(this.function, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(props.language)}${constants.NOT_SUPPORTED_SUFFIX}`);
        props.monitoring.watchful.watchLambdaFunction('Transliterator Function', lambda);
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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