"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = (lang) => new tasks.LambdaInvoke(this, `Add to catalog.json (${lang})`, {
            lambdaFunction: catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 });
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        const definition = new aws_stepfunctions_1.Parallel(this, 'DocGen')
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            const task = new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
                lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, language }).function,
                resultPath: '$.docGenOutput',
                resultSelector: { [`${language}.$`]: '$.Payload' },
            }).addRetry({ interval: core_1.Duration.seconds(30) });
            return task.next(addToCatalog(language));
        }))
            .addCatch(sendToDeadLetterQueue.next(new aws_stepfunctions_1.Fail(this, 'Fail', {
            error: 'Failed',
            cause: 'Input was submitted to dead letter queue',
        })), { resultPath: '$._error' });
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        const reprocessAll = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(reprocessAll);
        this.stateMachine.grantStartExecution(reprocessAll);
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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