"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const discovery_1 = require("./discovery");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.timeout = core_1.Duration.minutes(15);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [
                {
                    prefix: "staged/" /* STAGED_KEY_PREFIX */,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        // Note: the handler is designed to stop processing more batches about 2 minutes ahead of the timeout.
        this.function = new discovery_1.Discovery(this, 'Default', {
            description: '[ConstructHub/Discovery] Periodically query npm.js index for new construct libraries',
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: this.timeout,
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
        });
        this.bucket.grantReadWrite(this.function);
        props.queue.grantSendMessages(this.function);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.timeout),
            targets: [new aws_events_targets_1.LambdaFunction(this.function)],
        });
        props.monitoring.watchful.watchLambdaFunction('Discovery Function', this.function);
        this.alarmErrors = this.function.metricErrors({ period: core_1.Duration.minutes(15) }).createAlarm(this, 'ErrorsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) failed to run',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.alarmNoInvocations = this.function.metricInvocations({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmDescription: 'The discovery function (on npmjs.com) is not running as scheduled',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of staging failures.
     */
    metricStagingFailureCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "StagingFailureCount" /* STAGING_FAILURE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The average time it took to stage a package to S3.
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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