"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const crypto_1 = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: '# Catalog Overview',
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                            props.inventory.metricMissingPythonDocsCount({ label: 'Missing Py-Docs' }),
                            props.inventory.metricMissingTypeScriptDocsCount({ label: 'Missing Ts-Doc' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Discovery Function',
                            '',
                            `[button:Search Log Group](${lambdaSearchLogGroupUrl(props.discovery.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.discovery.function.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.discovery.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.discovery.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            props.discovery.metricChangeCount({ label: 'Change Count' }),
                            props.discovery.metricUnprocessableEntity({ label: 'Unprocessable' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            fillMetric(props.discovery.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            fillMetric(props.discovery.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(15),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Search Log Group](${lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:State Machine](${stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:Redrive](${lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
            ],
        });
    }
}
exports.BackendDashboard = BackendDashboard;
function lambdaFunctionUrl(lambda) {
    return `/lambda/home#/functions/${lambda.functionName}`;
}
function lambdaSearchLogGroupUrl(lambda) {
    return `/cloudwatch/home#logsV2:log-groups/log-group/$252Faws$252flambda$252f${lambda.functionName}/log-events`;
}
function stateMachineUrl(stateMachine) {
    return `/states/home#/statemachines/view/${stateMachine.stateMachineArn}`;
}
function fillMetric(metric, value = 0) {
    // We assume namespace + name is enough to uniquely identify a metric here.
    // This is true locally at this time, but in case this ever changes, consider
    // also processing dimensions and period.
    const h = crypto_1.createHash('sha256')
        .update(metric.namespace)
        .update('\0')
        .update(metric.metricName)
        .digest('hex');
    const metricName = `m${h}`;
    return new aws_cloudwatch_1.MathExpression({
        expression: `FILL(${metricName}, ${value})`,
        label: metric.label,
        usingMetrics: { [metricName]: metric },
    });
}
//# sourceMappingURL=data:application/json;base64,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