"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const lib_1 = require("../lib");
describe('Fields', () => {
    test('deep replace correctly handles fields in arrays', () => {
        expect(lib_1.FieldUtils.renderObject({
            unknown: undefined,
            bool: true,
            literal: 'literal',
            field: lib_1.JsonPath.stringAt('$.stringField'),
            listField: lib_1.JsonPath.listAt('$.listField'),
            deep: [
                'literal',
                {
                    deepField: lib_1.JsonPath.numberAt('$.numField'),
                },
            ],
        })).toStrictEqual({
            'bool': true,
            'literal': 'literal',
            'field.$': '$.stringField',
            'listField.$': '$.listField',
            'deep': [
                'literal',
                {
                    'deepField.$': '$.numField',
                },
            ],
        });
    }),
        test('exercise contextpaths', () => {
            expect(lib_1.FieldUtils.renderObject({
                str: lib_1.JsonPath.stringAt('$$.Execution.StartTime'),
                count: lib_1.JsonPath.numberAt('$$.State.RetryCount'),
                token: lib_1.JsonPath.taskToken,
                entire: lib_1.JsonPath.entireContext,
            })).toStrictEqual({
                'str.$': '$$.Execution.StartTime',
                'count.$': '$$.State.RetryCount',
                'token.$': '$$.Task.Token',
                'entire.$': '$$',
            });
        }),
        test('find all referenced paths', () => {
            expect(lib_1.FieldUtils.findReferencedPaths({
                bool: false,
                literal: 'literal',
                field: lib_1.JsonPath.stringAt('$.stringField'),
                listField: lib_1.JsonPath.listAt('$.listField'),
                deep: [
                    'literal',
                    {
                        field: lib_1.JsonPath.stringAt('$.stringField'),
                        deepField: lib_1.JsonPath.numberAt('$.numField'),
                    },
                ],
            })).toStrictEqual(['$.listField', '$.numField', '$.stringField']);
        }),
        test('cannot have JsonPath fields in arrays', () => {
            expect(() => lib_1.FieldUtils.renderObject({
                deep: [lib_1.JsonPath.stringAt('$.hello')],
            })).toThrowError(/Cannot use JsonPath fields in an array/);
        }),
        test('datafield path must be correct', () => {
            expect(lib_1.JsonPath.stringAt('$')).toBeDefined();
            expect(() => lib_1.JsonPath.stringAt('$hello')).toThrowError(/exactly '\$', '\$\$', start with '\$.' or start with '\$\$.'/);
            expect(() => lib_1.JsonPath.stringAt('hello')).toThrowError(/exactly '\$', '\$\$', start with '\$.' or start with '\$\$.'/);
        }),
        test('context path must be correct', () => {
            expect(lib_1.JsonPath.stringAt('$$')).toBeDefined();
            expect(() => lib_1.JsonPath.stringAt('$$hello')).toThrowError(/exactly '\$', '\$\$', start with '\$.' or start with '\$\$.'/);
            expect(() => lib_1.JsonPath.stringAt('hello')).toThrowError(/exactly '\$', '\$\$', start with '\$.' or start with '\$\$.'/);
        }),
        test('test contains task token', () => {
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.JsonPath.taskToken,
            }));
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.JsonPath.stringAt('$$.Task'),
            }));
            expect(true).toEqual(lib_1.FieldUtils.containsTaskToken({
                field: lib_1.JsonPath.entireContext,
            }));
            expect(false).toEqual(lib_1.FieldUtils.containsTaskToken({
                oops: 'not here',
            }));
            expect(false).toEqual(lib_1.FieldUtils.containsTaskToken({
                oops: lib_1.JsonPath.stringAt('$$.Execution.StartTime'),
            }));
        }),
        test('arbitrary JSONPath fields are not replaced', () => {
            expect(lib_1.FieldUtils.renderObject({
                field: '$.content',
            })).toStrictEqual({
                field: '$.content',
            });
        }),
        test('fields cannot be used somewhere in a string interpolation', () => {
            expect(() => lib_1.FieldUtils.renderObject({
                field: `contains ${lib_1.JsonPath.stringAt('$.hello')}`,
            })).toThrowError(/Field references must be the entire string/);
        });
});
//# sourceMappingURL=data:application/json;base64,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