"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const ga = require("../lib");
const util_1 = require("./util");
test('create accelerator', () => {
    // GIVEN
    const { stack } = util_1.testFixture();
    // WHEN
    new ga.Accelerator(stack, 'Accelerator');
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::Accelerator', {
        Enabled: true,
    }));
});
test('create listener', () => {
    // GIVEN
    const { stack } = util_1.testFixture();
    // WHEN
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 80,
                toPort: 80,
            },
        ],
    });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::Listener', {
        AcceleratorArn: {
            'Fn::GetAtt': [
                'Accelerator8EB0B6B1',
                'AcceleratorArn',
            ],
        },
        PortRanges: [
            {
                FromPort: 80,
                ToPort: 80,
            },
        ],
        Protocol: 'TCP',
        ClientAffinity: 'NONE',
    }));
});
test('create endpointgroup', () => {
    // GIVEN
    const { stack } = util_1.testFixture();
    // WHEN
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 80,
                toPort: 80,
            },
        ],
    });
    new ga.EndpointGroup(stack, 'Group', { listener });
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::EndpointGroup', {
        EndpointGroupRegion: {
            Ref: 'AWS::Region',
        },
        ListenerArn: {
            'Fn::GetAtt': [
                'Listener828B0E81',
                'ListenerArn',
            ],
        },
    }));
});
test('addEndpoint', () => {
    // GIVEN
    const { stack, vpc } = util_1.testFixture();
    // WHEN
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 80,
                toPort: 80,
            },
        ],
    });
    const endpointGroup = new ga.EndpointGroup(stack, 'Group', { listener });
    const instance = new ec2.Instance(stack, 'Instance', {
        vpc,
        machineImage: new ec2.AmazonLinuxImage(),
        instanceType: new ec2.InstanceType('t3.small'),
    });
    endpointGroup.addEndpoint('endpoint', instance.instanceId);
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::EndpointGroup', {
        EndpointConfigurations: [
            {
                EndpointId: {
                    Ref: 'InstanceC1063A87',
                },
            },
        ],
    }));
});
test('addLoadBalancer', () => {
    // GIVEN
    const { stack, vpc } = util_1.testFixture();
    // WHEN
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 80,
                toPort: 80,
            },
        ],
    });
    const endpointGroup = new ga.EndpointGroup(stack, 'Group', { listener });
    const alb = new elbv2.ApplicationLoadBalancer(stack, 'ALB', { vpc, internetFacing: true });
    endpointGroup.addLoadBalancer('endpoint', alb);
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::EndpointGroup', {
        EndpointConfigurations: [
            {
                EndpointId: {
                    Ref: 'ALBAEE750D2',
                },
            },
        ],
    }));
});
test('addElasticIpAddress', () => {
    // GIVEN
    const { stack } = util_1.testFixture();
    // WHEN
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 80,
                toPort: 80,
            },
        ],
    });
    const endpointGroup = new ga.EndpointGroup(stack, 'Group', { listener });
    const eip = new ec2.CfnEIP(stack, 'ElasticIpAddress');
    endpointGroup.addElasticIpAddress('endpoint', eip);
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::EndpointGroup', {
        EndpointConfigurations: [
            {
                EndpointId: {
                    'Fn::GetAtt': [
                        'ElasticIpAddress',
                        'AllocationId',
                    ],
                },
            },
        ],
    }));
});
test('addEc2Instance', () => {
    // GIVEN
    const { stack, vpc } = util_1.testFixture();
    // WHEN
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 80,
                toPort: 80,
            },
        ],
    });
    const endpointGroup = new ga.EndpointGroup(stack, 'Group', { listener });
    const instance = new ec2.Instance(stack, 'Instance', {
        vpc,
        machineImage: new ec2.AmazonLinuxImage(),
        instanceType: new ec2.InstanceType('t3.small'),
    });
    endpointGroup.addEc2Instance('endpoint', instance);
    // THEN
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::GlobalAccelerator::EndpointGroup', {
        EndpointConfigurations: [
            {
                EndpointId: {
                    Ref: 'InstanceC1063A87',
                },
            },
        ],
    }));
});
//# sourceMappingURL=data:application/json;base64,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