"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const ga = require("../lib");
const util_1 = require("./util");
test('custom resource exists', () => {
    // GIVEN
    const { stack, vpc } = util_1.testFixture();
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 443,
                toPort: 443,
            },
        ],
    });
    const endpointGroup = new ga.EndpointGroup(stack, 'Group', { listener });
    // WHEN
    ga.AcceleratorSecurityGroup.fromVpc(stack, 'GlobalAcceleratorSG', vpc, endpointGroup);
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('Custom::AWS', {
        Properties: {
            ServiceToken: {
                'Fn::GetAtt': [
                    'AWS679f53fac002430cb0da5b7982bd22872D164C4C',
                    'Arn',
                ],
            },
            Create: {
                action: 'describeSecurityGroups',
                service: 'EC2',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                'GlobalAccelerator',
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                {
                                    Ref: 'VPCB9E5F0B4',
                                },
                            ],
                        },
                    ],
                },
                physicalResourceId: {
                    responsePath: 'SecurityGroups.0.GroupId',
                },
            },
        },
        DependsOn: [
            'GroupC77FDACD',
        ],
    }, assert_1.ResourcePart.CompleteDefinition));
});
test('can create security group rule', () => {
    // GIVEN
    const { stack, alb, vpc } = util_1.testFixtureAlb();
    const accelerator = new ga.Accelerator(stack, 'Accelerator');
    const listener = new ga.Listener(stack, 'Listener', {
        accelerator,
        portRanges: [
            {
                fromPort: 443,
                toPort: 443,
            },
        ],
    });
    const endpointGroup = new ga.EndpointGroup(stack, 'Group', { listener });
    endpointGroup.addLoadBalancer('endpoint', alb);
    // WHEN
    const sg = ga.AcceleratorSecurityGroup.fromVpc(stack, 'GlobalAcceleratorSG', vpc, endpointGroup);
    alb.connections.allowFrom(sg, aws_ec2_1.Port.tcp(443));
    // THEN
    assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
        IpProtocol: 'tcp',
        FromPort: 443,
        GroupId: {
            'Fn::GetAtt': [
                'ALBSecurityGroup8B8624F8',
                'GroupId',
            ],
        },
        SourceSecurityGroupId: {
            'Fn::GetAtt': [
                'GlobalAcceleratorSGCustomResourceC1DB5287',
                'SecurityGroups.0.GroupId',
            ],
        },
        ToPort: 443,
    }));
});
//# sourceMappingURL=data:application/json;base64,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