"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EndpointGroup = exports.EndpointConfiguration = void 0;
const cdk = require("@aws-cdk/core");
const ga = require("./globalaccelerator.generated");
/**
 * The class for endpoint configuration
 */
class EndpointConfiguration extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        props.endpointGroup._linkEndpoint(this);
    }
    /**
     * render the endpoint configuration for the endpoint group
     */
    renderEndpointConfiguration() {
        return {
            clientIpPreservationEnabled: this.props.clientIpReservation,
            endpointId: this.props.endpointId,
            weight: this.props.weight,
        };
    }
}
exports.EndpointConfiguration = EndpointConfiguration;
/**
 * EndpointGroup construct
 */
class EndpointGroup extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        /**
         * The array of the endpoints in this endpoint group
         */
        this.endpoints = new Array();
        const resource = new ga.CfnEndpointGroup(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            endpointGroupRegion: (_a = props.region) !== null && _a !== void 0 ? _a : cdk.Stack.of(this).region,
            endpointConfigurations: cdk.Lazy.anyValue({ produce: () => this.renderEndpoints() }, { omitEmptyArray: true }),
        });
        this.endpointGroupArn = resource.attrEndpointGroupArn;
        this.endpointGroupName = (_b = props.endpointGroupName) !== null && _b !== void 0 ? _b : resource.logicalId;
    }
    /**
     * import from ARN
     */
    static fromEndpointGroupArn(scope, id, endpointGroupArn) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.endpointGroupArn = endpointGroupArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add an endpoint
     */
    addEndpoint(id, endpointId, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointGroup: this,
            endpointId,
            ...props,
        });
    }
    /**
     * Add an Elastic Load Balancer as an endpoint in this endpoint group
     */
    addLoadBalancer(id, lb, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: lb.loadBalancerArn,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Add an EIP as an endpoint in this endpoint group
     */
    addElasticIpAddress(id, eip, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: eip.attrAllocationId,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Add an EC2 Instance as an endpoint in this endpoint group
     */
    addEc2Instance(id, instance, props = {}) {
        return new EndpointConfiguration(this, id, {
            endpointId: instance.instanceId,
            endpointGroup: this,
            ...props,
        });
    }
    /**
     * Links a endpoint to this endpoint group
     * @internal
     */
    _linkEndpoint(endpoint) {
        this.endpoints.push(endpoint);
    }
    renderEndpoints() {
        return this.endpoints.map(e => e.renderEndpointConfiguration());
    }
}
exports.EndpointGroup = EndpointGroup;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW5kcG9pbnQtZ3JvdXAuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJlbmRwb2ludC1ncm91cC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBcUM7QUFDckMsb0RBQW9EO0FBOEdwRDs7R0FFRztBQUNILE1BQWEscUJBQXNCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFLdEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFpQztRQUM3RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLEtBQUssQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQzFDLENBQUM7SUFFRDs7T0FFRztJQUNJLDJCQUEyQjtRQUNoQyxPQUFPO1lBQ0wsMkJBQTJCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBbUI7WUFDM0QsVUFBVSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsVUFBVTtZQUNqQyxNQUFNLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNO1NBQzFCLENBQUM7SUFDSixDQUFDO0NBQ0Y7QUFyQkQsc0RBcUJDO0FBRUQ7O0dBRUc7QUFDSCxNQUFhLGFBQWMsU0FBUSxHQUFHLENBQUMsUUFBUTtJQXdCN0MsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUF5Qjs7UUFDckUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQU5uQjs7V0FFRztRQUNnQixjQUFTLEdBQUcsSUFBSSxLQUFLLEVBQXlCLENBQUM7UUFLaEUsTUFBTSxRQUFRLEdBQUcsSUFBSSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUN6RCxXQUFXLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxXQUFXO1lBQ3ZDLG1CQUFtQixRQUFFLEtBQUssQ0FBQyxNQUFNLG1DQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU07WUFDOUQsc0JBQXNCLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsRUFBRSxFQUFFLEVBQUUsRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLENBQUM7U0FDL0csQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdCQUFnQixHQUFHLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQztRQUN0RCxJQUFJLENBQUMsaUJBQWlCLFNBQUcsS0FBSyxDQUFDLGlCQUFpQixtQ0FBSSxRQUFRLENBQUMsU0FBUyxDQUFDO0lBQ3pFLENBQUM7SUFsQ0Q7O09BRUc7SUFDSSxNQUFNLENBQUMsb0JBQW9CLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsZ0JBQXdCO1FBQzNGLE1BQU0sTUFBTyxTQUFRLEdBQUcsQ0FBQyxRQUFRO1lBQWpDOztnQkFDa0IscUJBQWdCLEdBQUcsZ0JBQWdCLENBQUM7WUFDdEQsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQTRCRDs7T0FFRztJQUNJLFdBQVcsQ0FBQyxFQUFVLEVBQUUsVUFBa0IsRUFBRSxRQUNuRCxFQUFFO1FBQ0EsT0FBTyxJQUFJLHFCQUFxQixDQUFDLElBQUksRUFBRSxFQUFFLEVBQUU7WUFDekMsYUFBYSxFQUFFLElBQUk7WUFDbkIsVUFBVTtZQUNWLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLGVBQWUsQ0FBQyxFQUFVLEVBQUUsRUFBZ0IsRUFBRSxRQUFzQyxFQUFFO1FBQzNGLE9BQU8sSUFBSSxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3pDLFVBQVUsRUFBRSxFQUFFLENBQUMsZUFBZTtZQUM5QixhQUFhLEVBQUUsSUFBSTtZQUNuQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxtQkFBbUIsQ0FBQyxFQUFVLEVBQUUsR0FBcUIsRUFBRSxRQUFzQyxFQUFFO1FBQ3BHLE9BQU8sSUFBSSxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3pDLFVBQVUsRUFBRSxHQUFHLENBQUMsZ0JBQWdCO1lBQ2hDLGFBQWEsRUFBRSxJQUFJO1lBQ25CLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLGNBQWMsQ0FBQyxFQUFVLEVBQUUsUUFBcUIsRUFBRSxRQUFzQyxFQUFFO1FBQy9GLE9BQU8sSUFBSSxxQkFBcUIsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3pDLFVBQVUsRUFBRSxRQUFRLENBQUMsVUFBVTtZQUMvQixhQUFhLEVBQUUsSUFBSTtZQUNuQixHQUFHLEtBQUs7U0FDVCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7OztPQUdHO0lBQ0ksYUFBYSxDQUFDLFFBQStCO1FBQ2xELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2hDLENBQUM7SUFFTyxlQUFlO1FBQ3JCLE9BQU8sSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsMkJBQTJCLEVBQUUsQ0FBQyxDQUFDO0lBQ2xFLENBQUM7Q0FDRjtBQTdGRCxzQ0E2RkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBnYSBmcm9tICcuL2dsb2JhbGFjY2VsZXJhdG9yLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBJTGlzdGVuZXIgfSBmcm9tICcuL2xpc3RlbmVyJztcblxuLyoqXG4gKiBUaGUgaW50ZXJmYWNlIG9mIHRoZSBFbmRwb2ludEdyb3VwXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSUVuZHBvaW50R3JvdXAgZXh0ZW5kcyBjZGsuSVJlc291cmNlIHtcbiAgLyoqXG4gICAqIEVuZHBvaW50R3JvdXAgQVJOXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50R3JvdXBBcm46IHN0cmluZztcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgYWRkTG9hZEJhbGFuY2VyYCwgYGFkZEVsYXN0aWNJcEFkZHJlc3NgIGFuZCBgYWRkRWMySW5zdGFuY2VgIHRvIGFkZCBlbmRwb2ludHMgaW50byB0aGUgZW5kcG9pbnQgZ3JvdXBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmRwb2ludENvbmZpZ3VyYXRpb25PcHRpb25zIHtcbiAgLyoqXG4gICAqIEluZGljYXRlcyB3aGV0aGVyIGNsaWVudCBJUCBhZGRyZXNzIHByZXNlcnZhdGlvbiBpcyBlbmFibGVkIGZvciBhbiBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyIGVuZHBvaW50XG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGNsaWVudElwUmVzZXJ2YXRpb24/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgd2VpZ2h0IGFzc29jaWF0ZWQgd2l0aCB0aGUgZW5kcG9pbnQuIFdoZW4geW91IGFkZCB3ZWlnaHRzIHRvIGVuZHBvaW50cywgeW91IGNvbmZpZ3VyZSBBV1MgR2xvYmFsIEFjY2VsZXJhdG9yXG4gICAqIHRvIHJvdXRlIHRyYWZmaWMgYmFzZWQgb24gcHJvcG9ydGlvbnMgdGhhdCB5b3Ugc3BlY2lmeS4gRm9yIGV4YW1wbGUsIHlvdSBtaWdodCBzcGVjaWZ5IGVuZHBvaW50IHdlaWdodHMgb2YgNCwgNSxcbiAgICogNSwgYW5kIDYgKHN1bT0yMCkuIFRoZSByZXN1bHQgaXMgdGhhdCA0LzIwIG9mIHlvdXIgdHJhZmZpYywgb24gYXZlcmFnZSwgaXMgcm91dGVkIHRvIHRoZSBmaXJzdCBlbmRwb2ludCwgNS8yMCBpc1xuICAgKiByb3V0ZWQgYm90aCB0byB0aGUgc2Vjb25kIGFuZCB0aGlyZCBlbmRwb2ludHMsIGFuZCA2LzIwIGlzIHJvdXRlZCB0byB0aGUgbGFzdCBlbmRwb2ludC5cbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZ2xvYmFsLWFjY2VsZXJhdG9yL2xhdGVzdC9kZy9hYm91dC1lbmRwb2ludHMtZW5kcG9pbnQtd2VpZ2h0cy5odG1sXG4gICAqIEBkZWZhdWx0IC0gbm90IHNwZWNpZmllZFxuICAgKi9cbiAgcmVhZG9ubHkgd2VpZ2h0PzogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgdG8gY3JlYXRlIEVuZHBvaW50Q29uZmlndXJhdGlvblxuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmRwb2ludENvbmZpZ3VyYXRpb25Qcm9wcyBleHRlbmRzIEVuZHBvaW50Q29uZmlndXJhdGlvbk9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIGVuZG9wb2ludCBncm91cCByZWVzb3VyY2VcbiAgICpcbiAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV1cbiAgICovXG4gIHJlYWRvbmx5IGVuZHBvaW50R3JvdXA6IEVuZHBvaW50R3JvdXA7XG5cbiAgLyoqXG4gICAqIEFuIElEIGZvciB0aGUgZW5kcG9pbnQuIElmIHRoZSBlbmRwb2ludCBpcyBhIE5ldHdvcmsgTG9hZCBCYWxhbmNlciBvciBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyLFxuICAgKiB0aGlzIGlzIHRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiB0aGUgcmVzb3VyY2UuIElmIHRoZSBlbmRwb2ludCBpcyBhbiBFbGFzdGljIElQIGFkZHJlc3MsXG4gICAqIHRoaXMgaXMgdGhlIEVsYXN0aWMgSVAgYWRkcmVzcyBhbGxvY2F0aW9uIElELiBGb3IgRUMyIGluc3RhbmNlcywgdGhpcyBpcyB0aGUgRUMyIGluc3RhbmNlIElELlxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRJZDogc3RyaW5nO1xufVxuXG4vKipcbiAqIExvYWRCYWxhbmNlciBJbnRlcmZhY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMb2FkQmFsYW5jZXIge1xuICAvKipcbiAgICogVGhlIEFSTiBvZiB0aGlzIGxvYWQgYmFsYW5jZXJcbiAgICovXG4gIHJlYWRvbmx5IGxvYWRCYWxhbmNlckFybjogc3RyaW5nO1xufVxuXG4vKipcbiAqIEVDMiBJbnN0YW5jZSBpbnRlcmZhY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFYzJJbnN0YW5jZSAge1xuICAvKipcbiAgICogVGhlIGlkIG9mIHRoZSBpbnN0YW5jZSByZXNvdXJjZVxuICAgKi9cbiAgcmVhZG9ubHkgaW5zdGFuY2VJZDogc3RyaW5nO1xufVxuXG4vKipcbiAqIEVJUCBJbnRlcmZhY2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbGFzdGljSXBBZGRyZXNzIHtcbiAgLyoqXG4gICAqIGFsbG9jYXRpb24gSUQgb2YgdGhlIEVJUCByZXNvcnVjZVxuICAgKi9cbiAgcmVhZG9ubHkgYXR0ckFsbG9jYXRpb25JZDogc3RyaW5nXG59XG5cbi8qKlxuICogUHJvcGVydHkgb2YgdGhlIEVuZHBvaW50R3JvdXBcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbmRwb2ludEdyb3VwUHJvcHMge1xuICAvKipcbiAgICogTmFtZSBvZiB0aGUgZW5kcG9pbnQgZ3JvdXBcbiAgICpcbiAgICogQGRlZmF1bHQgLSBsb2dpY2FsIElEIG9mIHRoZSByZXNvdXJjZVxuICAgKi9cbiAgcmVhZG9ubHkgZW5kcG9pbnRHcm91cE5hbWU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUmVzb3VyY2UgTmFtZSAoQVJOKSBvZiB0aGUgbGlzdGVuZXIuXG4gICAqL1xuICByZWFkb25seSBsaXN0ZW5lcjogSUxpc3RlbmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgQVdTIFJlZ2lvbiB3aGVyZSB0aGUgZW5kcG9pbnQgZ3JvdXAgaXMgbG9jYXRlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB0aGUgcmVnaW9uIG9mIHRoZSBjdXJyZW50IHN0YWNrXG4gICAqL1xuICByZWFkb25seSByZWdpb24/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogVGhlIGNsYXNzIGZvciBlbmRwb2ludCBjb25maWd1cmF0aW9uXG4gKi9cbmV4cG9ydCBjbGFzcyBFbmRwb2ludENvbmZpZ3VyYXRpb24gZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFRoZSBwcm9wZXJ0eSBjb250YWluaW5nIGFsbCB0aGUgY29uZmlndXJhdGlvbiB0byBiZSByZW5kZXJlZFxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHByb3BzOiBFbmRwb2ludENvbmZpZ3VyYXRpb25Qcm9wcztcbiAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBFbmRwb2ludENvbmZpZ3VyYXRpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgIHByb3BzLmVuZHBvaW50R3JvdXAuX2xpbmtFbmRwb2ludCh0aGlzKTtcbiAgfVxuXG4gIC8qKlxuICAgKiByZW5kZXIgdGhlIGVuZHBvaW50IGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBlbmRwb2ludCBncm91cFxuICAgKi9cbiAgcHVibGljIHJlbmRlckVuZHBvaW50Q29uZmlndXJhdGlvbigpOiBnYS5DZm5FbmRwb2ludEdyb3VwLkVuZHBvaW50Q29uZmlndXJhdGlvblByb3BlcnR5IHtcbiAgICByZXR1cm4ge1xuICAgICAgY2xpZW50SXBQcmVzZXJ2YXRpb25FbmFibGVkOiB0aGlzLnByb3BzLmNsaWVudElwUmVzZXJ2YXRpb24sXG4gICAgICBlbmRwb2ludElkOiB0aGlzLnByb3BzLmVuZHBvaW50SWQsXG4gICAgICB3ZWlnaHQ6IHRoaXMucHJvcHMud2VpZ2h0LFxuICAgIH07XG4gIH1cbn1cblxuLyoqXG4gKiBFbmRwb2ludEdyb3VwIGNvbnN0cnVjdFxuICovXG5leHBvcnQgY2xhc3MgRW5kcG9pbnRHcm91cCBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElFbmRwb2ludEdyb3VwIHtcbiAgLyoqXG4gICAqIGltcG9ydCBmcm9tIEFSTlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmcm9tRW5kcG9pbnRHcm91cEFybihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgZW5kcG9pbnRHcm91cEFybjogc3RyaW5nKTogSUVuZHBvaW50R3JvdXAge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElFbmRwb2ludEdyb3VwIHtcbiAgICAgIHB1YmxpYyByZWFkb25seSBlbmRwb2ludEdyb3VwQXJuID0gZW5kcG9pbnRHcm91cEFybjtcbiAgICB9XG4gICAgcmV0dXJuIG5ldyBJbXBvcnQoc2NvcGUsIGlkKTtcbiAgfVxuXG4gIHB1YmxpYyByZWFkb25seSBlbmRwb2ludEdyb3VwQXJuOiBzdHJpbmc7XG4gIC8qKlxuICAgKlxuICAgKiBUaGUgbmFtZSBvZiB0aGUgZW5kcG9pbnQgZ3JvdXBcbiAgICpcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGVuZHBvaW50R3JvdXBOYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgYXJyYXkgb2YgdGhlIGVuZHBvaW50cyBpbiB0aGlzIGVuZHBvaW50IGdyb3VwXG4gICAqL1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgZW5kcG9pbnRzID0gbmV3IEFycmF5PEVuZHBvaW50Q29uZmlndXJhdGlvbj4oKTtcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEVuZHBvaW50R3JvdXBQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBnYS5DZm5FbmRwb2ludEdyb3VwKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGxpc3RlbmVyQXJuOiBwcm9wcy5saXN0ZW5lci5saXN0ZW5lckFybixcbiAgICAgIGVuZHBvaW50R3JvdXBSZWdpb246IHByb3BzLnJlZ2lvbiA/PyBjZGsuU3RhY2sub2YodGhpcykucmVnaW9uLFxuICAgICAgZW5kcG9pbnRDb25maWd1cmF0aW9uczogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnJlbmRlckVuZHBvaW50cygpIH0sIHsgb21pdEVtcHR5QXJyYXk6IHRydWUgfSksXG4gICAgfSk7XG5cbiAgICB0aGlzLmVuZHBvaW50R3JvdXBBcm4gPSByZXNvdXJjZS5hdHRyRW5kcG9pbnRHcm91cEFybjtcbiAgICB0aGlzLmVuZHBvaW50R3JvdXBOYW1lID0gcHJvcHMuZW5kcG9pbnRHcm91cE5hbWUgPz8gcmVzb3VyY2UubG9naWNhbElkO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhbiBlbmRwb2ludFxuICAgKi9cbiAgcHVibGljIGFkZEVuZHBvaW50KGlkOiBzdHJpbmcsIGVuZHBvaW50SWQ6IHN0cmluZywgcHJvcHM6IEVuZHBvaW50Q29uZmlndXJhdGlvbk9wdGlvbnMgPVxuICB7fSkge1xuICAgIHJldHVybiBuZXcgRW5kcG9pbnRDb25maWd1cmF0aW9uKHRoaXMsIGlkLCB7XG4gICAgICBlbmRwb2ludEdyb3VwOiB0aGlzLFxuICAgICAgZW5kcG9pbnRJZCxcbiAgICAgIC4uLnByb3BzLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhbiBFbGFzdGljIExvYWQgQmFsYW5jZXIgYXMgYW4gZW5kcG9pbnQgaW4gdGhpcyBlbmRwb2ludCBncm91cFxuICAgKi9cbiAgcHVibGljIGFkZExvYWRCYWxhbmNlcihpZDogc3RyaW5nLCBsYjogTG9hZEJhbGFuY2VyLCBwcm9wczogRW5kcG9pbnRDb25maWd1cmF0aW9uT3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIG5ldyBFbmRwb2ludENvbmZpZ3VyYXRpb24odGhpcywgaWQsIHtcbiAgICAgIGVuZHBvaW50SWQ6IGxiLmxvYWRCYWxhbmNlckFybixcbiAgICAgIGVuZHBvaW50R3JvdXA6IHRoaXMsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYW4gRUlQIGFzIGFuIGVuZHBvaW50IGluIHRoaXMgZW5kcG9pbnQgZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBhZGRFbGFzdGljSXBBZGRyZXNzKGlkOiBzdHJpbmcsIGVpcDogRWxhc3RpY0lwQWRkcmVzcywgcHJvcHM6IEVuZHBvaW50Q29uZmlndXJhdGlvbk9wdGlvbnMgPSB7fSkge1xuICAgIHJldHVybiBuZXcgRW5kcG9pbnRDb25maWd1cmF0aW9uKHRoaXMsIGlkLCB7XG4gICAgICBlbmRwb2ludElkOiBlaXAuYXR0ckFsbG9jYXRpb25JZCxcbiAgICAgIGVuZHBvaW50R3JvdXA6IHRoaXMsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYW4gRUMyIEluc3RhbmNlIGFzIGFuIGVuZHBvaW50IGluIHRoaXMgZW5kcG9pbnQgZ3JvdXBcbiAgICovXG4gIHB1YmxpYyBhZGRFYzJJbnN0YW5jZShpZDogc3RyaW5nLCBpbnN0YW5jZTogRWMySW5zdGFuY2UsIHByb3BzOiBFbmRwb2ludENvbmZpZ3VyYXRpb25PcHRpb25zID0ge30pIHtcbiAgICByZXR1cm4gbmV3IEVuZHBvaW50Q29uZmlndXJhdGlvbih0aGlzLCBpZCwge1xuICAgICAgZW5kcG9pbnRJZDogaW5zdGFuY2UuaW5zdGFuY2VJZCxcbiAgICAgIGVuZHBvaW50R3JvdXA6IHRoaXMsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBMaW5rcyBhIGVuZHBvaW50IHRvIHRoaXMgZW5kcG9pbnQgZ3JvdXBcbiAgICogQGludGVybmFsXG4gICAqL1xuICBwdWJsaWMgX2xpbmtFbmRwb2ludChlbmRwb2ludDogRW5kcG9pbnRDb25maWd1cmF0aW9uKSB7XG4gICAgdGhpcy5lbmRwb2ludHMucHVzaChlbmRwb2ludCk7XG4gIH1cblxuICBwcml2YXRlIHJlbmRlckVuZHBvaW50cygpIHtcbiAgICByZXR1cm4gdGhpcy5lbmRwb2ludHMubWFwKGUgPT4gZS5yZW5kZXJFbmRwb2ludENvbmZpZ3VyYXRpb24oKSk7XG4gIH1cbn1cbiJdfQ==