"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AcceleratorSecurityGroup = void 0;
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const custom_resources_1 = require("@aws-cdk/custom-resources");
/**
 * The security group used by a Global Accelerator to send traffic to resources in a VPC.
 */
class AcceleratorSecurityGroup {
    /**
     * Lookup the Global Accelerator security group at CloudFormation deployment time.
     *
     * As of this writing, Global Accelerators (AGA) create a single security group per VPC. AGA security groups are shared
     * by all AGAs in an account. Additionally, there is no CloudFormation mechanism to reference the AGA security groups.
     *
     * This makes creating security group rules which allow traffic from an AGA complicated in CDK. This lookup will identify
     * the AGA security group for a given VPC at CloudFormation deployment time, and lets you create rules for traffic from AGA
     * to other resources created by CDK.
     */
    static fromVpc(scope, id, vpc, endpointGroup) {
        // The security group name is always 'GlobalAccelerator'
        const globalAcceleratorSGName = 'GlobalAccelerator';
        // How to reference the security group name in the response from EC2
        const ec2ResponseSGIdField = 'SecurityGroups.0.GroupId';
        // The AWS Custom Resource that make a call to EC2 to get the security group ID, for the given VPC
        const lookupAcceleratorSGCustomResource = new custom_resources_1.AwsCustomResource(scope, id + 'CustomResource', {
            onCreate: {
                service: 'EC2',
                action: 'describeSecurityGroups',
                parameters: {
                    Filters: [
                        {
                            Name: 'group-name',
                            Values: [
                                globalAcceleratorSGName,
                            ],
                        },
                        {
                            Name: 'vpc-id',
                            Values: [
                                vpc.vpcId,
                            ],
                        },
                    ],
                },
                // We get back a list of responses, but the list should be of length 0 or 1
                // Getting no response means no resources have been linked to the AGA
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse(ec2ResponseSGIdField),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: custom_resources_1.AwsCustomResourcePolicy.ANY_RESOURCE,
            }),
        });
        // Look up the security group ID
        const sg = aws_ec2_1.SecurityGroup.fromSecurityGroupId(scope, id, lookupAcceleratorSGCustomResource.getResponseField(ec2ResponseSGIdField));
        // We add a dependency on the endpoint group, guaranteeing that CloudFormation won't
        // try and look up the SG before AGA creates it. The SG is created when a VPC resource
        // is associated with an AGA
        lookupAcceleratorSGCustomResource.node.addDependency(endpointGroup);
        return sg;
    }
    constructor() { }
}
exports.AcceleratorSecurityGroup = AcceleratorSecurityGroup;
//# sourceMappingURL=data:application/json;base64,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