# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['esdb',
 'esdb.client',
 'esdb.client.streams',
 'esdb.client.subscriptions',
 'esdb.generated']

package_data = \
{'': ['*']}

install_requires = \
['grpcio>=1.47.0,<2.0.0', 'protobuf<4.0']

setup_kwargs = {
    'name': 'esdb',
    'version': '0.2.0',
    'description': 'gRPC client for EventStore DB',
    'long_description': '# esdb-py\n\n[![PyPI version](https://badge.fury.io/py/esdb.svg)](https://pypi.org/project/esdb/)\n\n**EventStoreDB Python gRPC client**\n> NOTE: This project is still work in progress\n\n<!-- TOC -->\n* [Completed features](#completed-features)\n* [Installation](#installation)\n* [Development](#development)\n* [Usage](#usage)\n  * [Append/Read](#appendread)\n  * [Batch append](#batch-append)\n  * [Transient subscription to all events with filtering](#transient-subscription-to-all-events-with-filtering)\n  * [Persistent subscriptions](#persistent-subscriptions)\n<!-- TOC -->\n\n## Completed features\n- [x] secure connection\n- [x] basic auth\n- [x] streams\n  - [x] append\n  - [x] batch append\n  - [x] delete\n  - [x] read stream\n  - [x] read all with stream/event type filters\n  - [x] transient subscriptions\n  - [x] tombstone\n  - [x] filtering\n- [x] persistent subscriptions\n    - [x] create\n    - [x] read\n    - [ ] update\n    - [ ] delete\n    - [ ] list\n    - [ ] info\n    - [ ] reply parked events \n- [ ] CRUD for projections\n- [ ] users\n- [ ] other connection options\n  - [ ] multi-node gossip\n\n## Installation\nUsing pip:\n```sh\npip install esdb\n```\nUsing poetry:\n```sh\npoetry add esdb\n```\n\n# Development\n1. Install [poetry](https://python-poetry.org/docs/#installation)\n2. Create virtualenv (i.e. using pyenv):\n```sh\npyenv install 3.10.5\npyenv virtualenv 3.10.5 esdb-py\npyenv local esdb-py\n```\n3. Install deps with `poetry install`\n4. Start eventstore in docker: `make run-esdb`\n5. Run the tests: `pytest tests`\n\n# Usage\n\nHave a look at [tests](https://github.com/andriykohut/esdb-py/tree/main/tests) for more examples.\n\n## Append/Read\n\n```py\nimport asyncio\nimport datetime\nimport uuid\n\nfrom esdb.client import ESClient\n\n# For insecure connection without basic auth:\n# client = ESClient("localhost:2113", tls=False)\nwith open("certs/ca/ca.crt", "rb") as fh:\n  root_cert = fh.read()\n\nclient = ESClient(\n  "localhost:2111",\n  root_certificates=root_cert,\n  username="admin",\n  password="changeit",\n  keepalive_time_ms=5000,\n  keepalive_timeout_ms=10000,\n)\n\nstream = f"test-{str(uuid.uuid4())}"\n\n\nasync def streams():\n  async with client.connect() as conn:\n    for i in range(10):\n      append_result = await conn.streams.append(\n        stream=stream,\n        event_type="test_event",\n        data={"i": i, "ts": datetime.datetime.utcnow().isoformat()},\n      )\n\n    print("Forwards!")\n    async for result in conn.streams.read(stream=stream, count=10):\n      print(result.data)\n\n    print("Backwards!")\n    async for result in conn.streams.read(stream=stream, count=10, backwards=True):\n      print(result.data)\n\n    print("Forwards start from middle!")\n    async for result in conn.streams.read(stream=stream, count=10, revision=5):\n      print(result.data)\n\n    print("Backwards start from middle!")\n    async for result in conn.streams.read(stream=stream, count=10, backwards=True, revision=5):\n      print(result.data)\n\n    # Create a transient subscription to a stream\n    async for result in conn.streams.read(stream=stream, subscribe=True):\n      print(result.data)\n\n\nasyncio.run(streams())\n```\n\n## Batch append\n```py\nimport asyncio\nimport uuid\n\nfrom esdb.client import ESClient\nfrom esdb.client.streams import Message\n\n\nasync def batch_append():\n    stream = str(uuid.uuid4())\n    messages: list[Message] = [\n        Message(event_type="one", data={"item": 1}),\n        Message(event_type="one", data={"item": 2}),\n        Message(event_type="one", data={"item": 3}),\n        Message(event_type="two", data={"item": 1}),\n        Message(event_type="two", data={"item": 2}),\n        Message(event_type="two", data={"item": 3}),\n    ]\n    async with ESClient("localhost:2113", tls=False).connect() as conn:\n        response = await conn.streams.batch_append(stream=stream, messages=messages)\n        assert response.current_revision == 5\n        events = [e async for e in conn.streams.read(stream=stream, count=50)]\n        assert len(events) == 6\n\n\nasyncio.run(batch_append())\n```\n\n## Transient subscription to all events with filtering\n\n```py\nimport uuid\nimport asyncio\n\nfrom esdb.client import ESClient\nfrom esdb.client.streams import Filter\n\n\nasync def filters():\n  async with ESClient("localhost:2113", tls=False).connect() as conn:\n    for i in range(10):\n        await conn.streams.append(stream=str(uuid.uuid4()), event_type=f"prefix-{i}", data=b"")\n    async for event in conn.streams.read_all(\n            subscribe=True,  # subscribe will wait for events, use count=<n> to read <n> events and stop\n            filter_by=Filter(\n              kind=Filter.Kind.EVENT_TYPE,\n              regex="^prefix-",\n              # Checkpoint only required when subscribe=True, it\'s not needed when using count=<int>\n              checkpoint_interval_multiplier=1000,\n            ),\n    ):\n      print(event)\n\n\nasyncio.run(filters())\n```\n\n## Persistent subscriptions\n\n```python\nimport asyncio\nfrom esdb.client import ESClient\nfrom esdb.client.subscriptions import SubscriptionSettings, NackAction\n\nclient = ESClient("localhost:2113", tls=False)\n\nstream = "stream-foo"\ngroup = "group-bar"\n\n\nasync def persistent():\n  async with client.connect() as conn:\n    # emit some events to the same stream\n    for i in range(50):\n      await conn.streams.append(stream, "foobar", {"i": i})\n\n    # create a subscription\n    await conn.subscriptions.create_stream_subscription(\n      stream=stream,\n      group_name=group,\n      settings=SubscriptionSettings(\n        max_subscriber_count=50,\n        read_batch_size=5,\n        live_buffer_size=10,\n        history_buffer_size=10,\n        consumer_strategy=SubscriptionSettings.ConsumerStrategy.ROUND_ROBIN,\n        checkpoint=SubscriptionSettings.DurationType(\n          type=SubscriptionSettings.DurationType.Type.MS,\n          value=10000,\n        ),\n      ),\n    )\n\n  async with client.connect() as conn:\n    subscription = conn.subscriptions.subscribe_to_stream(stream=stream, group_name=group, buffer_size=5)\n    async for event in subscription:\n      try:\n        # do work with event\n        print(event)\n        await subscription.ack([event])\n      except Exception as err:\n        await subscription([event], NackAction.RETRY, reason=str(err))\n\n\nasyncio.run(persistent())\n```',
    'author': 'Andrii Kohut',
    'author_email': 'kogut.andriy@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/andriykohut/esdb-py',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
